---
title: "module lang::flybytes::Syntax"
id: Syntax
slug: /Packages/org.rascalmpl.flybytes/API/lang/flybytes/Syntax
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-0.41.2</div> <div class="theme-doc-version-badge badge badge--secondary">org.rascalmpl.flybytes-0.2.10</div>

#### Usage

```rascal
import lang::flybytes::Syntax;
```

#### Dependencies
```rascal
import List;
```


## data Class {#lang-flybytes-Syntax-Class}
The top-level compilation unit for flybytes is the class or the interface

```rascal
data Class (list[Annotation] annotations = [], loc src = |unknown:///|) 
     = class(Type \type /* object(str name) */, 
      set[Modifier] modifiers = {\public()},
      Type super              = object(),
      list[Type]   interfaces = [],
      list[Field]  fields     = [], 
      list[Method] methods    = []
      //list[Class] children = [],
    )
     | interface(Type \type /* object(str name) */,
      list[Type]   interfaces = [],
      list[Field]  fields  = [],
      list[Method] methods = []
    )
     ;
```

## data Modifier {#lang-flybytes-Syntax-Modifier}

```rascal
data Modifier  
     = \public()
     | \private()
     | \protected()
     | \friendly()
     | \static()
     | \final()
     | \synchronized()
     | \abstract()
     ;
```

## data Field {#lang-flybytes-Syntax-Field}
Class fields members have optional initialization expressions.

```rascal
data Field (list[Annotation] annotations = [], set[Modifier] modifiers = {\private()}, loc src=|unknown:///|) 
     = field(Type \type, str name, Exp init = defVal(\type))
     ;
```

## data Method {#lang-flybytes-Syntax-Method}
Four kinds of methods can be generated

```rascal
data Method (list[Annotation] annotations = [], loc src=|unknown:///|) 
     = method(Signature desc, list[Formal] formals, list[Stat] block, set[Modifier] modifiers = {\public()})
     | procedure(Signature desc, list[Formal] formals, list[Instruction] instructions, set[Modifier] modifiers = {\public()})
     | method(Signature desc, set[Modifier] modifiers={\abstract(), \public()})
     | static(list[Stat] block)
     ;
```


* `procedure` is a special extension since it contains only JVM bytecode instructions as a body, instead of high-level programming constructs like expressions and statements.
* one alternative of `method` is *abstract* methods, which do not have a body, the other is methods
* static blocks are also modeled as methods but without a signature.
* normal static methods are methods with a static [Modifier](../../../../..//Packages/org.rascalmpl.flybytes/API/lang/flybytes/Syntax.md#lang-flybytes-Syntax-Modifier).

The `src` field is important for stack traces and other debugging features.

## function method {#lang-flybytes-Syntax-method}

Shorthand for directly producing a method with its signature and code block.

```rascal
Method method(Modifier access, Type ret, str name, list[Formal] formals, list[Stat] block)
```

## data Signature {#lang-flybytes-Syntax-Signature}
Method and constructor signatures

```rascal
data Signature  
     = methodDesc(Type \return, str name, list[Type] formals)
     | constructorDesc(list[Type] formals)
     ;
```

## data Type {#lang-flybytes-Syntax-Type}
JVM type symbols

```rascal
data Type  
     = byte()
     | boolean()
     | short()
     | character()
     | integer()
     | float()
     | double()
     | long()
     | object(str name)
     | array(Type arg)
     | \void()
     | string()
     ;
```

## data Annotation {#lang-flybytes-Syntax-Annotation}

```rascal
data Annotation (RetentionPolicy retention=runtime()) 
     = \anno(str annoClass, Type \type, value val, str name = "value")
     | \tag(str annoClass)
     ;
```

## data RetentionPolicy {#lang-flybytes-Syntax-RetentionPolicy}

```rascal
data RetentionPolicy  
     = class()
     | runtime()
     | source()
     ;
```

## data Formal {#lang-flybytes-Syntax-Formal}
optional init expressions will be used at run-time if `null` is passed as actual parameter

```rascal
data Formal  
     = var(Type \type, str name, Exp init = defVal(\type))
     ;
```

## data Stat {#lang-flybytes-Syntax-Stat}
Structured programming with statements, OO primitives like `new`, and JVM monitor blocks and breakpoints

```rascal
data Stat (loc src = |unknown:///|) 
     = \store(str name, Exp \value)
     | \decl(Type \type, str name, Exp init = defVal(\type))
     | \astore(Exp array, Exp index, Exp arg)
     | \do(Exp exp)
     | \return()
     | \return(Exp arg)
     | \putField(Type class, Exp receiver, Type \type, str name, Exp arg)
     | \putStatic(Type class, str name, Type \type, Exp arg)
     | \if(Exp condition, list[Stat] thenBlock)
     | \if(Exp condition, list[Stat] thenBlock, list[Stat] elseBlock)
     | \for(list[Stat] init, 
         Exp condition, 
         list[Stat] next, 
         list[Stat] statements, str label = "")
     | \block(list[Stat] block, str label = "")
     | \break(str label = "")
     | \continue(str label = "")
     | \while(Exp condition, list[Stat] block, str label = "")
     | \doWhile(list[Stat] block, Exp condition, str label = "")
     | \throw(Exp arg)
     | \monitor(Exp arg, list[Stat] block)
     | \acquire(Exp arg)
     | \release(Exp arg)
     | \try(list[Stat] block, list[Handler] \catch)
     | \switch(Exp arg, list[Case] cases, SwitchOption option = lookup(/*for best performance on current JVMs*/))
     | invokeSuper(Signature desc, list[Exp] args)
     | \asm(list[Instruction] instructions)
     ;
```


* `invokeSuper` is typically only used as the first statement of a constructor
* `monitor` guarantees release of the lock in case of exceptions, break and continue that break out of the block, but only if `release` and `acquire` are not used on the same lock object anywhere.  
* `asm` allows for inling raw bytecode instructions, but please understand that the monitor blocks' semantics can be broken by jumping out of the block unseen.

## data SwitchOption {#lang-flybytes-Syntax-SwitchOption}

```rascal
data SwitchOption  
     = table()
     | lookup()
     | auto()
     ;
```

## data Case {#lang-flybytes-Syntax-Case}

```rascal
data Case  
     = \case(int key, list[Stat] block)
     | \default(list[Stat] block)
     ;
```

## data Handler {#lang-flybytes-Syntax-Handler}

```rascal
data Handler  
     = \catch(Type \type, str name, list[Stat] block)
     | \finally(list[Stat] block)
     ;
```

## data Exp {#lang-flybytes-Syntax-Exp}
JVM bytecode instructions lifted to the expression level.

```rascal
data Exp (loc src = |unknown:///|) 
     = null()
     | \true()
     | \false()
     | load(str name)
     | aload(Exp array, Exp index)
     | \const(Type \type, value constant)
     | sblock(list[Stat] statements, Exp arg)
     | invokeStatic(Type class, Signature desc, list[Exp] args)
     | invokeSpecial(Type class, Exp receiver, Signature desc, list[Exp] args)
     | invokeVirtual(Type class, Exp receiver, Signature desc, list[Exp] args)
     | invokeInterface(Type class, Exp receiver, Signature desc, list[Exp] args)
     | invokeDynamic(BootstrapCall handle, Signature desc, list[Exp] args)
     | newInstance(Type class, Signature desc, list[Exp] args)
     | getField(Type class, Exp receiver, Type \type, str name)
     | getStatic(Type class, Type \type, str name)
     | instanceof(Exp arg, Type class)
     | eq(Exp lhs, Exp rhs)
     | ne(Exp lhs, Exp rhs)
     | le(Exp lhs, Exp rhs)
     | gt(Exp lhs, Exp rhs)
     | ge(Exp lhs, Exp rhs)
     | lt(Exp lhs, Exp rhs)
     | newArray(Type \type, Exp size)
     | newInitArray(Type \type, list[Exp] args)
     | alength(Exp arg)
     | checkcast(Exp arg, Type \type)
     | coerce(Type from, Type to, Exp arg)
     | shr(Exp lhs, Exp shift)
     | shl(Exp lhs, Exp shift)
     | ushr(Exp lhs, Exp shift)
     | and(Exp lhs, Exp rhs)
     | sand(Exp lhs, Exp rhs)
     | or(Exp lhs, Exp rhs)
     | sor(Exp lhs, Exp rhs)
     | xor(Exp lhs, Exp rhs)
     | add(Exp lhs, Exp rhs)
     | sub(Exp lhs, Exp rhs)
     | div(Exp lhs, Exp rhs)
     | rem(Exp lhs, Exp rhs)
     | mul(Exp lhs, Exp rhs)
     | neg(Exp arg)
     | inc(str name, int inc)
     | cond(Exp condition, Exp thenExp, Exp elseExp)
     ;
```


The `Exp` syntax allows for expression all JVM bytecode expression stackmachine operations but with a nested recursive
syntax that is closer to programming languages. 

* with `load`, name resolution is limited to local variables and parameter names. 
* method invocation *always* requires the full method signature.
* field access and update also require the full class name and field name.
* about invokving methods:
   * `invokeStatic` must be used when invoking methods with the `static` modifier.
   * `invokeSpecial` can be used when no dynamic dispatch is needed, or searching superclasses is required, and you know which class implements the method. Use this to invoke a method for efficiency's sake. **The invocation is type-checked at class load time.** 
   * `invokeVirtual` should be used if you **do need** dynamic dispatch, or the method is implemented in a superclass, and this is not a default method of an interface, use this invocation method. You need to be sure the method exists _somewhere_ reachable from the \class reference type. **The invocation is type-checked at class load time.** 
   * `invokeInterface` is for invoking methods you know only from interfaces, such as default methods. The method can even be absent at runtime in which case this throws a RuntimeException. **The type-check occurs at the first invocation at run-time.** 
   * `invokeDynamic` is very special. It generates a new call site at run-time using a static "bootstrap" method, then caches the site and invokes it. Depending on the semantics of the bootstrap method the call site has the samen semantics or a different semantics the next time it is invoked.
      * The first type in `desc` of `invokeDynamic must be the receiver type if the method is not static.
      * The first argument in `args` is then also the receiver itself and not one of  the method's arguments.

## data Instruction {#lang-flybytes-Syntax-Instruction}
The JVM low-level instruction set

```rascal
data Instruction  
     = LABEL(str label)
     | LINENUMBER(int line, str label)
     | LOCALVARIABLE(str name, Type \type, str \start, str end, int var)
     | TRYCATCH(Type \type, str \start, str end, str handler)
     | NOP()
     | ACONST_NULL()
     | ICONST_M1()
     | ICONST_0()
     | ICONST_1()
     | ICONST_2()
     | ICONST_3()
     | ICONST_4()
     | ICONST_5()
     | LCONST_0()
     | LCONST_1()
     | FCONST_0()
     | FCONST_1()
     | FCONST_2()
     | DCONST_0()
     | DCONST_1()
     | IALOAD()
     | LALOAD()
     | FALOAD()
     | DALOAD()
     | AALOAD()
     | BALOAD()
     | CALOAD()
     | SALOAD()
     | IASTORE()
     | LASTORE()
     | FASTORE()
     | DASTORE()
     | AASTORE()
     | BASTORE()
     | CASTORE()
     | SASTORE()
     | POP()
     | POP2()
     | DUP()
     | DUP_X1()
     | DUP_X2()
     | DUP2()
     | DUP2_X1()
     | DUP2_X2()
     | SWAP()
     | IADD()
     | LADD()
     | FADD()
     | DADD()
     | ISUB()
     | LSUB()
     | FSUB()
     | DSUB()
     | IMUL()
     | LMUL()
     | FMUL()
     | DMUL()
     | IDIV()
     | LDIV()
     | FDIV()
     | DDIV()
     | IREM()
     | LREM()
     | FREM()
     | DREM()
     | INEG()
     | LNEG()
     | FNEG()
     | DNEG()
     | ISHL()
     | LSHL()
     | ISHR()
     | LSHR()
     | IUSHR()
     | LUSHR()
     | IAND()
     | LAND()
     | IOR()
     | LOR()
     | IXOR()
     | LXOR()
     | I2L()
     | I2F()
     | I2D()
     | L2I()
     | L2F()
     | L2D()
     | F2I()
     | F2L()
     | F2D()
     | D2I()
     | D2L()
     | D2F()
     | I2B()
     | I2C()
     | I2S()
     | LCMP()
     | FCMPL()
     | FCMPG()
     | DCMPL()
     | DCMPG()
     | IRETURN()
     | LRETURN()
     | FRETURN()
     | DRETURN()
     | ARETURN()
     | RETURN()
     | ARRAYLENGTH()
     | ATHROW()
     | MONITORENTER()
     | MONITOREXIT()
     | ILOAD(int var)
     | LLOAD(int var)
     | FLOAD(int var)
     | DLOAD(int var)
     | ALOAD(int var)
     | ISTORE(int var)
     | LSTORE(int var)
     | FSTORE(int var)
     | DSTORE(int var)
     | ASTORE(int var)
     | RET(int var)
     | BIPUSH(int val)
     | SIPUSH(int val)
     | NEWARRAY(Type element)
     | LDC(Type \type, value constant)
     | IINC(int var, int inc)
     | IFEQ(str label)
     | IFNE(str label)
     | IFLT(str label)
     | IFGE(str label)
     | IFGT(str label)
     | IFLE(str label)
     | IF_ICMPEQ(str label)
     | IF_ICMPNE(str label)
     | IF_ICMPLT(str label)
     | IF_ICMPGE(str label)
     | IF_ICMPGT(str label)
     | IF_ICMPLE(str label)
     | IF_ACMPEQ(str label)
     | IF_ACMPNE(str label)
     | GOTO(str label)
     | JSR(str label)
     | IFNULL(str label)
     | IFNONNULL(str label)
     | TABLESWITCH(int min, int max, str defaultLabel, list[str] labels)
     | LOOKUPSWITCH(str defaultLabel, list[int] keys, list[str] labels)
     | GETSTATIC(Type class, str name, Type \type)
     | PUTSTATIC(Type class, str name, Type \type)
     | GETFIELD(Type class, str name, Type \type)
     | PUTFIELD(Type class, str name, Type \type)
     | INVOKEVIRTUAL(Type class, Signature desc, bool isInterface)
     | INVOKESPECIAL(Type class, Signature desc, bool isInterface)
     | INVOKESTATIC(Type class, Signature desc, bool isInterface)
     | INVOKEINTERFACE(Type class, Signature desc, bool isInterface)
     | INVOKEDYNAMIC(Signature desc, BootstrapCall handle)
     | NEW(Type \type)
     | ANEWARRAY(Type \type)
     | CHECKCAST(Type \type)
     | INSTANCEOF(Type \type)
     | MULTIANEWARRAY(Type \type, int numDimensions)
     | exp(Exp expression, str label="")
     | stat(Stat statement, str label="")
     ;
```

## function defVal {#lang-flybytes-Syntax-defVal}

```rascal
Exp defVal(boolean())

Exp defVal(integer())

Exp defVal(long())

Exp defVal(byte())

Exp defVal(character())

Exp defVal(short())

Exp defVal(float())

Exp defVal(double())

Exp defVal(object(str _))

Exp defVal(array(Type _))

Exp defVal(string())
```

## function object {#lang-flybytes-Syntax-object}

Object is the top of the JVMs type system

```rascal
Type object()
```

## function incr {#lang-flybytes-Syntax-incr}

Generates `name+=i;`

```rascal
Stat incr(str name, int i)
```

## function invokeSuper {#lang-flybytes-Syntax-invokeSuper}

Generates `super(f1, f2); for a given anonymous constructor of type (F1 f1, F2 f2)`

```rascal
Stat invokeSuper(list[Type] formals, list[Exp] args)
```

## function invokeSuper {#lang-flybytes-Syntax-invokeSuper}

Generates `super();`

```rascal
Stat invokeSuper()
```

## function main {#lang-flybytes-Syntax-main}

Generates a main method `public static final void main(String[] args) { block }`

```rascal
Method main(str args, list[Stat] block)
```

## function method {#lang-flybytes-Syntax-method}

Short-hand for generating a normal method

```rascal
Method method(Modifier access, Type ret, str name, list[Formal] args, list[Stat] block)
```

## function method {#lang-flybytes-Syntax-method}

Short-hand for generating a normal public method

```rascal
Method method(Type ret, str name, list[Formal] args, list[Stat] block)
```

## function staticMethod {#lang-flybytes-Syntax-staticMethod}

Short-hand for generating a static method

```rascal
Method staticMethod(Modifier access, Type ret, str name, list[Formal] args, list[Stat] block)
```

## function staticMethod {#lang-flybytes-Syntax-staticMethod}

Short-hand for generating a public static method

```rascal
Method staticMethod(Type ret, str name, list[Formal] args, list[Stat] block)
```

## function constructor {#lang-flybytes-Syntax-constructor}

Short-hand for generating a constructor.

```rascal
Method constructor(Modifier access, list[Formal] formals, list[Stat] block)
```

#### Pitfalls

Don't forgot to generate a super call.

## function new {#lang-flybytes-Syntax-new}

"new" short-hand with parameters

```rascal
Exp new(Type class, list[Type] argTypes, list[Exp] args)
```

## function new {#lang-flybytes-Syntax-new}

"new" short-hand, without parameters

```rascal
Exp new(Type class)
```

## function this {#lang-flybytes-Syntax-this}

Load the standard "this" reference for every object.

```rascal
Exp this()
```

#### Pitfalls

This works only inside non-static methods and inside constructors

## function current {#lang-flybytes-Syntax-current}

the `<current>` class refers to the class currently being compiled, for convenience's sake.

```rascal
Type current()
```

## function getField {#lang-flybytes-Syntax-getField}

Load a field from the currently compiled class

```rascal
Exp getField(Type \type, str name)
```

## function getStatic {#lang-flybytes-Syntax-getStatic}

Load a static field from the currently compiled class

```rascal
Exp getStatic(Type \type, str name)
```

## function putField {#lang-flybytes-Syntax-putField}

Store a field in the currently compiled class

```rascal
Stat putField(Type \type, str name, Exp arg)
```

## function putStatic {#lang-flybytes-Syntax-putStatic}

Store a static field in the currently defined class

```rascal
Stat putStatic(Type \type, str name, Exp arg)
```

## function invokeStatic {#lang-flybytes-Syntax-invokeStatic}

Invoke a static method on the currently defined class

```rascal
Exp invokeStatic(Signature desc, list[Exp] args)
```

## function invokeSpecial {#lang-flybytes-Syntax-invokeSpecial}

Invoke a method on the currently defined class using invokeSpecial

```rascal
Exp invokeSpecial(Exp receiver, Signature desc, list[Exp] args)
```

## function invokeVirtual {#lang-flybytes-Syntax-invokeVirtual}

Invoke a method on the currently defined class using invokeVirtual

```rascal
Exp invokeVirtual(Exp receiver, Signature desc, list[Exp] args)
```

## function invokeInterface {#lang-flybytes-Syntax-invokeInterface}

Invoke a method on the currently defined interface using invokeInterface

```rascal
Exp invokeInterface(Exp receiver, Signature desc, list[Exp] args)
```

## function iconst {#lang-flybytes-Syntax-iconst}

```rascal
Exp iconst(int i)
```

## function sconst {#lang-flybytes-Syntax-sconst}

```rascal
Exp sconst(int i)
```

## function bconst {#lang-flybytes-Syntax-bconst}

```rascal
Exp bconst(int i)
```

## function cconst {#lang-flybytes-Syntax-cconst}

```rascal
Exp cconst(int i)
```

## function zconst {#lang-flybytes-Syntax-zconst}

```rascal
Exp zconst(bool i)
```

## function jconst {#lang-flybytes-Syntax-jconst}

```rascal
Exp jconst(int i)
```

## function sconst {#lang-flybytes-Syntax-sconst}

```rascal
Exp sconst(str i)
```

## function dconst {#lang-flybytes-Syntax-dconst}

```rascal
Exp dconst(real i)
```

## function fconst {#lang-flybytes-Syntax-fconst}

```rascal
Exp fconst(real i)
```

## data BootstrapCall {#lang-flybytes-Syntax-BootstrapCall}

A bootstrap handle is a name of a static method (as defined by its host class,
its name and its type signature), and a list of "constant" arguments. 


```rascal
data BootstrapCall  
     = bootstrap(Type class, Signature desc, list[CallSiteInfo] args)
     ;
```


These **constant**
arguments can be used to declare properties of the call site which can then be used by
the bootstrap method to define in which way the dynamic call must be resolved. So these
argument help to avoid having to define a combinatorially large number of bootstrap methods
(one for each call site situation).  

It's advisable to use the convenience function below to create a `BootstrapCall` instance:
* `bootstrap(Type class, str name, list[BootstrapInfo] args)`
  
That function makes sure to line up the additional information in the extra arguments about 
the call site with the static type of the static bootstrap method.

## function bootstrap {#lang-flybytes-Syntax-bootstrap}

generate a bootstrap call with all the required standard parameters, and optionally more.

```rascal
BootstrapCall bootstrap(Type class, str name, list[CallSiteInfo] args)
```

#### Benefits


* A raw BootstrapCall must return a CallSite and take a MethodHandle.Lookup, a string and a MethodType as the first three parameters.    
This convenience function guarantees that this the true, but allows for adding additional static information about
the call site.
* The types of the additional parameters are inferred automatically from the CallSiteInfo structures, so 
you do not have to distribute this information during code generation.

#### Pitfalls


* the signature of the bootstrap method `name` in `class` (which you might have written in Java, or generated
in a different part of your compiler) must be the exactly the same as generated here.

## function bootstrap {#lang-flybytes-Syntax-bootstrap}

Generate a basic bootstrap caller with only the minimally required information for a dynamic invoke.

```rascal
BootstrapCall bootstrap(str name, list[CallSiteInfo] args)
```

#### Benefits

This is the starting point for any use of invokeDynamic. Get this working first and add additional information
later

#### Pitfalls

Writing bootstrap method implementations is hard.

## function bootstrapMethod {#lang-flybytes-Syntax-bootstrapMethod}


Convenience function to use existing BootstrapCall information to generate a fitting bootstrap 
method to call.


```rascal
Method bootstrapMethod(BootstrapCall b, list[Stat] body)
```


This function mirrors the `bootstrap` function; it generates the method referenced by the output of that function.

#### Benefits


* the bootstrap method and the bootstrapCall information have to be aligned perfectly. If you use the pair
of functions `bootstrapMethod` and `bootstrap` together then this alignment is guaranteed. 

#### Pitfalls


* generating the body of a bootstrap method can be challenging. It is recommended to first write the example
in Java and decompile the resulting method, and copy the result into the `body` parameter of this function.
* another solution is to keep the source of the bootstrap method completely in Java, but making sure the 
signatures keep matching inside the BootstrapCall data. This is usually possible; when the bootstrap method
is generic enough.

## data CallSiteInfo {#lang-flybytes-Syntax-CallSiteInfo}

```rascal
data CallSiteInfo  
     = stringInfo(str s)
     | classInfo(str name)
     | integerInfo(int i)
     | longInfo(int l)
     | floatInfo(real f)
     | doubleInfo(real d)
     | methodTypeInfo(Signature desc)
     | virtualHandle(Type class, str name, Signature desc)
     | specialHandle(Type class, str name, Signature desc, Type caller)
     | getterHandle(Type class, str name, Type \type)
     | setterHandle(Type class, str name, Type \type)
     | staticGetterHandle(Type class, str name, Type \type)
     | staticSetterHandle(Type class, str name, Type \type)
     | constructorHandle(Type class, Signature desc)
     ;
```

## function callsiteInfoType {#lang-flybytes-Syntax-callsiteInfoType}

Type inference for callsiteInfo parameters to bootstrap methods.

```rascal
Type callsiteInfoType(stringInfo(_))

Type callsiteInfoType(classInfo(_))

Type callsiteInfoType(integerInfo(_))

Type callsiteInfoType(longInfo(_))

Type callsiteInfoType(floatInfo(_))

Type callsiteInfoType(doubleInfo(_))

Type callsiteInfoType(virtualHandle(_,_,_))

Type callsiteInfoType(specialHandle(_,_,_,_))

Type callsiteInfoType(getterHandle(_,_,_))

Type callsiteInfoType(setterHandle(_,_,_))

Type callsiteInfoType(staticGetterHandle(_,_,_))

Type callsiteInfoType(staticSetterHandle(_,_,_))

Type callsiteInfoType(constructorHandle(_,_))

Type callsiteInfoType(methodTypeInfo(_))
```

