---
title: "module util::LanguageServer"
id: LanguageServer
slug: /Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-0.41.1</div> <div class="theme-doc-version-badge badge badge--secondary">org.rascalmpl.rascal-lsp-2.22.0-RC3</div>

Bridges {DSL,PL,Modeling} language features to the language server protocol.
#### Usage

```rascal
import util::LanguageServer;
```

#### Dependencies
```rascal
import util::Reflective;
import analysis::diff::edits::TextEdits;
import IO;
import ParseTree;
import Message;
import Exception;
```

#### Description


Using the [Register Language](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-registerLanguage) function you can connect any parsers, checkers,
source-to-source transformers, visualizers, etc. that are made with Rascal, to the
Language Server Protocol.

#### Benefits


* Turn your language implementation into an interactive IDE at almost zero cost.


## data Language {#util-LanguageServer-Language}
Definition of a language server by its meta-data.

```rascal
data Language  
     = language(PathConfig pcfg, str name, set[str] extensions, str mainModule, str mainFunction)
     ;
```


The [Register Language](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-registerLanguage) function takes this as its parameter to generate and run
a fresh language protocol server. Every language server is run in its own Rascal execution
environment. The [Language](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Language) data-type defines the parameters of this run-time, such
that [Register Language](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-registerLanguage) can boot and initialize new instances.

* `pcfg` sets up search paths for Rascal modules and libraries required to run the language server
* `name` is the name of the language
* `extensions` are the file extensions that bind this server to editors of files with these extensions.
* `mainModule` is the Rascal module to load to start the language server
* `mainFunction` is a function of type `set[LanguageService] ()` that produces the implementation of the language server
as a set of collaborating [Language Service](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-LanguageService)s.

#### Benefits


* each registered language is run in its own Rascal run-time environment.
* reloading a language is always done in a fresh environment.
* instances of [Language](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Language) can be easily serialized and communicated in interactive language engineering environments.

#### Pitfalls


* even though [Register Language](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-registerLanguage) is called in a given run-time environment,
the registered language runs in another instance of the JVM and of Rascal.

## function language {#util-LanguageServer-language}

```rascal
Language language(PathConfig pcfg, str name, str extension, str mainModule, str mainFunction)
```

## alias Parser {#util-LanguageServer-Parser}

Function profile for parser contributions to a language server

```rascal
Tree (str _input, loc _origin)
```


The parser function takes care of parsing the tree once after every change in the IDE.
This parse tree is then used for both syntax highlighting and other language server functions.

#### Pitfalls


* use `ParseTree::parser` instead of writing your own function to ensure syntax highlighting is fast

## alias Summarizer {#util-LanguageServer-Summarizer}

Function profile for summarizer contributions to a language server

```rascal
Summary (loc _origin, Tree _input)
```

## alias Focus {#util-LanguageServer-Focus}

A focus provides the currently selected language constructs around the cursor.

```rascal
list[Tree]
```


A [Focus](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Focus) list starts with the bottom tree, commonly a lexical identifier if
the cursor is inside an identifer, and ends with the start non-terminal (the whole tree). Everything
in between is a spine of language constructs [Parse Tree](../../../..//Library/ParseTree.md) nodes between the top and the bottom node.

The location of each element in the focus list is around (inclusive) the current cursor selection.
This means that:
* every next element in the list is (one of) the parents of the previous.
* typically the list starts with a smallest tree and ends with the entire `start` tree.
* singleton lists may occur in case the cursor is on a layout or literal element of the top production.
* the `start[X]` tree is typically preceded by the `X` tree.
* the first tree is a whole lexical tree if the cursor is inside an identifier or constant
* the first tree is a (context-free) syntax tree if the cursor is on the whitespace in between literals and lexicals.
* the focus list may be empty in case of top-level ambiguity or parse errors.

The [Focus](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Focus) is typically provided to the [Language Service](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-LanguageService)s below, such that language
engineers can provide language-directed tools, which are relevant to the current interest
of the user.

#### Benefits


* All functions that accept a [Focus](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Focus) can use list matching to filter locations of interest.

#### Pitfalls


* Functions that use list matching on their [Focus](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Focus) parameter must provide a default that returns the empty list or empty set.

## alias Outliner {#util-LanguageServer-Outliner}

Function profile for outliner contributions to a language server

```rascal
list[DocumentSymbol] (Tree _input)
```

## alias LensDetector {#util-LanguageServer-LensDetector}

Function profile for lenses contributions to a language server

```rascal
rel[loc src, Command lens] (Tree _input)
```

## alias OrderedLensDetector {#util-LanguageServer-OrderedLensDetector}

Function profile for lenses contributions to a language server

```rascal
lrel[loc src, Command lens] (Tree _input)
```

## alias CommandExecutor {#util-LanguageServer-CommandExecutor}

Function profile for executor contributions to a language server

```rascal
value (Command _command)
```

## alias InlayHinter {#util-LanguageServer-InlayHinter}

Function profile for inlay contributions to a language server

```rascal
list[InlayHint] (Tree _input)
```

## alias Documenter {#util-LanguageServer-Documenter}

```rascal
set[str] (loc _origin, Tree _fullTree, Tree _lexicalAtCursor)
```

## alias CodeActionContributor {#util-LanguageServer-CodeActionContributor}

```rascal
list[CodeAction] (Focus _focus)
```

## alias Definer {#util-LanguageServer-Definer}

Function profile for definer contributions to a language server

```rascal
set[loc] (loc _origin, Tree _fullTree, Tree _lexicalAtCursor)
```

## alias Referrer {#util-LanguageServer-Referrer}

Function profile for referrer contributions to a language server

```rascal
set[loc] (loc _origin, Tree _fullTree, Tree _lexicalAtCursor)
```

## alias Implementer {#util-LanguageServer-Implementer}

Function profile for implementer contributions to a language server

```rascal
set[loc] (loc _origin, Tree _fullTree, Tree _lexicalAtCursor)
```

## data LanguageService {#util-LanguageServer-LanguageService}
Each kind of service contibutes the implementation of one (or several) IDE features.

```rascal
data LanguageService  
     = parsing(Tree (str _input, loc _origin) parsingService
        , bool usesSpecialCaseHighlighting = true)
     | analysis(Summary (loc _origin, Tree _input) analysisService
        , bool providesDocumentation = true
        , bool providesHovers = providesDocumentation
        , bool providesDefinitions = true
        , bool providesReferences = true
        , bool providesImplementations = true)
     | build(Summary (loc _origin, Tree _input) buildService
        , bool providesDocumentation = true
        , bool providesHovers = providesDocumentation
        , bool providesDefinitions = true
        , bool providesReferences = true
        , bool providesImplementations = true)
     | documentSymbol(list[DocumentSymbol] (Tree _input) documentSymbolService)
     | codeLens      (lrel[loc src, Command lens] (Tree _input) codeLensService)
     | inlayHint     (list[InlayHint] (Tree _input) inlayHintService)
     | execution     (value (Command _command) executionService)
     | hover         (set[str] (Focus _focus) hoverService)
     | definition    (set[loc] (Focus _focus) definitionService)
     | references    (set[loc] (Focus _focus) referencesService)
     | implementation(set[loc] (Focus _focus) implementationService)
     | codeAction    (list[CodeAction] (Focus _focus) codeActionService)
     | rename        (tuple[list[DocumentEdit], set[Message]] (Focus _focus, str newName) renameService
        , loc (Focus _focus) prepareRenameService = defaultPrepareRenameService)
     | didRenameFiles(tuple[list[DocumentEdit], set[Message]] (list[DocumentEdit] fileRenames) didRenameFilesService)
     | selectionRange(list[loc](Focus _focus) selectionRangeService)
     | callHierarchy (
        list[CallHierarchyItem] (Focus _focus) prepareService,
        lrel[CallHierarchyItem item, loc call] (CallHierarchyItem _ci, CallDirection _dir) callsService)
     ;
```


Each LanguageService constructor provides one aspect of definining the language server protocol (LSP).
Their names coincide with the services which are documented [here](https://microsoft.github.io/language-server-protocol/).

* The [Parsing](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-parsing) service that maps source code strings to a [Tree](../../../..//Library/ParseTree.md#ParseTree-Tree) is essential and non-optional.
All other other services are optional.
   * By providing a parser which produces annotated parse [Tree](../../../..//Library/ParseTree.md#ParseTree-Tree)s, editor features such as parse error locations, syntax highlighting and
selection assistance are immediately enabled.
   * The [Parsing](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-parsing) service is activated after every change in an editor document (when a suitable pause has occurred)
   * All downstream services are based on the [Tree](../../../..//Library/ParseTree.md#ParseTree-Tree) that is produced here. In
particular downstream services make use of the `src` origin fields that the parser must produce.
   * Parsers can be obtained automatically using the [parser](../../../..//Library/ParseTree.md#ParseTree-parser) or [parsers](../../../..//Library/ParseTree.md#ParseTree-parsers) functions, like so `parser(#start[Program])`.
Like this a fast parser is obtained that does not require a global interpreter lock. If you pass in a normal Rascal function, which is fine, the global
interpreter lock will make the editor services less responsive.
   * Currently, `@category` tags are ignored in the following special case:
        * if a parse tree has a `syntax` non-terminal node `n` with a category
          (either declared as part of `n`, or inherited from an ancestors),
        * and if `n` has a `syntax` non-terminal node `m` as a child,
        * then the category of `n` is ignored in the subtree rooted at `m`
          (regardless of whether a category is declared as part of `m`).
     This special case is deprecated and will be removed in a future release. In
     anticipation of the removal, users that rely on this special case for
     syntax highlighting can update their grammars and explicitly opt-out of the
     special case by passing `usesSpecialCaseHighlighting = false` when
     registering the [Parsing](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-parsing) service.
   * You can enable error recovery in the parser like by setting `allowRecovery` to `true`: `parser(#start[Program], allowRecovery=true)`.
With error recovery enabled  "hard" parse errors can still occur but that will be rare. In most cases parsing with error recovery enabled
will produce a parse tree with error nodes. Syntax highlighting will still work on such trees. Note that any contributions that you add must be
able to handle such error trees or unexpected things will happen like strange results and crashes. More information on error recovery and error trees
can be found in [Production](../../../..//Library/ParseTree.md#ParseTree-Production), [parser](../../../..//Library/ParseTree.md#ParseTree-parser), and [ParseErrorRecovery](../../../..//Library/util/ParseErrorRecovery.md).
* The [Analysis](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analysis) service indexes a file as a [Summary](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Summary), offering precomputed relations for looking up
hover documentation, definition with uses, references to declarations, implementations of types and compiler errors and warnings.
   * [Analysis](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analysis) focuses on their own file, but may reuse cached or stored indices from other files.
   * [Analysis](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analysis) has to be quick since they run in an interactive editor setting.
   * [Analysis](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analysis) may store previous results (in memory) for incremental updates.
   * [Analysis](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analysis) is triggered on-demand during typing, in a short typing pause. So you have to provide a reasonable fast function (0.5 seconds is a good target response time).
   * [Analysis](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analysis) pushes their result on a local stack; which is efficiently queried by the LSP features on-demand.
* The [build](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-build) service is similar to an [Analysis](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analysis), but it may perform computation-heavier additional checks or take time generate source code and binary code that makes the code in the editor executable.
   * [build](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-build)s typically run whole-program analyses and compilation steps.
   * [build](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-build)s have side-effects, they store generated code or code indices for future usage by the next build step, or by the next analysis step.
   * [build](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-build)s are triggered on _save-file_ events; they _push_ information to an internal cache.
   * Warning: [build](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-build)s are _not_ triggered when a file changes on disk outside of VS Code; instead, this results in a change event (not a save event), which triggers the [Analyzer](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analyzer).
   * If `providesDocumentation` is false, then the [Hover](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-hover) service may be activated. Same for `providesDefinitions` and `providesDocumentation`
))
* the following contributions are _on-demand_ (pull) versions of information also provided by the [Analysis](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analysis) and [build](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-build) summaries.
   * you can provide these more lightweight on-demand services _instead of_ the [Summary](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Summary) versions.
   * these functions are run synchronously after a user interaction. The run-time of each service corresponds directly to the UX response time.
   * a [Hover](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-hover) service is a fast and location specific version of the `documentation` relation in a [Summary](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Summary).
   * a [Definition](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-definition) service is a fast and location specific version of the `definitions` relation in a [Summary](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Summary).
   * a [References](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-references) service is a fast and location specific version of the `references` relation in a [Summary](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Summary).
   * an [Implementation](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-implementation) service is a fast and location specific version of the `implementations` relation in a [Summary](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Summary).
* The [Document Symbol](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-documentSymbol) service maps a source file to a pretty hierarchy for visualization in the "outline" view and "symbol search" features.
* The [Code Lens](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-codeLens) service discovers places to add "lenses" (little views embedded in the editor on a separate line) and connects commands to execute to each lense
* The [Inlay Hint](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-inlayHint) service discovers plances to add "inlays" (little views embedded in the editor on the same line). Unlike [Lenses](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-lenses) inlays do not offer command execution.
* The [Execution](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-execution) service executes the commands registered by [Lenses](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-lenses) and [Inlay Hinter](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-inlayHinter)s.
* The [Actions](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-actions) service discovers places in the editor to add "code actions" (little hints in the margin next to where the action is relevant) and connects [Code Action](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CodeAction)s to execute when the users selects the action from a menu.
* The [rename](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-rename) service renames an identifier by collecting the edits required to rename all occurrences of that identifier. It might fail and report why in diagnostics.
   * The optional `prepareRename` service argument discovers places in the editor where a [rename](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-rename) is possible. If renameing the location is not supported, it should throw an exception.
* The [Did Rename Files](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-didRenameFiles) service collects [Document Edit](../../../..//Library/analysis/diff/edits/TextEdits.md#analysis-diff-edits-TextEdits-DocumentEdit)s corresponding to renamed files (e.g. to rename a class when the class file was renamed). The IDE applies the edits after moving the files. It might fail and report why in diagnostics.
* The [Selection Range](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-selectionRange) service discovers selections around a cursor, that a user might want to select. It expects the list of source locations to be in ascending order of size (each location should be contained by the next) - similar to [Focus](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Focus) trees.
* The [Call Hierarchy](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-callHierarchy) service discovers callable definitions and call sites. It consists of two subservices.
    1. The first argument, `callableItem`, computes [Call Hierarchy Item](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CallHierarchyItem)s (definitions) for a given cursor.
    2. The second argument, `calculateCalls`, computes [Incoming](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-incoming) or [Outgoing](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-outgoing) calls (uses) of a given [Call Hierarchy Item](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CallHierarchyItem) `ci`. It returns a list relation of [Call Hierarchy Item](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CallHierarchyItem)s and the location(s) of the call(s) to `ci` these definitions have.

Many services receive a [Focus](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Focus) parameter. The focus lists the syntactical constructs under the current cursor, from the current
leaf all the way up to the root of the tree. This list helps to create functionality that is syntax-directed, and always relevant to the
programmer.

To start developing an LSP extension step-by-step:
1. first write a SyntaxDefinition in Rascal and register it via the [Parsing](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-parsing) service. Use [Register Language](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-registerLanguage) from the terminal [REPL](../../../..//Library/util/REPL.md#util-REPL-REPL) to
test it immediately. Create some example files for your language to play around with.
2. either make an [Analysis](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analysis) service that produces a [Summary](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Summary) _or_ start [Hover](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-hover), [Definition](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-definition), [References](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-references) and [Implementation](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-implementation)
lookup services. Each of those four services require the same information that is useful for filling a [Summary](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Summary) with an [Analysis](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analysis) or a [Builder](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-builder).
3. the [Document Symbol](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-documentSymbol) service is next, good for the outline view and also quick search features.
4. the to add interactive features, optionally [Inlay Hint](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-inlayHint), [Code Lens](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-codeLens) and [Code Action](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-codeAction) can be created to add visible hooks in the UI to trigger
your own [Code Action](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CodeAction)s and Commands
   * create an [Execution](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-execution) service to give semantics to each command. This includes creating [Document Edit](../../../..//Library/analysis/diff/edits/TextEdits.md#analysis-diff-edits-TextEdits-DocumentEdit)s but also [IDEServices](../../../..//Library/util/IDEServices.md)
   can be used to have interesting effects in the IDE.
   * [Code Action](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CodeAction)s can also be attached to error, warning and into [Message](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Message)s as a result of [Parsing](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-parsing), [Analysis](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-analysis) or [build](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-build).
   Such actions will lead to "quick-fix" UX options in the editor.

#### Benefits


* You can create editor services thinking only of your programming language or domain-specific language constructs. All of the communication
and (de)serialization and scheduling is taken care of.
* It is always possible and useful to test your services manually in the [REPL](../../../..//Library/util/REPL.md#util-REPL-REPL). This is the preferred way of testing and debugging language services.
* Except for the [Parsing](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-parsing) service, all services are independent of each other. If one fails, or is removed, the others still work.
* Language services in general can be unit-tested easily by providing example parse trees and testing properties of their output. Write lots of test functions!
* LanguageServices are editor-independent/IDE-independent via the LSP protocol. In principle they can work with any editor that implements LSP 3.17 or higher.
* Older Eclipse DSL plugins via the rascal-eclipse plugin are easily ported to [LanguageServer](../../../../Packages/org.rascalmpl.rascal-lsp//Library/util/LanguageServer.md).

#### Pitfalls


* If one of the services does not type-check in Rascal, or throws an exception at [Register Language](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-registerLanguage) time, the extension fails completely. Typically the editor produces a parse error on the first line of the code. The
failure is printed in the log window of the IDE.
* Users have expectations with the concepts of [References](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-references), [Definition](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-definition), [Implementation](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-implementation) which are based on
typical programming language concepts. Since these are all just `rel[loc, loc]` it can be easy to confound them.
   * [References](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-references) point from declarations sites to use sites
   * [Definition](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-definition) points the other way around, from a use to the declaration, but only if a value is associated there explicitly or implicitly.
   * [Implementation](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-implementation) points from abstract declarations (interfaces, classes, function signatures) to more concrete realizations of those declarations.
* `providesDocumentation` is deprecated. Use `providesHovers` instead.

## function defaultPrepareRenameService {#util-LanguageServer-defaultPrepareRenameService}

```rascal
loc defaultPrepareRenameService(Focus _:[Tree tr, *_])

default loc defaultPrepareRenameService(Focus focus)
```

## data CallHierarchyItem {#util-LanguageServer-CallHierarchyItem}
A node in a call hierarchy, either a caller or a callee.

```rascal
data CallHierarchyItem  
     = callHierarchyItem(
        str name,
        DocumentSymbolKind kind,
        loc src,                            // location of the definition
        loc selection,                      // location of the name of the definition
        set[DocumentSymbolTag] tags = {},
        str detail = "",                    // detailed description, e.g. the function signature
        CallHierarchyData \data = none()    // shared state between `callHierarchy::prepareService` and `callHierarchy::callsService`
    )
     ;
```


A [Call Hierarchy Item](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CallHierarchyItem) represents a single function, method, or procedure in the call hierarchy.
* `name` is the name of the callable/calling entity.
* `kind` is the [Document Symbol Kind](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-DocumentSymbolKind) of the callable/calling entity, e.g., function, method, constructor, etc.
* `src` is the location of the definition of the callable/calling entity.
* `selection` is the location of the name of the definition of the callable/calling entity, or another range within `src` to select when the hierarchy item is clicked.
* `tags` are additional metadata tags for the item, e.g., `deprecated`.
* `detail` has additional information about the callable/calling entity, e.g., the function signature.
* `data` can be used to store state that is shared between the `prepareService` and `callsService`.

## data CallHierarchyData {#util-LanguageServer-CallHierarchyData}

```rascal
data CallHierarchyData  
     = none()
     ;
```

## data CallDirection {#util-LanguageServer-CallDirection}

```rascal
data CallDirection  
     = incoming()
     | outgoing()
     ;
```

## function parser {#util-LanguageServer-parser}

Construct a `parsing` ((LanguageService))

```rascal
LanguageService parser(Parser parser)
```

## function lenses {#util-LanguageServer-lenses}

Construct a ((codeLens)) ((LanguageService))

```rascal
LanguageService lenses(LensDetector detector)
```


Not only translates to the old name of the LanguageService,
it also maps the list to an arbitrarily ordered set as it was before.

#### Benefits


* If you need your lenses in a stable order in the editor,
use the [Code Lens](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-codeLens) constructor instead to provide a function that
uses an ordered list.

## function actions {#util-LanguageServer-actions}

Construct a ((codeAction)) ((LanguageService))

```rascal
LanguageService actions(CodeActionContributor contributor)
```

## function builder {#util-LanguageServer-builder}

Construct a ((util::LanguageServer::build)) ((LanguageService))

```rascal
LanguageService builder(Summarizer summarizer)
```

## function outliner {#util-LanguageServer-outliner}

Construct a ((documentSymbol)) ((LanguageService))

```rascal
LanguageService outliner(Outliner outliner)
```

## function inlayHinter {#util-LanguageServer-inlayHinter}

Construct a ((inlayHint)) ((LanguageService))

```rascal
LanguageService inlayHinter(InlayHinter hinter)
```

## function executor {#util-LanguageServer-executor}

Construct a ((execution)) ((LanguageService))

```rascal
LanguageService executor(CommandExecutor executor)
```

## function documenter {#util-LanguageServer-documenter}

```rascal
LanguageService documenter(Documenter d)
```

## function definer {#util-LanguageServer-definer}

```rascal
LanguageService definer(Definer d)
```

## function referrer {#util-LanguageServer-referrer}

Registers an old-style ((Referrer))

```rascal
LanguageService referrer(Referrer d)
```

## function implementer {#util-LanguageServer-implementer}

Registers an old-style ((Implementer))

```rascal
LanguageService implementer(Implementer d)
```

## function summarizer {#util-LanguageServer-summarizer}

A summarizer collects information for later use in interactive IDE features.

```rascal
LanguageService summarizer(Summarizer summarizer
        , bool providesDocumentation = true
        , bool providesHovers = providesDocumentation
        , bool providesDefinitions = true
        , bool providesReferences = true
        , bool providesImplementations = true)
```

## function analyzer {#util-LanguageServer-analyzer}

An analyzer collects information for later use in interactive IDE features.

```rascal
LanguageService analyzer(Summarizer summarizer
        , bool providesDocumentation = true
        , bool providesDefinitions = true
        , bool providesReferences = true
        , bool providesImplementations = true)
```

## data Summary {#util-LanguageServer-Summary}
A model encodes all IDE-relevant information about a single source file.

```rascal
data Summary  
     = summary(loc src,
    rel[loc, Message] messages = {},
    rel[loc, str]     documentation = {},
    rel[loc, str]     hovers = documentation,
    rel[loc, loc]     definitions = {},
    rel[loc, loc]     references = {},
    rel[loc, loc]     implementations = {}
)
     ;
```


* `src` refers to the "compilation unit" or "file" that this model is for.
* `messages` collects all the errors, warnings and error messages.
* `documentation` is the deprecated name for `hovers`
* `hovers` maps uses of concepts to a documentation message that can be shown as a hover.
* `definitions` maps use locations to declaration locations to implement "jump-to-definition".
* `references` maps declaration locations to use locations to implement "jump-to-references".
* `implementations` maps the declaration of a type/class to its implementations "jump-to-implementations".

## data Completion {#util-LanguageServer-Completion}

```rascal
data Completion  
     = completion(str newText, str proposal=newText)
     ;
```

## data DocumentSymbol {#util-LanguageServer-DocumentSymbol}
DocumentSymbol encodes a sorted and hierarchical outline of a source file

```rascal
data DocumentSymbol  
     = symbol(
        str name,
        DocumentSymbolKind kind,
        loc range,
        loc selection=range,
        str detail="",
        list[DocumentSymbol] children=[],
        set[DocumentSymbolTag] tags = {}
    )
     ;
```

## data DocumentSymbolKind {#util-LanguageServer-DocumentSymbolKind}

```rascal
data DocumentSymbolKind  
     = \file()
     | \module()
     | \namespace()
     | \package()
     | \class()
     | \method()
     | \property()
     | \field()
     | \constructor()
     | \enum()
     | \interface()
     | \function()
     | \variable()
     | \constant()
     | \string()
     | \number()
     | \boolean()
     | \array()
     | \object()
     | \key()
     | \null()
     | \enumMember()
     | \struct()
     | \event()
     | \operator()
     | \typeParameter()
     ;
```

## data DocumentSymbolTag {#util-LanguageServer-DocumentSymbolTag}

```rascal
data DocumentSymbolTag  
     = \deprecated()
     ;
```

## data CompletionProposal {#util-LanguageServer-CompletionProposal}

```rascal
data CompletionProposal  
     = sourceProposal(str newText, str proposal=newText)
     ;
```

## data Message {#util-LanguageServer-Message}
Attach any command to a message for it to be exposed as a quick-fix code action automatically.

```rascal
data Message (list[CodeAction] fixes = [])
```


The fixes you provide with a message will be hinted at by a light-bulb in the editor's margin.
Every fix listed here will be a menu item in the pop-up menu when the bulb is activated (via short-cut or otherwise).

Note that for a [Code Action](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CodeAction) to be executed, you must either provide `edits` directly and/or handle
a [Command](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Command) and add its execution to the [Command Executor](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CommandExecutor) contribution function.

#### Benefits


* the information required to produce an error message is usually also required for the fix. So this
coupling of message with fixes may come in handy.

#### Pitfalls


* the code for error messaging may become cluttered with code for fixes. It is advisable to only _collect_ information for the fix
and store it in a [Command](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Command) constructor inside the [Code Action](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CodeAction), or to delegate to a function that produces
the right [Document Edit](../../../..//Library/analysis/diff/edits/TextEdits.md#analysis-diff-edits-TextEdits-DocumentEdit)s immediately.
* don't forget to extend [Command](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Command) with a new constructor and [Command Executor](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CommandExecutor) with a new overload to handle that constructor.

## data TextEdit {#util-LanguageServer-TextEdit}
A ((analysis::diff::edits::TextEdits::TextEdit)) with additional context for LSP.

```rascal
data TextEdit (str label = "", str description = label, bool needsConfirmation = false)
```


In LSP, text edits can contain extra information w.r.t. [TextEdit](../../../..//Library/analysis/diff/edits/TextEdits.md#analysis-diff-edits-TextEdits-TextEdit).
* label: Human-readable string that describes the change.
* description: Human-readable string that additionally describes the change, rendered less prominently.
* needsConfirmation: Flags whether the user should confirm this change. By default, this is false, which means that [TextEdit](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-TextEdit)s are applied without user confirmation.

Typically, clients provide options to group edits by label/description when showing them to the user.
See the [LSP documentation](https://microsoft.github.io/language-server-protocol/specifications/lsp/3.17/specification/#changeAnnotation) for more details.

Note: to easily annotate all text edits in a [FileSystemChange](../../../..//Library/analysis/diff/edits/TextEdits.md#analysis-diff-edits-TextEdits-FileSystemChange), use the convenience keywords on [FileSystemChange](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-FileSystemChange).

#### Pitfalls


When `needsConfirmation = false` for all edits, the client will typically apply them without showing any information from the annotations to the user.

## data FileSystemChange {#util-LanguageServer-FileSystemChange}
A ((analysis::diff::edits::TextEdits::FileSystemChange)) with additional context for LSP.

```rascal
data FileSystemChange (str label = "", str description = "", bool needsConfirmation = false)
```


Provides extra context for all contained [TextEdit](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-TextEdit)s at once.

## data Command {#util-LanguageServer-Command}
A Command is a parameter to a CommandExecutor function.

```rascal
data Command (str title="") 
     = noop()
     ;
```


Commands can be any closed term a() pure value without open variables or function/closure values embedded in it). Add any constructor you need to express the execution parameters
of a command.

You write the [Command Executor](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CommandExecutor) to interpret each kind of [Command](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Command) individually.
A ((Command) constructor must have fields or keyword fields that hold the parameters of the
to-be-executed command.

Commands are produced for delayed and optional execution by:
* [Lens Detector](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-LensDetector), where the will be executed if the lens is selected in the editor
* [Code Action Contributor](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CodeActionContributor), where they will appear in context-menus for quick-fix and refactoring
* [Message](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Message), where they will appear in context-menus on lines with error or warning diagnostics

See also [Code Action](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CodeAction); a wrapper for [Command](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Command) for fine-tuning UI interactions.

#### Examples


```rascal
// here we invent a new command name `showFlowDiagram` which is parametrized by a loc:
data Command = showFlowDiagram(loc src);

// and we have our own evaluator that executes the showFlowDiagram command by starting an interactive view:
value evaluator(showFlowDiagram(loc src)) {
    showInteractiveContent(flowDiagram(src));
    return true;
}
```

#### Pitfalls


* Sometimes a command must be wrapped in a [Code Action](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CodeAction) to make it effective (see [Code Action Contributor](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CodeActionContributor) and [Message](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Message) )
* the `noop()` command will always be ignored.
* _never_ add first-class functions or closures as a parameter or keyword field to a `Command`. The Command will
be serialized, sent to the LSP client, and then sent back to the LSP server for execution. Functions can not be
serialized, so that would lead to run-time errors.

## data CodeAction {#util-LanguageServer-CodeAction}
Code actions encapsulate computed effects on source code like quick-fixes, refactorings or visualizations.

```rascal
data CodeAction (
    list[DocumentEdit] edits = [],
    Command command          = noop(),
    str title                = command.title,
    CodeActionKind kind      = quickfix()) 
     = action()
     ;
```


Code actions are an intermediate representation of what is about to happen to the source code that is loaded in the IDE,
or even in a live editor. They communicate what can (possibly) be done to improve the user's code, who might choose one of the options
from a list, or even look at different outcomes ahead-of-time in a preview.

The `edits` and `command` parameters are both optional, and can be provided at the same time as well.

If [Document Edit](../../../..//Library/analysis/diff/edits/TextEdits.md#analysis-diff-edits-TextEdits-DocumentEdit)[edits] are provided then:
1. edits can be used for preview of a quick-fix of refactoring
2. edits are always applied first before any `command` is executed.
3. edits can always be undone via the undo command of the IDE

If a [Command](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Command)[command] is provided, then:
1. The title of the command is shown to the user
2. The user picks this code action (from a list or pressed "OK" in a dialog)
3. Any `edits` (see above) are applied first
4. The command is executed on the server side via the [Command Executor](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CommandExecutor) contribution
   * Many commands use [applyDocumentsEdits](../../../..//Library/util/IDEServices.md#util-IDEServices-applyDocumentsEdits) to provide additional changes to the input
   * Other commands might use [showInteractiveContent](../../../..//Library/util/IDEServices.md#util-IDEServices-showInteractiveContent) to start a linked webview inside of the IDE
   * Also [registerDiagnostics](../../../..//Library/util/IDEServices.md#util-IDEServices-registerDiagnostics) is a typical effect of a [Code Action](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CodeAction) [Command](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Command).
5. The effects of commands can be undone if they where [Document Edit](../../../..//Library/analysis/diff/edits/TextEdits.md#analysis-diff-edits-TextEdits-DocumentEdit)s, but other effects like diagnostics and
interactive content have to be cleaned or closed in their own respective fashions.

#### Benefits


* CodeActions provide tight integration with the user experience in the IDE. Including sometimes previews, and always the undo stack.
* CodeActions can be implemented "on the language level", abstracting from UI and scheduling details. See also [edits](../../../../Library/analysis/diff/edits/index.md) for
tools that can produce lists of [Document Edit](../../../..//Library/analysis/diff/edits/TextEdits.md#analysis-diff-edits-TextEdits-DocumentEdit)s by diffing parse trees or abstract syntax trees.
* `edits` are applied on the latest editor content for the current editor; live to the user.
* [applyDocumentsEdits](../../../..//Library/util/IDEServices.md#util-IDEServices-applyDocumentsEdits) also works on open editor contents for the current editor.
* The parse tree for the current file is synchronized with the call to a [Code Action Contributor](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-CodeActionContributor) such that edits
and input are computed in-sync.

#### Pitfalls


* [applyDocumentsEdits](../../../..//Library/util/IDEServices.md#util-IDEServices-applyDocumentsEdits) and `edits` when pointing to other files than the current one, may
or may not work on the current editor contents. If you want to be safe it's best to only edit the current file.

## data CodeActionKind {#util-LanguageServer-CodeActionKind}
Kinds are used to prioritize menu options and choose relevant icons in the UI.

```rascal
data CodeActionKind  
     = empty()
     | refactor(RefactorKind refactor = rewrite())
     | quickfix()
     | source(SourceActionKind source = organizeImports())
     ;
```


This is an _open_ data type. The constructor names are used
to compute the string values for the LSP by capitalizing and
joining parent/children with periods.

#### Examples


`refactor(rewrite())` becomes `refactor.rewrite` under the hood of the LSP.

## data SourceActionKind {#util-LanguageServer-SourceActionKind}
Used to prioritize menu options and choose relevant icons in the UI.

```rascal
data SourceActionKind  
     = organizeImports()
     | fixAll()
     ;
```


This is an open list and can be extended by the language engineer at will.
These names should be indicative of what will happen to the source code when the action is chosen.

#### Pitfalls


* You as language engineer are responsible for implementing the right action with these names.

## data RefactorKind {#util-LanguageServer-RefactorKind}
Used to prioritize menu options and choose relevant icons in the UI.

```rascal
data RefactorKind  
     = extract()
     | inline()
     | rewrite()
     ;
```


This is an open list and can be extended by the language engineer at will.
These names should be indicative of what will happen to the source code when the action is chosen.

#### Pitfalls


* You as language engineer are responsible for implementing the right action with these names.

## data InlayHint {#util-LanguageServer-InlayHint}
Represents one inlayHint for display in an editor

```rascal
data InlayHint  
     = hint(
        loc position,
        str label,
        InlayKind kind,
        str toolTip = "",
        bool atEnd = false
    )
     ;
```


* `position` where the hint should be placed, by default at the beginning of this location, the `atEnd` can be set to true to change this
* `label` text that should be printed in the ide, spaces in front and back of the text are trimmed and turned into subtle spacing to the content around it.
* `kind` his either `type()` or `parameter()` which influences styling in the editor.
* `toolTip` optionally show extra information when hovering over the inlayhint.
* `atEnd` instead of appearing at the beginning of the position, appear at the end.

## data InlayKind {#util-LanguageServer-InlayKind}
Style of an inlay

```rascal
data InlayKind  
     = \type()
     | parameter()
     ;
```

## function registerLanguage {#util-LanguageServer-registerLanguage}

Generates and instantiates a new language server for the given language

```rascal
void registerLanguage(Language lang)
```


We register languages by uploading the meta-data of the implementation to a "lanuage-parametric" language server.
1. The meta-data is used to instantiate a fresh run-time to execute the main-module.
2. The extension is registered with the IDE to link to this new runtime.
3. Each specific extension is mapped to a specific part of the language server protocol.

By registering a language twice, more things can happen:
* existing contributions are re-loaded and overwritten with the newest version.
* new contributions to an existing language (`Language` constructor instance), will be added to the existing LSP server instance. You can use this to load expensive features later or more lazily.
* errors appear during loading or first execution of the contribution. The specific contribution is then usually aborted and unregistered.

Because registerLanguage has effect in a different OS process, errors and warnings are not printed in the calling execution context.
In general look at the "Parametric Rascal Language Server" log tab in the IDE to see what is going on.

However since language contributions are just Rascal functions, it is advised to simply test them first right there in the terminal.
Use `util::Reflective::getProjectPathConfig` for a representative configuration.

## function unregisterLanguage {#util-LanguageServer-unregisterLanguage}

Spins down and removes a previously registered language server

```rascal
void unregisterLanguage(Language lang)
```

## function unregisterLanguage {#util-LanguageServer-unregisterLanguage}

Spins down and removes a previously registered language server

```rascal
void unregisterLanguage(str name, set[str] extensions, str mainModule = "", str mainFunction = "")
```

## function unregisterLanguage {#util-LanguageServer-unregisterLanguage}

Spins down and removes a previously registered language server

```rascal
void unregisterLanguage(str name, str extension, str mainModule = "", str mainFunction = "")
```

## function computeFocusList {#util-LanguageServer-computeFocusList}

Produce a ((Focus)) for a given tree and cursor position

```rascal
Focus computeFocusList(Tree input, int line, int column)
```


This function exists to be able to unit test [Language Service](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-LanguageService)s that
accept a [Focus](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-Focus) parameter, indepently of using [Register Language](../../../..//Packages/org.rascalmpl.rascal-lsp/Library/util/LanguageServer.md#util-LanguageServer-registerLanguage).

* `line` is a 1-based indication of what the current line is
* `column` is a 0-based indication of what the current column is.

#### Benefits


* test services without spinning up an LSP server or having to run UI tests.
Each UI interaction is tested generically for you already.

#### Pitfalls


* LSP indexing is different, but those differences are resolved in the implementation of the protocol. On the Rascal side, we see the above.
Differences are width of the character encoding for non-ASCII characters, and lines are 0-based, etc.

