@license{
Copyright (c) 2018-2025, NWO-I CWI and Swat.engineering
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
}
@bootstrapParser
module lang::rascal::lsp::refactor::Rename

/**
 * Rename refactoring
 *
 * Implements rename refactoring according to the LSP.
 * Renaming collects information generated by the typechecker for the module/workspace, finds all definitions and
 * uses matching the position of the cursor, and computes file changes needed to rename these to the user-input name.
 */

import Exception;
import IO;
import Grammar;
import List;
import Location;
import Map;
import ParseTree;
import Relation;
import Set;
import String;

import lang::rascal::\syntax::Rascal;

import lang::rascalcore::check::Checker;
import lang::rascalcore::check::BasicRascalConfig;

import lang::rascal::lsp::refactor::rename::Common;
import lang::rascal::lsp::refactor::rename::Constructors;
import lang::rascal::lsp::refactor::rename::Fields;
import lang::rascal::lsp::refactor::rename::Functions;
import lang::rascal::lsp::refactor::rename::Grammars;
import lang::rascal::lsp::refactor::rename::Modules;
import lang::rascal::lsp::refactor::rename::Parameters;
import lang::rascal::lsp::refactor::rename::Types;
import lang::rascal::lsp::refactor::rename::Variables;

extend analysis::typepal::refactor::Rename;
import util::Util;

import util::FileSystem;
import util::LanguageServer;
import util::Maybe;
import util::Reflective;

private bool isQualifiedUse(loc use, Define _:<_, str id, _, _, _, _>) = size(id) != use.length;

void rascalCheckCausesOverlappingDefinitions(set[Define] currentDefs, str newName, Tree tr, TModel tm, Renamer r) {
    defUse = invert(tm.useDef);
    unescNewName = forceUnescapeNames(newName);
    reachable = rascalGetReflexiveModulePaths(tm).to;
    usedModules = {d.top | loc d <- tm.useDef<1>};
    usedModels = (m: tm | loc m <- usedModules, TModel tm := r.getConfig().tmodelForLoc(m)) + (tr.src.top: tm);
    newNameDefs = {nD | TModel tm <- range(usedModels), Define nD:<_, unescNewName, _, _, _, _> <- tm.defines};
    curAndNewDefinitions = (d.defined: d | d <- currentDefs + newNameDefs); // temporary map for overloading checks
    maybeImplicitDefs = {n.names[-1].src | /QualifiedName n := tr};

    bool isImplicitDef(Define d)
        = (d.idRole is variableId && d.defined in tm.useDef<0>) // variable that's both a use and a def
        || (d.idRole is patternVariableId && d.defined in maybeImplicitDefs) // or pattern variable without a type
        ;

    for (<Define c, Define n> <- currentDefs * newNameDefs) {
        set[loc] curUses = defUse[c.defined];
        set[loc] newUses = defUse[n.defined];

        for (loc cU <- curUses
           , isContainedInScope(cU, n.scope, tm)) {
            // Will this rename hide a used definition of `oldName` behind an existing definition of `newName` (shadowing)?
            if (isContainedInScope(n.scope, c.scope, tm)
              , !isQualifiedUse(cU, c)) {
                r.msg(error(cU, "Renaming this to \'<newName>\' would change the program semantics; its original definition would be shadowed by <n.defined>."));
            }

            // Is `newName` already resolvable from a scope where `oldName` is currently declared?
            // Double declarations of module variables are only a problem if a use is ambiguous
            if ({moduleVariableId()} := {c.idRole, n.idRole}) {
                for (loc nU <- newUses, isContainedInScope(nU, c.scope, tm)
                  , n.defined.top == c.defined.top || !(isQualifiedUse(nU, n) && isQualifiedUse(cU, c))) {
                    r.msg(error(cU, "Renaming this to \'<newName>\' would cause a double declaration (with <n.defined>)."));
                }
            }
        }

        // Will this rename hide a used definition of `newName` behind a definition of `oldName` (shadowing)?
        for (isContainedInScope(c.scope, n.scope, tm)
           , loc nU <- newUses
           , isContainedInScope(nU, c.scope, tm)
           , !isQualifiedUse(nU, n)) {
            r.msg(error(c.defined, "Renaming this to \'<newName>\' would change the program semantics; it would shadow the declaration of <nU>."));
        }

        // Is `newName` already resolvable from a scope where `oldName` is currently declared?
        if (rascalMayOverload({c.defined, n.defined}, curAndNewDefinitions)) {
            // Overloading
            if (c.scope in reachable || isContainedInScope(c.defined, n.scope, tm) || isContainedInScope(n.defined, c.scope, tm)) {
                r.msg(error(c.defined, "Renaming this to \'<newName>\' would overload an existing definition at <n.defined>."));
            }
        } else if (isContainedInScope(c.defined, n.scope, tm) && {moduleVariableId(), _} !:= {c.idRole, n.idRole}) {
            // Double declaration
            r.msg(error(c.defined, "Renaming this to \'<newName>\' would cause a double declaration (with <n.defined>)."));
        }

        // Will this rename turn an implicit declaration of `newName` into a use of a current declaration?
        if (isImplicitDef(n) && isContainedInScope(n.defined, c.scope, tm)) {
            r.msg(error(c.defined, "Renaming this declaration to \'<newName>\' would change the program semantics; this implicit declaration would become a use: <n.defined>."));
        }
    }
}

void rascalCheckLegalNameByRole(Define _:<_, _, _, role, at, dt>, str name, Renamer r) {
    escName = normalizeEscaping(name);
    <t, desc> = asType(role, dt);
    if (tryParseAs(t, escName) is nothing) {
        r.msg(error(at, "<escName> is not a valid <desc>"));
    }
}

void rascalCheckDefinitionOutsideWorkspace(Define d, Renamer r) {
    f = d.defined.top;
    pcfg = r.getConfig().getPathConfig(f);
    if (!any(srcFolder <- pcfg.srcs, isPrefixOf(srcFolder, f))) {
        r.msg(error(d.defined, "Since this definition is not in the sources of open projects, it cannot be renamed."));
    }
}

@synopsis{Rename the Rascal symbol under the cursor. Renames all related (overloaded) definitions and uses of those definitions.}
@description {
    Rename the Rascal symbol under the cursor, across all currently open projects in the workspace.

    The following symbols are currently unsupported.
    - Annotations (on functions)

    *Name resolution*
    A renaming triggers the typechecker on the currently open file to determine the scope of the renaming.
    If the renaming is not function-local, it might trigger the type checker on all files in the workspace to find rename candidates.
    A renaming requires all files in which the name is used to be without errors.

    *Overloading*
    Considers recognizes overloaded definitions and renames those as well.

    Functions are considered overloaded when they have the same name, even when the arity or type signature differ.
    This means that the following functions defitions will be renamed in unison:
    ```
    list[&T] concat(list[&T] _, list[&T] _) = _;
    set[&T] concat(set[&T] _, set[&T] _) = _;
    set[&T] concat(set[&T] _, set[&T] _, set[&T] _) = _;
    ```

    ADT and grammar definitions are considered overloaded when they have the same name and type, and
    there is a common use from which they are reachable.
    As an example, modules `A` and `B` have a definition for ADT `D`:
    ```
    module A
    data D = a();
    ```
    ```
    module B
    data D = b();
    ```
    With no other modules in the workspace, renaming `D` in one of those modules, will not rename `D` in
    the other module, as they are not considered an overloaded definition. However, if a third module `C`
    exists, that imports both and uses the definition, the definitions will be considered overloaded, and
    renaming `D` from either module `A`, `B` or `C` will result in renaming all occurrences.
    ```
    module C
    import A;
    import B;
    D f() = a();
    ```

    *Validity checking*
    Once all rename candidates have been resolved, validity of the renaming will be checked. A rename is valid iff
    1. It does not introduce parse errors.
    2. It does not introduce type errors.
    3. It does not change the semantics of the application.
    4. It does not change definitions outside of the current workspace.
}

alias Edits = tuple[list[DocumentEdit], set[Message]];

Tree findCursorInTree(Tree t, loc cursorLoc) {
    top-down visit (t) {
        case Name n: if (isContainedIn(n.src, cursorLoc)) return n;
        case Nonterminal n: if (isContainedIn(n.src, cursorLoc)) return n;
        case NonterminalLabel n: if (isContainedIn(n.src, cursorLoc)) return n;
    }
    return t;
}

@synopsis{Due to how the focus list is computed and the grammar for concrete syntax, we cannot easily find the exact name that the cursor is at.}
list[Tree] extendFocusWithConcreteSyntax([Concrete c, *tail], loc cursorLoc) = [findCursorInTree(c, cursorLoc), c, *tail];
default list[Tree] extendFocusWithConcreteSyntax(list[Tree] cursor, loc _) = cursor;

data AugmentComponents = augmentUses() | augmentDefs();

bool requiresAugmentation(set[Define] defs, {augmentUses(), *_}) = (defs.idRole & {constructorId(), functionId(), fieldId(), keywordFieldId(), keywordFormalId(), typeVarId()}) != {};
bool requiresAugmentation(set[Define] defs, {augmentDefs()}) = (defs.idRole & {fieldId(), typeVarId()}) != {};

TModel getConditionallyAugmentedTModel(loc l, set[Define] defs, set[AugmentComponents] useOrDef, Renamer r)
    = requiresAugmentation(defs, useOrDef)
    ? r.getConfig().augmentedTModelForLoc(l, r)
    : r.getConfig().tmodelForLoc(l);

public Edits rascalRenameSymbol(loc cursorLoc, list[Tree] cursor, str newName, set[loc] workspaceFolders, PathConfig(loc) getPathConfig) {
    ModuleStatus ms = moduleStatus({}, {}, (), [], (), [], {}, (), (), (), (), pathConfig(), tconfig());

    TModel tmodelForTree(Tree tr) = tmodelForLoc(tr.src.top);

    TModel tmodelForLoc(loc l) {
        pcfg = getPathConfig(l);
        mname = getModuleName(l, pcfg);

        ccfg = rascalCompilerConfig(pcfg);
        <found, tm, ms> = getTModelForModule(mname, ms);
        if (found) return tm;

        ms = rascalTModelForNames([mname], ccfg, dummy_compile1);

        <found, tm, ms> = getTModelForModule(mname, ms);
        if (!found) throw "No TModel for module \'<mname>\'";
        return tm;
    }

    @synopsis{
        Augment the TModel with 'missing' use/def information.
        Workaround until the typechecker generates this. https://github.com/usethesource/rascal/issues/2172
    }
    @memo{maximumSize(100), expireAfter(minutes=5)}
    TModel augmentTModel(loc l, Renamer r) {
        TModel getModel(loc f) = f.top == l.top ? tm : r.getConfig().tmodelForLoc(f);

        tm = r.getConfig().tmodelForLoc(l);
        try {
            tr = parseModuleWithSpaces(l);
            tm = augmentExceptProductions(tr, tm, getModel);
            tm = augmentFieldUses(tr, tm, getModel);
            tm = augmentFormalUses(tr, tm, getModel);
            tm = augmentTypeParams(tr, tm);
        } catch value e: {
            println("Suppressed error during TModel augmentation: <e>");
        }
        return tm;
    }

    return rename(
        extendFocusWithConcreteSyntax(cursor, cursorLoc)
      , newName
      , rconfig(
            Tree(loc l) { return parseModuleWithSpaces(l); }
          , tmodelForTree
          , tmodelForLoc = tmodelForLoc
            // Call functions from the config so we re-use its cache
          , augmentedTModelForLoc = TModel(loc l, Renamer r) { return augmentTModel(l, r); }
          , workspaceFolders = workspaceFolders
          , getPathConfig = getPathConfig
          , debug = false
        )
    );
}

public Edits rascalRenameModule(list[tuple[loc old, loc new]] renames, set[loc] workspaceFolders, PathConfig(loc) getPathConfig) =
    propagateModuleRenames(renames, workspaceFolders, getPathConfig);


private set[Define] tryGetCursorDefinitions(list[Tree] cursor, TModel(loc) getModel) {
    loc cursorLoc = cursor[0].src;
    TModel tm = getModel(cursorLoc.top);

    set[Define] cursorDefs = {};
    if ([*pre, Tree c, *_] := cursor) {
        if (tm.definitions[c.src]?) {
            // Cursor at definition
            cursorDefs = {tm.definitions[c.src]};
        } else if (defs: {_, *_} := tm.useDef[c.src]) {
            // Cursor at use
            cursorDefs = flatMapPerFile(defs, set[Define](loc f, set[loc] localDefs) {
                localTm = f.top == cursorLoc.top ? tm : getModel(f);
                return {localTm.definitions[d] | loc d <- localDefs, localTm.definitions[d]?};
            });
        }

        bool isDefNameInFocus(Tree name)
            = any(t <- [*pre, c], name.src == t.src) && forceUnescapeNames("<name>") in cursorDefs.id;

        // Check if the name of the found declaration(s) actually appears in the focus list.
        // If this is not the case, we went too far up.
        if (cursorDefs != {}) {
            visit (c) {
                case Name tr: if (isDefNameInFocus(tr)) return cursorDefs;
                case QualifiedName tr: {
                    if (isDefNameInFocus(tr)) return cursorDefs;
                }
                case Nonterminal tr: if (isDefNameInFocus(tr)) return cursorDefs;
                case NonterminalLabel tr: if (isDefNameInFocus(tr)) return cursorDefs;
            }
        }
        // Try next cursor candidate in focus list
        fail;
    }

    return {};
}

set[Define] getCursorDefinitions(list[Tree] cursor, Tree(loc) _, TModel(Tree) _, Renamer r) {
    if (isUnsupportedCursor(cursor, r)) return {};

    loc cursorLoc = cursor[0].src;
    TModel tm = r.getConfig().tmodelForLoc(cursorLoc.top);
    if (isUnsupportedCursor(cursor, tm, r)) return {};

    set[Define] cursorDefs = tryGetCursorDefinitions(cursor, r.getConfig().tmodelForLoc);
    if (cursorDefs == {}) {
        tm = r.getConfig().augmentedTModelForLoc(cursorLoc.top, r);
        if (isUnsupportedCursor(cursor, tm, r)) return {};
        cursorDefs = tryGetCursorDefinitions(cursor, TModel(loc l) { return r.getConfig().augmentedTModelForLoc(l, r); });
    }

    if ({} := cursorDefs) {
        r.msg(error(cursorLoc, "Could not find definition to rename."));
    } else if (isUnsupportedCursor(cursor, cursorDefs, tm, r)) {
        return {};
    }
    return cursorDefs;
}

tuple[set[loc], set[loc], set[loc]] findOccurrenceFiles(set[Define] defs, list[Tree] cursor, str newName, Tree(loc) getTree, Renamer r) {
    escNewName = normalizeEscaping(newName);
    for (<role, dt> <- defs<idRole, defInfo>) {
        hasError = false;
        <t, desc> = asType(role, dt);
        if (tryParseAs(t, escNewName) is nothing) {
            hasError = true;
            r.msg(error(cursor[0], "\'<escNewName>\' is not a valid <desc>"));
        }

        if (hasError) return <{}, {}, {}>;
    }

    return findOccurrenceFilesUnchecked(defs, cursor, escNewName, getTree, r);
}

void validateNewNameOccurrences(set[Define] cursorDefs, str newName, Tree tr, Renamer r) {
    tm = getConditionallyAugmentedTModel(tr.src.top, cursorDefs, {augmentDefs(), augmentUses()}, r);
    rascalCheckCausesOverlappingDefinitions(cursorDefs, newName, tr, tm, r);
}

default void renameDefinitionUnchecked(Define _, loc nameLoc, str newName, Renamer r) {
    r.textEdit(replace(nameLoc, newName));
}

void renameDefinition(Define d, loc nameLoc, str newName, TModel _, Renamer r) {
    rascalCheckLegalNameByRole(d, newName, r);
    rascalCheckDefinitionOutsideWorkspace(d, r);

    renameDefinitionUnchecked(d, nameLoc, normalizeEscaping(newName), r);
}

private loc nameSuffix(loc l, set[Define] defs, Renamer r) {
    if ({str id} := defs.id) {
        if (l.length == size(id)) return l;
        if ({moduleId()} := defs.idRole) return l;
        return trim(l, removePrefix = l.length - size(id));
    }

    r.msg(error(l, "Cannot perform rename - definitions for this use have multiple names."));
    return l;
}

void renameUses(set[Define] defs, str newName, TModel tm, Renamer r) {
    escName = normalizeEscaping(newName);
    tm = getConditionallyAugmentedTModel(getModuleScopes(tm)[tm.modelName].top, defs, {augmentUses()}, r);

    definitions = {<d.defined, d> | d <- defs};
    useDefs = toMap(tm.useDef o definitions);
    for (loc u <- useDefs) {
        if (set[Define] ds:{_, *_} := useDefs[u], u notin defs.defined) {
            r.textEdit(replace(nameSuffix(u, ds, r), escName));
        }
    }

    renameAdditionalUses(defs, escName, tm, r);
}
