---
title: "module analysis::m3::AST"
id: AST
slug: /Library/analysis/m3/AST
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

a symbolic representation for abstract syntax trees of programming languages.
#### Usage

```rascal
import analysis::m3::AST;
```

#### Dependencies
```rascal
import Message;
import Node;
import IO;
import Set;
import util::Monitor;
import analysis::m3::TypeSymbol;
```

#### Description


We provide a general set of data types for the syntactic constructs of programming languages: `Expression`, `Statement`, `Declaration` and `Type`.
Also, very common syntactic constructs are added to this, such as `if`, `while`, etc.

The idea is that parsers for different languages will map to common abstract syntax elements, when this can be done meaningfully.
If not, then these front-ends will extend the existing types with new constructor definitions, or even new kinds of types will
be added. The shared representation limits the element of surprise when working with different languages, and perhaps may
make some downstream analyses reusable.

The concept of a source location is important for abstract syntax trees. The annotation `src` will always point to value of type `loc`, pointing to the physical location of the construct in the source code.

The concept of _declaration_ is also relevant. A `decl` annotation points from a use of a concept to its definition, but always via an indirection (i.e. fully qualified name). The `decl` annotation is also of type `loc`, where each location is a fully qualified name of the definition that is used. 

Finally, the concept of a _type_ is relevant for ASTs. In particular an `Expression` may have a `typ` annotation, or a variable declaration, etc.

#### Benefits


* Symbolic abstract syntax trees can be analyzed and transformed easily using Rascal primitives such as patterns, comprehensions and visit.
* By re-using recognizable names for different programming languages, it's easier to switch between languages to analyze.
* Some algorithms made be reusable on different programming languages, but please be aware of the _pitfalls_.

#### Pitfalls


*  Even though different languages may map to the same syntactic construct, this does not mean that the semantics is the same. Downstream
metrics or other analysis tools should still take semantic differences between programming languages into account.


## data \AST {#analysis-m3-AST-\AST}
For metric purposes we can use a true AST declaration tree, a simple list of lines for generic metrics, or the reason why we do not have an AST.

```rascal
data \AST (loc file = |unknown:///|) 
     = declaration(Declaration declaration)
     | lines(list[str] contents)
     | noAST(Message msg)
     ;
```

## data Declaration {#analysis-m3-AST-Declaration}
Uniform name for everything that is declared in programming languages: variables, functions, classes, etc.

```rascal
data Declaration (
	loc src = |unknown:///|,
	loc decl = |unresolved:///|,
	TypeSymbol typ = unresolved()
)
```


Instances of the Declaration type represent the _syntax_ of declarations in programming languages.

| field name | description |
| ---------- | ----------- |
| `src`      | the exact source location of the declaration in a source file |
| `decl`     | the resolved fully qualified name of the artefact that is being declared here |
| `typ`      | a symbolic representation of the static type of the declared artefact here (not the syntax of the type) |

## data Statement {#analysis-m3-AST-Statement}
Uniform name for everything that is typically a _statement_ programming languages: assignment, loops, conditionals, jumps

```rascal
data Statement (
	loc src = |unknown:///|,
	loc decl = |unresolved:///| 
)
```


Instances of the Statement type represent the _syntax_ of statements in programming languages.

| field name | description |
| ---------- | ----------- |
| `src`      | the exact source location of the statement in a source file |
| `decl`     | if the statement directly represent a usage of a declared artefact, then this points to the fully qualified name of the used artifact.

## data Expression {#analysis-m3-AST-Expression}
Uniform name for everything that is an _expression_ in programming languages: arithmetic, comparisons, function invocations, ...

```rascal
data Expression (
	loc src = |unknown:///|,
	loc decl = |unresolved:///|, 
	TypeSymbol typ = \unresolved()
)
```


Instances of the Expression type represent the _syntax_ of expressions in programming languages.

| field name | description |
| ---------- | ----------- |
| `src`      | the exact source location of the expression in a source file |
| `decl`     | if this expression represents a usage, decl is the resolved fully qualified name of the artefact that is being used here |
| `typ`      | a symbolic representation of the static type of the _result_ of the expression |

## data Type {#analysis-m3-AST-Type}
Uniform name for everything that is an _type_ in programming languages syntax: `int`, `void`, `List<Expression>`

```rascal
data Type (
	loc src = |unknown:///|,
	loc decl = |unresolved:///|, 
	TypeSymbol typ = \unresolved()
)
```


Instances of the Type type represent the _syntax_ of types in programming languages.

| field name | description |
| ---------- | ----------- |
| `src`      | the exact source location of the expression in a source file |
| `decl`     | the fully qualified name of the type, if resolved and if well-defined |
| `typ`      | a symbolic representation of the static type that is the meaning of this type expression |

## data Modifier {#analysis-m3-AST-Modifier}
Uniform name for everything that is a _modifier_ in programming languages syntax: public, static, final, etc.

```rascal
data Modifier (
	loc src = |unknown:///|
)
```


Instances of the Modifer type represent the _syntax_ of modifiers in programming languages.

| field name | description |
| ---------- | ----------- |
| `src`      | the exact source location of the expression in a source file |

## function astNodeSpecification {#analysis-m3-AST-astNodeSpecification}

Test for the consistency characteristics of an M3 annotated abstract syntax tree

```rascal
bool astNodeSpecification(node n, str language = "java", bool checkNameResolution=false, bool checkSourceLocation=true)
```

## function astNodeSpecification {#analysis-m3-AST-astNodeSpecification}

Check the AST node specification on a (large) set of ASTs and monitor the progress.

```rascal
bool astNodeSpecification(set[node] toCheck, str language = "java", bool checkNameResolution=false, bool checkSourceLocation=true)
```

