---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://10f944ea-19b3-4a19-9b01-aeb4976eb4f7|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <3,9,|uuid://e62901f8-6bc2-4868-af48-d1154ff05677|>,
  <9,81,|uuid://3e829366-6046-43bb-a495-93d6fa5107b1|>,
  <7,49,|uuid://6ba9644e-713a-4ed2-913d-2c425bfc5cf9|>,
  <4,16,|uuid://e87977c1-9ddc-43cb-b202-cd42445cae80|>,
  <2,4,|uuid://e4fa7a72-f79f-420b-8e8a-d91bf18638f4|>,
  <5,25,|uuid://fcc6cc67-29f6-4b28-9fd7-5517f07072ae|>,
  <10,100,|uuid://a01f18d9-0317-4f6c-ab9f-ff96a98f70bf|>,
  <8,64,|uuid://79672f6d-8803-4be7-94ca-19f5b428d779|>,
  <6,36,|uuid://977c2d04-4eb1-4d24-9233-841d3a11b55c|>,
  <1,1,|uuid://fbb49139-3182-41c6-b920-a6bd18ea3981|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://4ae5e55d-144f-4b59-81c1-6cbd66c88dab|,
  <2,4>:|uuid://7d8a5c50-9baf-4a87-9685-0091ed03a93d|,
  <7,49>:|uuid://d111dd6c-6d55-4bda-9d39-213a3b29d5bb|,
  <9,81>:|uuid://f49cdc27-76d6-47a2-bccc-c0cebe8baf31|,
  <8,64>:|uuid://c769b7db-76c7-4f1d-8fee-b373fe756442|,
  <5,25>:|uuid://42da327e-1c6f-4e18-987e-e544e2fbc697|,
  <4,16>:|uuid://44becce6-8342-426b-aea2-9a2659c9ddf6|,
  <1,1>:|uuid://e372dcb3-0055-4e58-8f7d-1800eec12959|,
  <10,100>:|uuid://3474039a-8463-4e0c-ab97-687a6be325cd|,
  <3,9>:|uuid://d0f12cce-7abc-4c67-a213-fd3998f7e4db|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

