---
title: "module vis::Charts"
id: Charts
slug: /Library/vis/Charts
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

Simple data visualization using charts
#### Usage

```rascal
import vis::Charts;
```

#### Dependencies
```rascal
import lang::html::IO;
import lang::html::AST;
import Content;
import Set;
import List;
```

#### Description


This modules provides a simple API to create charts for Rascal
(numerical) data, based on [chart.js](https://chartjs.org/). 
This library mirrors chart.js' JSON-based configuration API more or less one-to-one
using data types of Rascal. Documentation about chart.js should be easy
to interpret.

This module is quite new and may undergo some tweaks in the coming time.

#### Examples



```rascal-shell 
rascal>import vis::Charts;
ok
rascal>import util::Math;
ok
rascal>scatterChart([<x-arbInt(20),x+arbInt(20)> | x <- [1..100]])
```
![image](/assets/Library/vis/Charts_screenshot_23.png)
```rascal-shell
rascal>scatterChart(["down", "up"], [<x,100-x+arbInt(20)> | x <- [1..100]], [<x,x+arbInt(20)> | x <- [1..100]])
```
![image](/assets/Library/vis/Charts_screenshot_24.png)
```rascal-shell
```


```rascal-shell -continue
rascal>barChart([<"<x>",x-arbInt(20)> | x <- [1..100]])
```
![image](/assets/Library/vis/Charts_screenshot_28.png)
```rascal-shell
rascal>barChart(["down", "up"], [<"<x>",100-x+arbInt(20)> | x <- [1..100]], [<"<x>",x+arbInt(20)> | x <- [1..100]])
```
![image](/assets/Library/vis/Charts_screenshot_29.png)
```rascal-shell
```


```rascal-shell -continue
rascal>lineChart([<"<x>",x+arbInt(20)> | x <- [1..100]])
```
![image](/assets/Library/vis/Charts_screenshot_33.png)
```rascal-shell
rascal>lineChart(["down", "up"],
|1 >>>>    [<"<x>",100-x+arbInt(20)> | x <- [1..100]], 
|2 >>>>    [<"<x>",x+arbInt(20)> | x <- [1..100]]
|3 >>>>)
```
![image](/assets/Library/vis/Charts_screenshot_37.png)
```rascal-shell
rascal>lineChart(["down", "up", "even"],
|1 >>>>    [<"<x>",100-x+arbInt(20)> | x <- [1..100]], 
|2 >>>>    [<"<x>",x+arbInt(20)> | x <- [1..100]], 
|3 >>>>    [<"<x>", 70-arbInt(20)> | x <- [1..100]]
|4 >>>>)
```
![image](/assets/Library/vis/Charts_screenshot_42.png)
```rascal-shell
```


```rascal-shell -continue
rascal>pieChart([<"<x>",x+arbInt(25)> | x <- [1..10]])
```
![image](/assets/Library/vis/Charts_screenshot_46.png)
```rascal-shell
```

#### Benefits


* Easy to use for basic charting.
* Uses [Chart Auto Colors](../..//Library/vis/Charts.md#vis-Charts-ChartAutoColors) extension for ease-of-use.
* Support for 8 [Chart Type](../..//Library/vis/Charts.md#vis-Charts-ChartType)s including multiple [Chart Data Set](../..//Library/vis/Charts.md#vis-Charts-ChartDataSet)s in one chart.
* This API is open to extension via adding common keyword parameters for supporting any extension to the basic chart.js configuration.
* You can roll your own HTML or Server based on the building blocks in this module to include and use extensions, or to combine different charts in the same view.

#### Pitfalls


* Where `num` is asked, still `rat` values are _not_ supported.
* All `real` values must stay within JVM's `double` datatype
* All `int` values must fit within JVM's `long` datatype


## function scatterChart {#vis-Charts-scatterChart}

A scatterplot from a binary numerical relation.

```rascal
Content scatterChart(lrel[num x,num y] v, str title="Scatterplot", ChartAutoColorMode colorMode=\dataset())

Content scatterChart(list[str] labels, lrel[num x,num y] values ..., str title="Scatterplots", ChartAutoColorMode colorMode=\dataset())

Content scatterChart(rel[num x ,num y] v, str title="Scatterplot", ChartAutoColorMode colorMode=\dataset())

Content scatterChart(list[str] labels, rel[num x,num y] values ..., str title="Scatterplots", ChartAutoColorMode colorMode=\dataset())
```

## function bubbleChart {#vis-Charts-bubbleChart}

A bubblechart from a binary numerical list relation.

```rascal
Content bubbleChart(lrel[num x,num y, num r] v, str title="Scatterplot", ChartAutoColorMode colorMode=\data())

Content bubbleChart(list[str] labels, lrel[num x,num y, num r] values ..., str title="Scatterplots", ChartAutoColorMode colorMode=\data())

Content bubbleChart(rel[num x,num y, num r] v, str title="Scatterplot", ChartAutoColorMode colorMode=\data())

Content bubbleChart(list[str] labels, rel[num x,num y, num r] values ..., str title="Scatterplots", ChartAutoColorMode colorMode=\data())
```

#### Pitfalls


* the radius is in raw pixels rather than scaled to the chart's axis

## function barChart {#vis-Charts-barChart}

A bar chart from labeled numbers

```rascal
Content barChart(rel[str label, num val] values, str title="Bar Chart", ChartAutoColorMode colorMode=\data())

Content barChart(lrel[str label, num val] values, str title="Bar Chart", ChartAutoColorMode colorMode=\data())

Content barChart(list[str] labels, rel[str label, num val] values..., str title="Bar Chart", ChartAutoColorMode colorMode=\dataset())

Content barChart(list[str] labels, lrel[str label, num val] values..., str title="Bar Chart", ChartAutoColorMode colorMode=\dataset())
```

## function lineChart {#vis-Charts-lineChart}

A line chart from labeled numbers

```rascal
Content lineChart(rel[str label, num val] values, str title="Line Chart", ChartAutoColorMode colorMode=\dataset())

Content lineChart(lrel[str label, num val] values, str title="Line Chart", ChartAutoColorMode colorMode=\dataset())

Content lineChart(list[str] labels, rel[str label, num val] values..., str title="Line Chart", ChartAutoColorMode colorMode=\dataset())

Content lineChart(list[str] labels, lrel[str label, num val] values..., str title="Line Chart", ChartAutoColorMode colorMode=\dataset())
```

## function polarAreaChart {#vis-Charts-polarAreaChart}

A polar area chart from labeled numbers

```rascal
Content polarAreaChart(rel[str label, num val] values, str title="Polar Area Chart", ChartAutoColorMode colorMode=\data())

Content polarAreaChart(lrel[str label, num val] values, str title="Polar Area Chart", ChartAutoColorMode colorMode=\data())

Content polarAreaChart(list[str] labels, rel[str label, num val] values..., str title="Polar Area Chart", ChartAutoColorMode colorMode=\dataset())

Content polarAreaChart(list[str] labels, lrel[str label, num val] values..., str title="Polar Area Chart", ChartAutoColorMode colorMode=\dataset())
```

## function radarChart {#vis-Charts-radarChart}

A radar chart from labeled numbers

```rascal
Content radarChart(rel[str label, num val] values, str title="Radar Chart", ChartAutoColorMode colorMode=\data())

Content radarChart(lrel[str label, num val] values, str title="Radar Chart", ChartAutoColorMode colorMode=\data())

Content radarChart(list[str] labels, rel[str label, num val] values..., str title="Radar Chart", ChartAutoColorMode colorMode=\dataset())

Content radarChart(list[str] labels, lrel[str label, num val] values..., str title="Radar Chart", ChartAutoColorMode colorMode=\dataset())
```

## function pieChart {#vis-Charts-pieChart}

A pie chart from labeled numbers

```rascal
Content pieChart(rel[str label, num val] values, str title="Pie Chart", ChartAutoColorMode colorMode=\data())

Content pieChart(lrel[str label, num val] values, str title="Pie Chart", ChartAutoColorMode colorMode=\data())

Content pieChart(list[str] labels, rel[str label, num val] values..., str title="Pie Chart", ChartAutoColorMode colorMode=\dataset())

Content pieChart(list[str] labels, lrel[str label, num val] values..., str title="Pie Chart", ChartAutoColorMode colorMode=\dataset())
```

## function doughnutChart {#vis-Charts-doughnutChart}

A dougnut chart from labeled numbers

```rascal
Content doughnutChart(rel[str label, num val] values, str title="Doughnut Chart", ChartAutoColorMode colorMode=\data())

Content doughnutChart(lrel[str label, num val] values, str title="Doughnut Chart", ChartAutoColorMode colorMode=\data())

Content doughnutChart(list[str] labels, rel[str label, num val] values..., str title="Doughnut Chart", ChartAutoColorMode colorMode=\data())

Content doughnutChart(list[str] labels, lrel[str label, num val] values..., str title="Doughnut Chart", ChartAutoColorMode colorMode=\data())
```

## function chartDataSet {#vis-Charts-chartDataSet}

```rascal
ChartDataSet chartDataSet(str label, rel[num x, num y] r)

ChartDataSet chartDataSet(str label, map[num x, num y] r)

ChartDataSet chartDataSet(str label, rel[num x, num y, num rad] r)

ChartDataSet chartDataSet(str label, lrel[num x,num y] r)

ChartDataSet chartDataSet(str label, lrel[num x, num y, num r] r)
```

## function chartData {#vis-Charts-chartData}

```rascal
ChartData chartData(rel[str label, num val] v)

ChartData chartData(map[str label, num val] v)

ChartData chartData(lrel[str label, num val] v)

ChartData chartData(list[str] labels, lrel[num x, num y] values...)

ChartData chartData(list[str] labels, lrel[num x , num y , num r] values...)

ChartData chartData(list[str] labels, rel[num x, num y] values...)

ChartData chartData(list[str] labels, rel[num x, num y, num r] values...)

ChartData chartData(list[str] setLabels, lrel[str label, num val] values...)

ChartData chartData(list[str] setLabels, rel[str label, num val] values...)

ChartData chartData(list[str] labels, list[num] values...)

ChartData chartData(str label, lrel[num x, num y] values)

ChartData chartData(str label, map[num x, num y] values)

ChartData chartData(str label, lrel[num x, num y, num r] values)

ChartData chartData(str label, rel[num x, num y] values)

ChartData chartData(str label, rel[num x, num y, num r] values)
```

## data Chart {#vis-Charts-Chart}
Toplevel chart structure

```rascal
data Chart  
     = chart(
        ChartType \type = scatter(),
        ChartOptions options = chartOptions(),
        ChartData \data = chartData()
    )
     ;
```

## data ChartData {#vis-Charts-ChartData}
Wrapper for a set of datasets, each with a label

```rascal
data ChartData  
     = chartData(
        list[str]  labels=[],
        list[ChartDataSet] datasets = []
    )
     ;
```

## data ChartDataSet {#vis-Charts-ChartDataSet}
A dataset is a list of values to chart, with styling properties.

```rascal
data ChartDataSet (
        str label="",
        list[str] backgroundColor=[],
        list[str] borderColor=[],
        list[str] color=[]
    ) 
     = chartDataSet(list[value] \data)
     ;
```


The `data` field is a list of supported values, of which the constraints
are not expressible by data types. These are currently supported:

* [Chart Data Point](../..//Library/vis/Charts.md#vis-Charts-ChartDataPoint), with an without a `r`adius
* `num`, but within `double` precision (!) and no `rat`

## data ChartDataPoint {#vis-Charts-ChartDataPoint}
A data point is one of the types of values in a ChartDataSet

```rascal
data ChartDataPoint  
     = point(num x, num y, num r = 0)
     ;
```

## data ChartType {#vis-Charts-ChartType}

```rascal
data ChartType  
     = scatter()
     | bar()
     | bubble()
     | line()
     | polarArea()
     | radar()
     | pie()
     | doughnut()
     ;
```

## data ChartOptions {#vis-Charts-ChartOptions}

```rascal
data ChartOptions  
     = chartOptions(
        bool responsive=true,
        bool animations=true,
        ChartPlugins plugins = chartPlugins()  
    )
     ;
```

## data ChartPlugins {#vis-Charts-ChartPlugins}

```rascal
data ChartPlugins  
     = chartPlugins(
        ChartTitle title = chartTitle(),
        ChartLegend legend = chartLegend(),
        ChartColors colors = chartColors(),
        ChartAutoColors autocolors = chartAutoColors()
    )
     ;
```

## data ChartAutoColors {#vis-Charts-ChartAutoColors}

```rascal
data ChartAutoColors  
     = chartAutoColors(
        ChartAutoColorMode \mode = \data()
    )
     ;
```

## data ChartAutoColorMode {#vis-Charts-ChartAutoColorMode}

```rascal
data ChartAutoColorMode  
     = \data()
     | \dataset()
     ;
```

## data ChartLegend {#vis-Charts-ChartLegend}

```rascal
data ChartLegend  
     = chartLegend(
        LegendPosition position = top(),
        bool display=true
    )
     ;
```

## data LegendPosition {#vis-Charts-LegendPosition}

```rascal
data LegendPosition  
     = \top()
     | \bottom()
     | \left()
     | \right()
     ;
```

## data ChartTitle {#vis-Charts-ChartTitle}

```rascal
data ChartTitle  
     = chartTitle(
        str text="",
        bool display = true
    )
     ;
```

## data ChartColors {#vis-Charts-ChartColors}

```rascal
data ChartColors  
     = chartColors(
        bool enabled = true
    )
     ;
```

## function chartServer {#vis-Charts-chartServer}

Utility function that constructs a Chart from ChartData and a given Chart type and a title.

```rascal
Response(Request) chartServer(ChartData theData, ChartType \type=\bar(), str title="Chart", ChartAutoColorMode colorMode=\data(), bool legend=true, bool animations=false)
```


A chart has a typical default layout that we can reuse for all kinds of chart types. This function
provides the template and immediately instantiates the client and the server to start displaying the chart
in a browser.

## function chartServer {#vis-Charts-chartServer}

this is the main server generator for any chart value

```rascal
Response (Request) chartServer(Chart ch)
```


Given a Chart value this server captures the value and serves it
as a JSON value to the HTML client generated by [plotHTML](../..//Library/vis/Charts.md#vis-Charts-plotHTML).

## function plotHTML {#vis-Charts-plotHTML}

default HTML wrapper for a chart

```rascal
HTMLElement plotHTML()
```

