@license{
Copyright (c) 2018-2025, NWO-I CWI, Swat.engineering and Paul Klint
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
}
@synopsis{Provides access to the JDK javac compiler, to compile (generated) Java code.}
@benefits{
* Compile any Java code, stored anywhere reachable through a `loc` and using any library
reachable through a `loc`, to JVM bytecode using the standard JDK compiler.
}
@pitfalls{
* If you are looking for Java analysis and transformation support in Rascal please go find
the [java-air](http://www.rascalmpl.org/docs/Packages/RascalAir) package. The current module
only provides a Java to bytecode compilation API. 
}
module lang::java::Compiler

extend Message;

import IO;
import Location;
import String;
import util::FileSystem;
import util::PathConfig;
import util::Reflective;
import util::UUID;

@synopsis{Compile all the .java files in source folders in PathConfig}
@description{
See ((compileJavaSourceFolders)) for more information.
}
list[Message] compileJava(PathConfig pcfg) 
    = compileJavaSourceFolders(pcfg.srcs, pcfg.bin, libs=pcfg.libs);

@synopsis{Compile a single Java source file}
@pitfalls{
* `file` has to be reachable from `srcs`, because a fully qualified class name is computing by relativizing the source file `loc` against the `srcs` folders.
* The source files is read using ((readFile)).
* All classfiles, which could be many in the case of anonymous or nested classes, are written directly to the `bin` folder.
* `libs` is typically a list of `jar+file:///` or `mvn:///` locations, one for each (transitive) compile-time dependency.
Without these the compiler will complain about missing symbols and return error messages.
}
list[Message] compileJavaSourceFile(loc file, loc bin, list[loc] srcs, list[loc] libs=[]) 
    = compileJava({<file, qualifiedName(file, srcs), readFile(file)>}, bin, libs=libs);

@synopsis{Compile all java source files from a list of source folders.}
@description{
* Qualified class names are obtained by relativizing against the source folders.
* Source code is read in using ((readFile)).
* Bytecode files are written directly into the `bin` folder
}
list[Message] compileJavaSourceFolders(list[loc] srcFolders, loc bin, list[loc] libs=[])
    = compileJava({<f, qualifiedName(f, srcFolders), readFile(f)> | src <- srcFolders, f <- find(src, "java")}, bin, libs=libs);

@synopsis{Call the Java compiler on a list of files, reading in their contents first and computing their qualified names}
list[Message] compileJava(list[loc] sources, loc bin, list[loc] srcs=[], list[loc] libs=[])
    = compileJava({<s, qualifiedName(s, srcs), readFile(s)> | s <- sources}, bin, libs=libs);

@synopsis{Main workhorse for compiling Java source code}
@benefits{
* Use in memory cache to optimize compiling a lot of files at once
* Can work with any kind of source loc or target loc
}
@pitfalls{
* The sources relation must match the right file name to the right source code, for accurate error messages.
* The sources relation must give the right qualified name to the right source code, otherwise the compilation fails.
* While `compileJava` is running all source code and all generated bytecode will be in memory; this can be significant.
if you need to use less memory then call `compileJava` several times with a smaller relation.
}
@javaClass{org.rascalmpl.library.lang.java.JavaCompilerForRascal}
java list[Message] compileJava(rel[loc fileName, str qualifiedName, str sourceCode] sources, loc bin, list[loc] libs=[]);

@synopsis{computes the `packageName.className` fully qualified class name from a source filename `/.../src/packageName/className.java`} 
private str qualifiedName(loc path, list[loc] srcs)
    = replaceAll(relativize(srcs, path)[extension=""].path[1..], "/", ".") 
    when path.extension == "java";


@synopsis{tests folder structure of input .java files against the folder structure of the output .class files}
test bool compilerInputOutputFileTest() {
    root = uuid()[scheme="memory"];
    target = root + "target";

    writeFile(root + "A.java",           "class A { }");
    writeFile(root + "B.java",           "class B extends A { }");
    writeFile(root + "pack/C.java",      "package pack; \npublic class C { }");
    writeFile(root + "pack/pack/D.java", "package pack.pack; \nclass D extends pack.C { }");

    errors = compileJavaSourceFolders([root], target);
    assert errors == [] : "unexpected errors: <errors>";

    return find(target, "class") 
        ==  {
                target + "pack/C.class",
                target + "pack/pack/D.class",
                target + "B.class",
                target + "A.class"
        };
}

@synopsis{tests Java compilation with required libraries on the classpath}
test bool compilerClasspathTest() {
    root = uuid()[scheme="memory"];
    target = root + "target";
    
    writeFile(root + "A.java", "class A { org.rascalmpl.uri.URIResolverRegistry reg = org.rascalmpl.uri.URIResolverRegistry.getInstance(); }");
    
    rascalLib = resolvedCurrentRascalJar();

    errors = compileJavaSourceFolders([root], target, libs=[rascalLib]);

    assert errors == [] : "no errors expected: <errors>";

    return find(target,"class") ==  {target + "A.class"};
}