---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://1897c564-5e79-4481-b7c3-d13d43262984|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <9,81,|uuid://10a80f8a-4595-4260-a459-b29b12747748|>,
  <5,25,|uuid://8d5da109-e433-40d1-8cf9-c76250daadda|>,
  <8,64,|uuid://7293cda6-efda-406a-b1e9-069fde50bd13|>,
  <6,36,|uuid://8c1641ab-1c92-4339-8332-62e8a300906a|>,
  <2,4,|uuid://a9eccdb4-08bb-4bd9-8f94-1aab7f2509ab|>,
  <4,16,|uuid://6507f87b-da5e-4127-b840-9d30d97d8477|>,
  <10,100,|uuid://9f130a4c-d341-4fe2-a811-f048c3f84bb7|>,
  <7,49,|uuid://b72420e3-4b2e-4a0b-8659-1585169cdb39|>,
  <1,1,|uuid://3ff862b0-8e88-4d88-bae8-f3dbcdf125e9|>,
  <3,9,|uuid://ac5b9463-8c91-41d8-af1b-b6f4505ab94f|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://2607a978-c3a0-4aa0-8b00-805225b66456|,
  <2,4>:|uuid://583b7352-f582-4b93-8049-da5657c33fe3|,
  <7,49>:|uuid://55dcf1ad-3d8d-4f3d-bdaf-e218ad1b3ec8|,
  <9,81>:|uuid://295a8f63-f450-4108-ba44-3e1b7e01b2a2|,
  <8,64>:|uuid://0479065c-c068-40ca-aa06-9090d239ea62|,
  <5,25>:|uuid://8c0c9cf8-1b4c-4b6c-80a4-57187130416c|,
  <4,16>:|uuid://81d2600d-1d60-4b98-af09-8f5ba5c81a3f|,
  <1,1>:|uuid://403ba115-cd95-4c64-85b9-0a74ade233ba|,
  <10,100>:|uuid://d7905bc9-37b8-4a4e-84bc-87f8908d9c4e|,
  <3,9>:|uuid://e1a43a1a-b469-42ec-9f28-a41c800a5f30|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

