---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://18f04261-a978-4dbb-abc1-fb8d1b963e2e|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <2,4,|uuid://b666a45d-52e3-40db-9c20-a4c650e057a8|>,
  <8,64,|uuid://2cf5f8e1-a97a-480a-8e15-48aa7ade756c|>,
  <6,36,|uuid://1de1dbfd-cb28-400b-a1cd-cccb8427ebc1|>,
  <7,49,|uuid://d4671059-6b58-4424-8eb6-18572a509d15|>,
  <3,9,|uuid://f611f696-7f31-4b30-a71c-75b51611d4f8|>,
  <9,81,|uuid://d46d4ae0-c64a-4974-a01b-8aff364a9a50|>,
  <5,25,|uuid://64d6ca7b-d979-4a4c-8747-ef8d00e5646a|>,
  <10,100,|uuid://26675f10-05b3-4140-89a1-83d28b8b302e|>,
  <1,1,|uuid://74449d36-7e8e-4d8e-ab3b-3ad77a36aba3|>,
  <4,16,|uuid://b62792f0-d397-44bf-bb38-cb0a22d62f2b|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://f7b24ec0-59f1-4d10-8f55-d8d06919c99d|,
  <2,4>:|uuid://c95f5175-8efc-4d8d-a5a7-d95d1805a428|,
  <7,49>:|uuid://3ae35866-7c3b-47d9-8a1c-05a81344be74|,
  <9,81>:|uuid://747ac638-f7a4-4799-babd-61287ebd62bf|,
  <8,64>:|uuid://de17942f-b68d-48ab-9d47-84a46d81fba4|,
  <5,25>:|uuid://b32f88d3-aeb8-4446-8a18-0051b9cb846e|,
  <4,16>:|uuid://768a4cd7-3852-4d07-aa3a-9d3ad8deff7f|,
  <1,1>:|uuid://237de862-5415-4347-85ff-d6a90785e0c9|,
  <10,100>:|uuid://9d2ea6ff-0a07-455a-b546-36760f016a5e|,
  <3,9>:|uuid://e6342787-4598-4959-a243-5a6341417b33|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

