@license{
Copyright (c) 2018-2025, NWO-I CWI, Swat.engineering and Paul Klint
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
}
@bootstrapParser
module lang::rascalcore::compile::Rascal2muRascal::RascalPattern
   
import IO;
import ValueIO;
import Location;
import Node;
import Map;
import Set;
import String;
import ParseTree;
import util::Math;

import lang::rascal::\syntax::Rascal;
import lang::rascalcore::compile::muRascal::AST;

import lang::rascalcore::check::AType;
import lang::rascalcore::check::ATypeUtils;
import lang::rascalcore::check::BacktrackFree;
import lang::rascalcore::check::NameUtils;
import lang::rascalcore::compile::util::Names;

import lang::rascalcore::compile::Rascal2muRascal::Common;
import lang::rascalcore::compile::Rascal2muRascal::ModuleInfo;
import lang::rascalcore::compile::Rascal2muRascal::RascalType;
import lang::rascalcore::compile::Rascal2muRascal::TmpAndLabel;
import lang::rascalcore::compile::Rascal2muRascal::TypeUtils;

import lang::rascalcore::compile::Rascal2muRascal::RascalExpression;
/*
 * Compile the match operator and all possible patterns
 */

/*********************************************************************/
/*                  Match                                            */
/*********************************************************************/

default MuExp translateMatch(Pattern pat, Expression exp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont) {
    expType = getType(exp);
    expTrans = translate(exp);
    if(isVarOrTmp(expTrans)){
        return translatePat(pat, expType, expTrans, btscopes, trueCont, falseCont);
    } else {
        str fuid = topFunctionScope();
        subject_val = muTmpIValue(nextTmp("subject_val"), fuid, getType(exp));
        res = muValueBlock(abool(), [ muConInit(subject_val, expTrans), 
                                      translatePat(pat, expType, subject_val, btscopes, trueCont, falseCont) 
                                    ]);
        //iprintln(res);
        return res;
    }
}

/*********************************************************************/
/*                  Get Backtracking Scopes for a Pattern            */
/*********************************************************************/

alias BTSCOPE = tuple[str enter, str resume, str \fail];  // The enter/resume/fail labels of one backtracking scope
alias BTSCOPES = map[loc,BTSCOPE];                        // Map from program fragments to backtracking scopes
alias BTINFO = tuple[BTSCOPE btscope, BTSCOPES btscopes]; // Complete backtracking information: current bt scope + bt scope mapping

// Getters on backtracking scopes

str getEnter(BTSCOPE btscope) = btscope.enter;
str getResume(BTSCOPE btscope) = btscope.resume;
str getFail(BTSCOPE btscope) = btscope.\fail;

str getEnter(Tree t, BTSCOPES btscopes) = btscopes[getLoc(t)].enter;
str getEnter(loc l, BTSCOPES btscopes)  = btscopes[l].enter;

//str getDirectParentOf(str path, str begin){
//    parts = split("_", path);
//    if(size(parts) == 0) throw "Cannot find parent of <path> that begins with <begin>";
//    for(i <- reverse(index(parts))){
//        if(startsWith(parts[i], begin)){
//            return intercalate("_", parts[..i]);
//        }
//    }
//    println("Cannot find parent of <path> that begins with <begin>, returning <path>");
//    return path;
//}

str getParent(str path, str parent) {
    int i = findLast(path, parent);
    if(i < 0) throw "Cannot find root <parent> in path <path>";
    k = i + size(parent);
    digits = false;
    size_path = size(path);
    if(k < size_path && path[k] == "_"){ // take numeric suffices generated by nextTmp into account (e.g., _0, _23); but be aware of other suffixes, e.g. _GEN
        k += 1;
        digits = false;
        while(k < size_path && path[k] in {"0", "1", "2", "3", "4", "5", "6", "7", "8", "9"}){
            digits = true;
            k += 1;
        }
        return path[ .. digits ? k : k - 1];
    }
    return path[ .. k];
}
str getEnter(Tree t, str parent, BTSCOPES btscopes) = getParent(btscopes[getLoc(t)].enter, parent);
str getEnter(loc l, str parent, BTSCOPES btscopes)  = getParent(btscopes[l].enter, parent);

str getResume(Tree t, BTSCOPES btscopes) {

    try {
        //println("getResume: <getLoc(t)>");
        //println("btscopes[getLoc(t)]: < btscopes[getLoc(t)]>");
        return btscopes[getLoc(t)].resume;
    } catch _: {
        //println("getResume: <t>, <getLoc(t)>");
        //iprintln(btscopes);
        return "";
    }    
}

str getResume(loc l, BTSCOPES btscopes)  = btscopes[l].resume;

str getResume(Tree t, str parent, BTSCOPES btscopes) = getParent(btscopes[getLoc(t)].resume, parent);
str getResume(loc l, str parent, BTSCOPES btscopes)  = getParent(btscopes[l].resume, parent);

str getFail(Tree t, BTSCOPES btscopes) = btscopes[getLoc(t)].\fail;
str getFail(loc l, BTSCOPES btscopes)  = btscopes[l].\fail;

str getFail(Tree t, str parent, BTSCOPES btscopes) = getParent(btscopes[getLoc(t)].\fail, parent);
str getFail(loc l, str parent, BTSCOPES btscopes)  = getParent(btscopes[l].\fail, parent);

BTINFO registerBTScope(Tree t, BTSCOPE btscope, BTSCOPES btscopes){
    l = getLoc(t);
    //if(btscopes[l]? && btscopes[l] != btscope){
    //    println("btscope: <l> is being redefined:
    //            '<btscopes[l]> =\> 
    //            '<btscope>");
    //}
    btscopes[getLoc(t)] = btscope;
    return <btscope, btscopes>;
}

BTSCOPES getBTScopes(Tree t, str enter) = getBTInfo(t, <enter, enter, enter>, ()).btscopes;
BTSCOPES getBTScopes(Tree t, str enter, BTSCOPES btscopes) = getBTInfo(t, <enter, enter, enter>, btscopes).btscopes;

BTINFO getBTInfo(Tree t, str enter, BTSCOPES btscopes) = getBTInfo(t, <enter, enter, enter>, btscopes);

default BTINFO getBTInfo(Tree t, BTSCOPE btscope, BTSCOPES btscopes)
    = registerBTScope(t, btscope, btscopes);

//default BTINFO getBTInfo(Expression e, BTSCOPE btscope, BTSCOPES btscopes) {
//    if(!btscopes[getLoc(e)]?){
//        btscopes[getLoc(e)] = btscope;
//    }
//    return <btscopes[getLoc(e)], btscopes>;
//}

//str getResume(BTSCOPES btscopes){
//    println("btscopes:"); iprintln(btscopes);
//    r2l_scopes = sort(domain(btscopes), beginsAfter);
//    println("r2l_scopes:"); iprintln(r2l_scopes);
//   
//    resume = btscopes[r2l_scopes[0]].resume;
//    for(l <- r2l_scopes){
//        btscope = btscopes[l];
//        println("<l>: <btscope>");
//        resume = btscope.resume;
//        if(btscope.enter !=  btscope.resume){
//            return btscope.resume;
//        }
//    }
//    return resume;
//}

str getResume(BTSCOPES btscopes){
    r2l_scopes = reverse(sort(domain(btscopes)));
    resume = btscopes[r2l_scopes[-1]].resume;
    for(l <- r2l_scopes){
        btscope = btscopes[l];
        resume = btscope.resume;
        if(btscope.enter !=  btscope.resume){
            return btscope.resume;
        }
    }
    return resume;
}

bool haveEntered(str enter, BTSCOPES btscopes){
    return enter in range(btscopes)<0>;
}

void pretty(BTSCOPES btscopes){
    sortedEntries = sort(domain(btscopes), 
            bool(loc a, loc b) { return a.offset < b.offset ||  a.offset == b.offset && a.length > b.length;});
        
    emax = size("ENTER");
    rmax = size("RESUME");
    fmax = size("FAIL");
    for(l <- sortedEntries){
        btscope = btscopes[l];
        if(size(btscope.enter) > emax) emax = size(btscope.enter);
        if(size(btscope.resume) > rmax) rmax = size(btscope.resume);
        if(size(btscope.\fail) > fmax) fmax = size(btscope.\fail);
    }
    emax += 1;
    rmax += 1;
    fmax += 1;
    println("<left("ENTER", emax)><left("RESUME", rmax)><left("FAIL", fmax)>");
    for(l <- sortedEntries){
        btscope = btscopes[l];
        println("<left(btscope.enter, emax)><left(btscope.resume, rmax)><left(btscope.\fail, fmax)>: <readFile(l)>");
    }
}

/*********************************************************************/
/*                  Patterns                                         */
/*********************************************************************/

MuExp translatePatInSignatureWithTypeParameters(Pattern p, AType subjectType, MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore = muBlock([])){
    res = translatePat(p, subjectType, subject, btscopes, trueCont, falseCont/*, subjectAssigned=subjectAssigned*/);
    if(!isEmpty(getTypeParameters(subjectType))){
        precond = muMatchAndBind(subject, subjectType);
        return muIfElse(precond, res, falseCont);
    }
    return res;
}

// ==== literal pattern =======================================================

default MuExp translatePat(p:(Pattern) `<Literal lit>`, AType subjectType, MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore = muBlock([])) {
    return translateLitPat(lit, subjectType, subject, btscopes, trueCont, falseCont);
}
MuExp translatePat(Literal lit, AType subjectType, MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp _restore = muBlock([]))
    = translateLitPat(lit, subjectType, subject, btscopes, trueCont, falseCont);

MuExp translateLitPat(Literal lit, AType _subjectType, MuExp subject, BTSCOPES _btscopes, MuExp trueCont, MuExp falseCont) 
  = muIfElse(muEqual(translate(lit), subject), trueCont, falseCont);

// ==== negate pattern =======================================================

MuExp translatePat(p:(Pattern) `-<Pattern pat>`, AType subjectType, MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore = muBlock([])) {
    if(pat is literal){
        if(Literal lit := pat.literal && (lit is integer || lit is \real || lit is \rational)){
            code = muPrim("negative", getType(pat), [getType(lit)], [translate(lit)], p@\loc);
            return muIfElse(muEqual(code, subject), trueCont, falseCont);
        }
    }

    return translatePat(pat, subjectType, subject, btscopes, falseCont, trueCont); // TODO: check this
}

// ==== regexp pattern ========================================================

MuExp translatePat(p:(Pattern) `<RegExpLiteral r>`, AType subjectType, MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore = muBlock([])) 
    = translateRegExpLiteral(r, subjectType, subject, btscopes, trueCont, falseCont);

/*
lexical RegExpLiteral
	= "/" RegExp* "/" RegExpModifier ;

lexical NamedRegExp
	= "\<" Name "\>" 
	| [\\] [/ \< \> \\] 
	| NamedBackslash 
	| ![/ \< \> \\] ;

lexical RegExpModifier
	= [d i m s]* ;

lexical RegExp
	= ![/ \< \> \\] 
	| "\<" Name "\>" 
	| [\\] [/ \< \> \\] 
	| "\<" Name ":" NamedRegExp* "\>" 
	| Backslash 
	// | @category="MetaVariable" [\<]  Expression expression [\>] TODO: find out why this production existed 
	;
lexical NamedBackslash
	= [\\] !>> [\< \> \\] ;
*/

map[str,str] regexpEscapes = (
"(" : "(?:",
")" : ")"
);

MuExp translateRegExpLiteral(re: (RegExpLiteral) `/<RegExp* _>/<RegExpModifier _>`, AType _subjectType, MuExp subject, BTSCOPES _btscopes, MuExp trueCont, MuExp falseCont) {
   str fuid = topFunctionScope();
   <buildRegExp,vars> = processRegExpLiteral(re);
   matcher = muTmpMatcher(nextTmp("matcher"), fuid);
   found = muTmpBool(nextTmp("found"), fuid);
   btscope = nextLabel("REGEXP");
   code = [ muConInit(matcher, muRegExpCompile(buildRegExp, subject)),
            muVarInit(found, muCon(true)),
            muWhileDo("", found,
                      muBlock([ muAssign(found, muRegExpFind(matcher)),
                                muIfElse(found,
                                    muBlock([ *[ muVarInit(vars[i], muRegExpGroup(matcher, i+1)) | i <- index(vars) ],
                                              *( inStringVisit() ? [ muRegExpSetMatchedInVisit(matcher) ]
                                                                 : [] ),
                                               trueCont
                                            ]),
                                    falseCont)
                              ]))
           
          ];
   return muValueBlock(abool(), code);
}

MuExp translateRegExpLiteral(re: (RegExpLiteral) `/<RegExp* _>/<RegExpModifier _>`, MuExp _begin, MuExp _end) {
    throw "translateRegExpLiteral: <re>";
// TODO
   //<buildRegExp,varrefs> = processRegExpLiteral(re);
   //return muApply(mkCallToLibFun("Library", "MATCH_REGEXP_IN_VISIT"), 
   //              [ buildRegExp,
   //                muCallMuPrim("make_array", varrefs),
   //                begin,
   //                end
   //              ]); 
}

tuple[MuExp exp, list[MuExp] vars] processRegExpLiteral(e: (RegExpLiteral) `/<RegExp* rexps>/<RegExpModifier modifier>`){
   str fuid = topFunctionScope();
   fragmentCode = [];
   vars = [];
   map[str,int] varnames = ();
   str fragment = "";
   modifierString = "<modifier>";
   for(i <- [0 .. size(modifierString)]){
      fragment += "(?<modifierString[i]>)";
   }
   lrexps = [r | r <- rexps];
   len = size(lrexps); // library!
   i = 0;
   while(i < len){
      r = lrexps[i];
      //println("lregex[<i>]: <r>\nfragment = <fragment>\nfragmentCode = <fragmentCode>");
      if("<r>" == "\\"){
         suffix = "";
         if(i < len  - 1){
            suffix = "<lrexps[i + 1]>";
         }
         fragment += "\\"  + suffix;
         // TODO: compiler could not handle cond exp in: fragment += "\\" + (i < len  - 1 ? "<lrexps[i + 1]>" : "");
         i += 2;
      } else 
      if(size("<r>") == 1){
         if("<r>" == "(" && i < (len  - 1) && "<lrexps[i + 1]>" == "?"){
           fragment += "(";
         } else {
           fragment += escape("<r>", regexpEscapes);
         }
         i += 1;
      } else {
        if(size(fragment) > 0){
            fragmentCode += muCon(fragment);
            fragment = "";
        }
        switch(r){
          case (RegExp) `\<<Name name>\>`: {
                nm = "<name>";
            	if(varnames[nm]?){
            	   fragment += "\\<varnames[nm]>";
            	} else {
            	  fragmentCode += [ muPrim("str_escape_for_regexp", astr(), [getType(name)], [ translate(name) ], r@\loc)];
            	}
        	}
          case (RegExp) `\<<Name name>:<NamedRegExp* _>\>`: {
         		<varref, fragmentCode1> = extractNamedRegExp(r);
         		fragmentCode += fragmentCode1;
         		vars += varref;
         		varnames["<name>"] = size(vars);
         	}
          default:
        	fragmentCode += [muCon("<r>")];
        }
        i += 1;
      }
   }
   
   if(size(fragment) > 0){
      fragmentCode += muCon(fragment);
   }
   if(all(MuExp frag <- fragmentCode, muCon(_) := frag)){
      buildRegExp = muCon(intercalate("", [s | muCon(str s) <- fragmentCode]));
      return <buildRegExp, vars>;
   } else {
       swriter = muTmpStrWriter("swriter", fuid);
       buildRegExp = muValueBlock(astr(),
                                  muConInit(swriter, muPrim("open_string_writer", astr(), [], [], e@\loc)) + 
                                  [ muPrim("add_string_writer", astr(), [getType(exp)], [swriter, exp], e@\loc) | exp <- fragmentCode ] +
                                  muPrim("close_string_writer", astr(), [astr()], [swriter], e@\loc));
       return  <buildRegExp, vars>; 
   }  
}

tuple[MuExp var, list[MuExp] exps] extractNamedRegExp((RegExp) `\<<Name name>:<NamedRegExp* namedregexps>\>`) {
   exps = [];
   str fragment = "(";
   atStart = true;
   for(nr <- namedregexps){
       elm = "<nr>";
       if(atStart && size(trim(elm)) == 0){
       	 continue;
       }
       atStart = false;
       if(size(elm) == 1){
         fragment += escape(elm, regexpEscapes);
       } else if(elm[0] == "\\"){
         fragment += elm[0..];
       } else if((NamedRegExp) `\<<Name name2>\>` := nr){
         //println("Name case: <name2>");
         if(fragment != ""){
            exps += muCon(fragment);
            fragment = "";
         }
         exps += translate(name2);
       }
   }
   exps += muCon(fragment + ")");
   <fuid, pos> = getVariableScope("<name>", name@\loc);
   return <muVar("<name>", fuid, pos, astr(), patternVariableId()), exps>;
}

// ==== concrete syntax pattern ===============================================

// ---- getBTInfo 

BTINFO getBTInfo(t:appl(prod(label("concrete",sort("Pattern")),[label("concrete",lex("Concrete"))], {}),[Tree concrete1]),  BTSCOPE btscope, BTSCOPES btscopes){
    if(appl(prod(Symbol::label("parsed",Symbol::lex("Concrete")), [_],_),[Tree _concrete2]) := concrete1){
        btinfo = getBTInfoConcrete(concrete1, btscope, btscopes);
        return registerBTScope(concrete1, btinfo.btscope, btinfo.btscopes);
    }
    throw "getBTInfo: cannot match <concrete1>";
}

//BTINFO getBTInfoConcrete(appl(prod(lit(str s),_,_), _), BTSCOPE btscope, BTSCOPES btscopes){
//    return <btscope, btscopes>;
//}
//
//BTINFO getBTInfoConcrete(appl(prod(cilit(str s),_,_), _), BTSCOPE btscope, BTSCOPES btscopes){
//    return <btscope, btscopes>;
//}
//
//BTINFO getBTInfoConcrete(appl(prod(layouts(_),_,_), _), BTSCOPE btscope, BTSCOPES btscopes){
//    return <btscope, btscopes>;
//}
//
BTINFO getBTInfoConcrete(char(int i), BTSCOPE btscope, BTSCOPES btscopes){
    return <btscope, btscopes>;
}

BTINFO getBTInfoConcrete(t:appl(p:Production::regular(s:\iter(Symbol elem)), list[Tree] args), BTSCOPE btscope, BTSCOPES btscopes) {
    enterList = btscope.enter;
    BTSCOPE btscopeLast = <enterList, btscope.resume, btscope.resume>;
    for(pat <- args){
        <btscopeLast, btscopes> = getBTInfoConcrete(pat, btscopeLast, btscopes);
    }
    return registerBTScope(t, <enterList, btscopeLast.resume, btscope.resume>, btscopes);
 }
BTINFO getBTInfoConcrete(t:appl(p:Production::regular(s:\iter-star(Symbol elem)), list[Tree] args), BTSCOPE btscope, BTSCOPES btscopes) {
    enterList = btscope.enter;
    BTSCOPE btscopeLast = <enterList, btscope.resume, btscope.resume>;
    for(pat <- args){
        <btscopeLast, btscopes> = getBTInfoConcrete(pat, btscopeLast, btscopes);
    }
    return registerBTScope(t, <enterList, btscopeLast.resume, btscope.resume>, btscopes);
} 
BTINFO getBTInfoConcrete(t:appl(p:Production::regular(s:\iter-seps(Symbol elem, list[Symbol] seps)), list[Tree] args), BTSCOPE btscope, BTSCOPES btscopes) {
    enterList = btscope.enter;
    BTSCOPE btscopeLast = <enterList, btscope.resume, btscope.resume>;
    for(pat <- args){
        <btscopeLast, btscopes> = getBTInfoConcrete(pat, btscopeLast, btscopes);
    }
    return registerBTScope(t, <enterList, btscopeLast.resume, btscope.resume>, btscopes);
}
BTINFO getBTInfoConcrete(t:appl(p:Production::regular(s:\iter-star-seps(Symbol elem, list[Symbol] seps)), list[Tree] args), BTSCOPE btscope, BTSCOPES btscopes) {
    enterList = btscope.enter;
    BTSCOPE btscopeLast = <enterList, btscope.resume, btscope.resume>;
    for(pat <- args){
        <btscopeLast, btscopes> = getBTInfoConcrete(pat, btscopeLast, btscopes);
    }
    return registerBTScope(t, <enterList, btscopeLast.resume, btscope.resume>, btscopes);
}

BTINFO getBTInfoConcrete(t:appl(Production prod, list[Tree] args),  BTSCOPE btscope, BTSCOPES btscopes){
    if(appl(prod(Symbol::label("$MetaHole", Symbol _),[Symbol::sort("ConcreteHole")], {\tag("holeType"(Symbol holeType))}), [ConcreteHole _]) := t){
        if(isIterSymbol(holeType)){
            enter1 = btscope.enter;
            resume1 = enter1;
            fail1 = btscope.resume;
            return registerBTScope(t, <enter1, resume1, fail1>, btscopes);
        }
        return getBTInfo(t, btscope, btscopes);
    }
    enterAppl = nextLabel("<btscope.enter>_APPL");
    BTSCOPE btscopeLast = <enterAppl, btscope.resume, btscope.resume>;
    
    for(int i <- index(args)){
      arg = args[i];
      <btscopeLast, btscopes> = getBTInfoConcrete(arg, btscopeLast, btscopes);
      btscopeLast.enter += "_<i>";
    }
    return registerBTScope(t, <enterAppl, btscopeLast.resume, btscope.resume>, btscopes);
}

default BTINFO getBTInfoConcrete(Tree p, BTSCOPE btscope, BTSCOPES btscopes) = <btscope, btscopes>;  

// ----

// Concrete pattern was parsed correctly
MuExp translateConcretePattern(e:appl(prod(Symbol::label("parsed",Symbol::lex("Concrete")), [_],_),[Tree concrete]), 
                  AType subjectType, MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([]))
{
    return translateParsedConcretePattern(concrete, getType(e), subjectType, subject, btscopes, trueCont, falseCont, restore=restore); 
}
  
// Concrete pattern was not parsed correctly  
MuExp translateConcretePattern(e:appl(prod(Symbol::label("typed",Symbol::lex("Concrete")), [_],_),[Tree concrete]), 
                   AType _, MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) 
  = muValueBlock(avalue(),[muThrow(muCon("(compile-time) parse error in concrete syntax"), e@\loc)]);   

// Parsed Concrete pattern

// ---- lit, cilit

MuExp translateParsedConcretePattern(appl(prod(Symbol::lit(str s),_,_), _), AType patType, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) 
  = muIfElse(muEqual(muTreeUnparse(subjectExp), muCon(s)), trueCont, falseCont);
  
MuExp translateParsedConcretePattern(appl(prod(Symbol::cilit(str s),_,_), _), AType patType, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) 
    = muIfElse(muEqual(muTreeUnparseToLowerCase(subjectExp), muCon(toLowerCase(s))), trueCont, falseCont);

// ---- layouts

MuExp translateParsedConcretePattern(appl(prod(Symbol::layouts(_),_,_), _), AType patType, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) 
{ return trueCont; } 

// ---- Pattern variable

MuExp translateParsedConcretePattern(appl(prod(Symbol::label("$MetaHole", Symbol _),[Symbol::sort("ConcreteHole")], {\tag("holeType"(Symbol holeType))}), [ConcreteHole hole]),
                        AType patType, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) {
   holeName = prettyPrintName(hole.name);
   return isWildCard(holeName) ? trueCont :  muBlock([muVarInit(mkVar(holeName, hole.name@\loc), subjectExp), trueCont]);
}

// ---- char

MuExp translateParsedConcretePattern(t:Tree::char(int i),
                        AType patType, AType subjectType, MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) 
 = muIfElse(muEqual(muCon(char(i)), subject), trueCont, falseCont);                            

// ---- concrete lists

private MuExp translateParsedConcretePattern(t:appl(p:Production::regular(s:Symbol::iter(Symbol elem)), list[Tree] args),       
                        AType patType, AType subjectType, MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) {
      return translateConcreteListPat(t, symbol2atype(s), subjectType, subject, btscopes, trueCont, falseCont, restore=restore, delta=1);
}

private MuExp translateParsedConcretePattern(t:appl(p:Production::regular(s:Symbol::\iter-star(Symbol elem)), list[Tree] args),
                        AType patType, AType subjectType, MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) {
    return translateConcreteListPat(t, symbol2atype(s), subjectType, subject, btscopes, trueCont, falseCont, restore=restore, delta=1);
}
   
private MuExp translateParsedConcretePattern(t:appl(p:Production::regular(s:Symbol::\iter-seps(Symbol elem, list[Symbol] seps)), list[Tree] args),
                        AType patType, AType subjectType, MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) { 

    return translateConcreteListPat(t, symbol2atype(s), subjectType, subject, btscopes, trueCont, falseCont, restore=restore, delta=1+size(seps));
}

private MuExp translateParsedConcretePattern(t:appl(p:Production::regular(s:Symbol::\iter-star-seps(Symbol elem, list[Symbol] seps)), list[Tree] args),
                       AType patType, AType subjectType,  MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) {
    return translateConcreteListPat(t, symbol2atype(s), subjectType, subject, btscopes, trueCont, falseCont, restore=restore, delta=1+size(seps));
}

AType getTypeTree(appl(Production prod, list[Tree] args)) {
    return  symbol2atype(prod.def);
}
AType getTypeTree(t:Tree::char(int n)) = treeType;

default AType getTypeTree(Tree t){
    throw t;
}
                                             
private MuExp translateParsedConcretePattern(t:appl(prod:Production::regular(Symbol::\seq(list[Symbol] symbols)), list[Tree] args),
                       AType patType, AType subjectType,  MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) {
   body = trueCont;
   str fuid = topFunctionScope();
   for (int i <- reverse(index(symbols))) {         
        subject_arg = muTmpIValue(nextTmp("subject_arg<i>"), fuid, getTypeTree(args[i]));
        body = muBlock([ muConInit(subject_arg, muSubscript(muTreeGetArgs(subject), alist(treeType), muCon(i))),
                            translateParsedConcretePattern(args[i], symbol2atype(symbols[i]), getType(subject_arg), subject_arg, btscopes, body, falseCont, restore=restore)
                       ]);
   }
   
   ///*syn*/if(label(str _, Symbol tp) := prod.def){     
   //     prod.def = tp; 
   //} 
   cond = muTreeIsProductionEqual(subject, muCon(prod));
   return muIfElse(cond, body, falseCont);         
}

// ---- opt 

private MuExp translateParsedConcretePattern(t:appl(prod:Production::regular(s:Symbol::opt(Symbol symbol)), list[Tree] args),
                       AType patType, AType subjectType,  MuExp subject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([])) {
   
    body = isEmpty(args) ? trueCont : translateParsedConcretePattern(args[0], symbol2atype(symbol), getType(subject), subject, btscopes, trueCont, falseCont, restore=restore);
    ///*syn*/if(label(str _, Symbol tp) := prod.def){     
    //    prod.def = tp; 
    //} 
    cond = muTreeIsProductionEqual(subject, muCon(prod));
    return muIfElse(cond, body, falseCont);             
}
                       
// ---- any parse tree

default MuExp translateParsedConcretePattern(t:appl(Production prod, list[Tree] args),
                       AType patType, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore = muBlock([]))  {
   if(isConcreteHole(t)){
      holeName = getConcreteHoleName(t);
      holeSrc = getConcreteHoleVarLoc(t);
      return isWildCard(holeName) ? trueCont :  muBlock([muVarInit(mkVar(holeName, holeSrc), subjectExp), trueCont]);
   }
   body = trueCont;
   str fuid = topFunctionScope();
   for (int i <- reverse(index(args))) {
       if(!isLayoutPat(args[i])){
           subject_arg = muTmpIValue(nextTmp("subject_arg<i>"), fuid, getTypeTree(args[i]));   
           body = muBlock([ muConInit(subject_arg, muSubscript(muTreeGetArgs(subjectExp), alist(treeType), muCon(i))),
                            translateParsedConcretePattern(args[i], symbol2atype(prod has symbols ? prod.symbols[i] : prod.def), getType(subject_arg), subject_arg, btscopes, body, falseCont, restore=restore)
                          ]);
       }
   }   
   ///*syn*/if(label(str _, Symbol tp) := prod.def){     
   //     prod.def = tp; 
   //}   
   cond = muTreeIsProductionEqual(subjectExp, muCon(prod));        
   return muIfElse(cond, body, falseCont);
}

bool isConcreteHole(t:appl(prod(Symbol::label("$MetaHole", Symbol _),[Symbol::sort("ConcreteHole")], {\tag("holeType"(Symbol holeType))}), [ConcreteHole hole])) = true;
default bool isConcreteHole(Tree t) = false;

Symbol getConcreteHoleSymbol(appl(Production::prod(Symbol::label("$MetaHole", Symbol _),[Symbol::sort("ConcreteHole")], {\tag("holeType"(Symbol holeType))}), [ConcreteHole _]))
    = (Symbol::\iter-seps(s, []) := holeType) ? Symbol::\iter(s)
                                              : ((Symbol::\iter-star-seps(s, []) := holeType) ? Symbol::\iter-star(s) : holeType);
    
loc getConcreteHoleVarLoc(h: appl(Production _prod, list[Tree] args)) {
	//println("getConcreteHoleVarLoc: <h>");
	if(args[0].args[4].args[0]@\loc?){
	    //iprintln(args[0].args[4].args[0]);
		return args[0].args[4].args[0]@\loc;
	}
	if(args[0].args[4]@\loc?){
		println("getConcreteHoleVarLoc: moved up one level to get loc: <h>");
		return args[0].args[4]@\loc;
	}
	println("getConcreteHoleVarLoc: Missing loc:");
	iprintln(h);
	println("hole: <args[0].args[4].args[0]>");
	iprintln(args[0].args[4].args[0]);
	//println("<h@\loc?>, <(args[0])@\loc?>,  <(args[0].args[4])@\loc?>, <(args[0].args[4].args[0])@\loc?>");
	throw "getConcreteHoleVarLoc: Missing loc";
}

str getConcreteHoleName(appl(Production _prod, list[Tree] args)){
    return "<args[0].args[4].args[0]>";
}	

// The patterns callOrTree and reifiedType are ambiguous, therefore we need special treatment here.

//MuExp translateParsedConcretePattern(amb(set[Tree] alts), AType symbol) {
//   throw "translateParsedConcretePattern: ambiguous, <alts>";
//}
//
//default MuExp translateParsedConcretePattern(Tree c, AType symbol) {
//   //iprintln(c);
//   throw "translateParsedConcretePattern: Cannot handle <c> at <c@\loc>";
//}

bool isLayoutPat(Tree pat) = appl(Production::prod(Symbol::layouts(_), _, _), _) := pat;

//bool isSeparator(Tree pat, AType sep) = appl(Production::prod(sep, _, _), _) := pat;


//// Is a symbol an iterator type?
//
//bool isIterSymbol(\iter(Symbol symbol)) = true;
//bool isIterSymbol(\iter-star(Symbol symbol)) = true;
//bool isIterSymbol(\iter-seps(Symbol symbol, list[Symbol] separators)) = true;
//bool isIterSymbol(\iter-star-seps(Symbol symbol, list[Symbol] separators)) = true;
//default bool isIterSymbol(Symbol s) = false;
//
//// Is a symbol an iterator type with separators?
//bool isIterSymbolWithSeparator(\iter-seps(Symbol symbol, list[Symbol] separators)) = true;
//bool isIterSymbolWithSeparator(\iter-star-seps(Symbol symbol, list[Symbol] separators)) = true;
//default bool isIterSymbolWithSeparator(Symbol s) = false;

// What is is the minimal iteration count of a symbol?
int nIter(\iter(Symbol symbol)) = 1;
int nIter(\iter-star(Symbol symbol)) = 0;
int nIter(\iter-seps(Symbol symbol, list[Symbol] separators)) = 1;
int nIter(\iter-star-seps(Symbol symbol, list[Symbol] separators)) = 0;
default int nIter(Symbol s) { throw "Cannot determine iteration count: <s>"; }

// What is is the minimal iteration count of a pattern (as Tree)?
int nIter(Tree pat){
  if(t:appl(Production _prod, list[Tree] _args) := pat && isConcreteHole(t)){
     //varloc = getConcreteHoleVarLoc(t);
     //<fuid, pos> = getVariableScope("ConcreteVar", varloc);
     holeType = getConcreteHoleSymbol(t);
     if(isIterSymbolWithSeparator(holeType)){
        return nIter(holeType);
     } 
  }
  return 1;
}

// Is an appl node a concrete multivar?

bool isConcreteMultiVar(t:appl(Production prod, list[Tree] args)){
  return isConcreteHole(t) && isIterSymbol(getConcreteHoleSymbol(t));
}

default bool isConcreteMultiVar(Tree t) = false;

// Compute a list of lookaheads for a  list of patterns.
// Recall that a Lookahead is a tuple of the form <number-of-elements-following, number-of-multi-vars-following>

list[Lookahead] computeConcreteLookahead(list[Tree] pats){
    nElem = 0;
    nMultiVar = 0;
    
    rprops = for(Tree p <- reverse([p | Tree p <- pats])){
                 append <nElem, nMultiVar>;
                 if(isConcreteMultiVar(p)) {nMultiVar += 1; nElem += nIter(p);} else {nElem += 1;}
             };
    //println("result = <reverse(rprops)>");
    return reverse(rprops);
}

tuple[bool eq, int size] computeListMinSize(list[Pattern] pats){
    erase = {};
    no_multivar_seen = true;
    for(int i <- index(pats)){
        if(isMultiVar(pats[i])) no_multivar_seen = false;
        if(nIter(pats[i]) == 0){
                erase += i;
        }
   }
   return <no_multivar_seen, size(toSet(index(pats)) - erase)>;
}

tuple[bool eq, int size] computeConcreteListMinSize(list[Tree] pats, int delta){
    erase = {};
    only_empty_seen = true;
    no_multivar_seen = true;
    for(int i <- index(pats)){
        if(isConcreteMultiVar(pats[i])) no_multivar_seen = false;
        if(nIter(pats[i]) == 0){ 
            if(delta == 1){
                erase += i;
            } else if(delta == 2){
                erase += i - 1; 
                erase += i;
                if(only_empty_seen) erase += i + 1; // layout after first potentially empty multivar(s)
            } else {                                // Don't worry about out of range indices; they will just not be erased ;-)
                erase += i - 3; 
                erase += i - 2; 
                erase += i - 1;
                erase += i;
                if(only_empty_seen){                // layout and separator after first potentially empty multivar(s)
                    erase += i + 1;
                    erase += i + 2;
                    erase += i + 3;
                }
            }
        } else {
            if(i % delta == 0) only_empty_seen = false;
        }
   }
   return <no_multivar_seen, size(toSet(index(pats)) - erase)>;
}

MuExp translateConcreteListPat(p:appl(Production _prod, list[Tree] lpats), AType patType, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([]), int delta=1) {
    lookahead = computeConcreteLookahead(lpats); 
   
    npats = size(lpats);
    elmType = getIterElementType(subjectType);
   
    try {
        constantPat = translateConcretePatternAsConstant(p);
        return muIfExp(muMatch(subjectExp, muCon(constantPat)), trueCont, falseCont);
    } catch _: /* not a constant pattern, go on computing its translation */;
    
    str fuid = topFunctionScope();
    subj = nextTmp("subject");
    subject = muTmpIValue(subj, fuid, subjectType);
    cursor = muTmpInt(subj + "_cursor", fuid);
    sublen = muTmpInt(subj + "_len", fuid);
    typecheckNeeded = !asubtype(patType, subjectType);

    my_fail = getFail(p, btscopes);
    failLab = my_fail;
    if(!(npats == 0 || char(_) := lpats[-1])){
        failLab = getResume(lpats[-1], btscopes);
    }
    trueCont = muIfElse(muGreaterEqNativeInt(cursor, sublen), trueCont, muFail(failLab));
    //TODO: simplified for compiler: trueCont = muIfElse(muGreaterEqNativeInt(cursor, sublen), trueCont, muFail((npats == 0 || char(_) := lpats[-1]) ? my_fail : getResume(lpats[-1], btscopes)));
    int i = size(lpats) - 1;
    while(i >= 0){
        trueCont = translatePatAsConcreteListElem(lpats[i], lookahead[i], subjectType, subject, sublen, cursor, i, btscopes, 
                                                            trueCont, 
                                                            backtrackFreeConcrete(lpats[i-1]) ? muFail(my_fail) : computeFail(p, lpats, i-1, btscopes, falseCont),
                                                            restore=restore,
                                                            delta=delta);
        i -= delta;
    }
    
    <precise, minSize> = computeConcreteListMinSize(lpats, delta);
    size_test = precise ? muEqualNativeInt(sublen, muCon(minSize)) : muGreaterEqNativeInt(sublen, muCon(minSize));
    
    block = muBlock([ muConInit(sublen, muSize(subject, subjectType)),
                      muIfElse(size_test, 
                               trueCont,
                               falseCont
                              )
                    ]);

    code = muBlock([ *(subjectAssigned ? [muVarInit(subject, subjectExp)] : [muConInit(subject, subjectExp)]),   
                     muVarInit(cursor, muCon(0)), 
                     *(typecheckNeeded ? [muIfElse( muValueIsSubtypeOf(subject, subjectType),
                                                    block
                                                    , falseCont
                                                    )]                                                  
                                       : [ block ])
                     ]);
    //code = muExists(getEnter(p, btscopes), code); // <<
    //iprintln(code);
    return code;
}

MuExp translatePatAsConcreteListElem(t:appl(Production applProd, list[Tree] args), Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int posInPat, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([]), int delta=1) {   
    //println("translatePatAsConcreteListElem:"); iprintln(applProd);
    //println("lex: <lex(_) := applProd.def>");
  	isLex = lex(_) := applProd.def;
    if(isConcreteHole(t)){
        //println("hole: <t>");
        varloc = getConcreteHoleVarLoc(t);
        varname = getConcreteHoleName(t);
        <fuid, pos> = isWildCard(varname) ? <"", 0> : getVariableScope(varname, varloc);
        holeType = getConcreteHoleSymbol(t);
        
        
        //println("holeType = <holeType>");
        if(isIterSymbol(holeType)){
            var = muVar(varname, fuid, pos, symbol2atype(holeType), patternVariableId());
            return translateMultiVarAsConcreteListElem(var, isDefinition(varloc), lookahead, subjectType, subject, sublen, cursor, posInPat, getEnter(t, btscopes), trueCont, falseCont, restore=restore, delta=delta);
     }
   }
   return translateApplAsConcreteListElem(t, lookahead, subjectType, subject, sublen, cursor, posInPat, btscopes, trueCont, falseCont, restore=restore, delta=delta);
}

MuExp translateMultiVarAsConcreteListElem(MuExp var, bool isDefinition, Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int _posInPat, str enter, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([]), int delta=1) {
    fuid =  topFunctionScope();
    v = nextTmp("<unescapeAndStandardize(var.name)>_<abs(var.pos)>");
    startcursor = muTmpInt(v + "_start", fuid);
    savedcursor = muTmpInt(cursor.name + "_saved", fuid);
    len = muTmpInt(v + "_len", fuid);
    prevlen = muTmpInt(v + "_prevlen", fuid);
    //var.atype = alist(avalue()); // = muVar(prettyPrintName("<name>"), fuid, pos, alist(avalue()));
    varPresent = !isWildCard(var.name);
    needsCheck = !asubtype(subjectType, var.atype);
    
    code = muBlock([]);
    if(lookahead.nMultiVar == 0 && !(varPresent && isUsed(var, trueCont) || needsCheck)){
        minLen = nIter(atype2symbol(subjectType))*delta;
        code = muBlock([ muComment("_ case"), muConInit(startcursor, cursor), 
                         muConInit(len, muMulNativeInt(muSubNativeInt(muSubNativeInt(sublen, startcursor), muCon(lookahead.nElem)), muCon(delta))), 
                         muIfElse(muGreaterEqNativeInt(len, muCon(minLen)),
                                  muBlock([ muAssign(cursor, muAddNativeInt(startcursor, len)),
                                            muExists(enter, trueCont)
                                          ]),
                                  falseCont)
                       ]);
                       
    } else {
        if(isDefinition || !varPresent){
           if(needsCheck && !varPresent){
                var.name = var.name + nextTmp();
           }
           asgCursor = muBlock([ muAssign(cursor, muAddNativeInt(startcursor, len)), trueCont ]);
           if(needsCheck){
                asgCursor =  muIf(muValueIsSubtypeOf(var, var.atype), asgCursor);
           }
           code = muBlock([ muConInit(startcursor, cursor),
                            
                            muForRangeInt(enter, len, nIter(atype2symbol(subjectType))*delta, delta, muMulNativeInt(muAbsNativeInt(muSubNativeInt(sublen, startcursor)), muCon(delta)), 
                                          muBlock([ restore,
                                                    *( (varPresent && isUsed(var, trueCont) || (needsCheck && !varPresent)) ? [muConInit(var, muConcreteSubList(subject, startcursor, len, muCon(delta)))] : [] ),
                                                    asgCursor
                                                  ]), 
                                          falseCont)
                          ]);
        } else {
           code = muBlock([ muConInit(startcursor, cursor),
                            muConInit(len, muSubNativeInt(muSubNativeInt(sublen, startcursor), muCon(lookahead.nElem))),   
                            muConInit(prevlen, muSize(var, subjectType)),
                            muIfElse(muGreaterEqNativeInt(len, prevlen), 
                                 muIfElse(muIsInitialized(var),
                                          muIf(muEqual(var, muSubList(subject, startcursor, prevlen)),
                                               muBlock([ muAssign(cursor, muAddNativeInt(startcursor, prevlen)),
                                                         trueCont
                                                       ])),
                                          muBlock([ muAssign(var, muSubList(subject, startcursor, prevlen)),
                                                    muAssign(cursor, muAddNativeInt(startcursor, prevlen)),
                                                    trueCont
                                                  ]))
                                 , falseCont // <<<<
                                 )
                             , falseCont    // <<<<     
                          ]);
        }
   }
    if(needsCheck){
        code = muIf(muValueIsSubtypeOf(subject, var.atype), code);
    }
    return muExists(enter, code); 
}


MuExp translatePatAsConcreteListElem(cc: char(int c), Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int posInPat, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([]), int delta=1) {
   //lengthCheck = muLessNativeInt(cursor, sublen);            
   return muIfElse(muEqual(muCon(cc), muIterSubscript(subject, subjectType, cursor)),  // << add length check
                   muBlock([muIncNativeInt(cursor, muCon(delta)), trueCont ]),
                   falseCont);
}

default MuExp translateApplAsConcreteListElem(t:appl(Production prod, list[Tree] _args), Lookahead _lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int _posInPat, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([]), int delta=1) {
   str fuid = topFunctionScope();
   lengthCheck = muLessNativeInt(cursor, sublen);
   subject_elm = muTmpITree(nextTmp("subject_elm"), fuid);
   body = translateParsedConcretePattern(t, symbol2atype(prod.def), getIterElementType(subjectType), subject_elm, btscopes, trueCont, falseCont, restore=restore);                 
   return muIfElse(lengthCheck,
                   //muAndNativeBool(lengthCheck, muEqual(muCon(prod), muTreeGetProduction(muSubscript(subject, subjectType, muCon(posInPat))))),
                   muBlock([muConInit(subject_elm, muIterSubscript(subject, subjectType, cursor)), muIncNativeInt(cursor, muCon(delta)), body ]),
                   falseCont);
}
     
// ==== qualified name pattern ===============================================

//BTINFO getBTInfo(p:(Pattern) `<QualifiedName name>`, BTSCOPE btscope, BTSCOPES btscopes){
//    enter1 = btscope.enter + nameSuffix("VAR", name);
//    resume1 = enter1;
//    fail1 = btscope.resume;
//    return registerBTScope(p, <enter1, resume1, fail1>, btscopes);
//}  

MuExp inlineVar(MuExp var, MuExp replacement, MuExp cont){
    res = top-down-break visit(cont) { 
        case muAssign(MuExp v1, MuExp exp)  => muAssign(v1, inlineVar(var, replacement, exp))
        case muVarInit(MuExp v1, MuExp exp) => muVarInit(v1, inlineVar(var, replacement, exp))
        case muConInit(MuExp v1, MuExp exp) => muConInit(v1, inlineVar(var, replacement, exp))
        case muVarDecl(MuExp v1): ;
        case MuExp v1 => replacement when v1 is muVar, isSameVar(v1, var)
      };
    return res;
}

bool maybeUsedAsExternal(MuExp var, MuExp cont){
    return /muFun(_,_) := cont;
}

bool maybeAssignedTo(MuExp var, MuExp cont){
    visit(cont){
        case muAssign(MuExp v1, MuExp exp) : if(isSameVar(v1, var)) return true;
        case muVarInit(MuExp v1, MuExp exp) : if(isSameVar(v1, var)) return true;
        case muConInit(MuExp v1, MuExp exp) : if(isSameVar(v1, var)) return true;
    }
    return false;
}

bool inliningBlocked(MuExp var, MuExp trueCont) 
    = maybeUsedAsExternal(var, trueCont) || maybeAssignedTo(var, trueCont);

MuExp translatePat(p:(Pattern) `<QualifiedName name>`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])){
   if(isWildCard("<name>")){
      return trueCont;
   }
   var = mkVar(prettyPrintName(name), name@\loc);
   if(isDefinition(name@\loc) && !subjectAssigned){
     //return inlineVar(var, subjectExp, trueCont);
     //return muValueBlock(abool(), [muVarDecl(var), inlineVar(var, subjectExp, trueCont)]);
     //return muValueBlock(abool(), [muVarInit(var, subjectExp), trueCont]);
     
     // Here is a counter example for which the code below fails.
     // A more precise maybeUsedAsExternal would help
     // See lang::rascal::tests::basic::CompilerIssues::QualifiedName for counter example
    
     if(inliningBlocked(var, trueCont)){ // we cannot inlineVar
         return muValueBlock(abool(), [muVarInit(var, subjectExp), trueCont]);
     } else {
        return muValueBlock(abool(), [muVarDecl(var), inlineVar(var, subjectExp, trueCont)]);
     }
     
   } else {
    return muIfElse(muIsInitialized(var), muIfElse(muMatch(var, subjectExp), trueCont, falseCont),
                                          muBlock([ muAssign(var, subjectExp), trueCont ]));
   }
} 

// ==== typed name pattern ====================================================
     
MuExp translatePat(p:(Pattern) `<Type tp> <Name name>`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])){
   trType = translateType(tp);
   sizeCheck = isIterType(trType) ? muTreeListSize(subjectExp, subjectType) : muSize(subjectExp, subjectType);
   minSizeCheck = (AType::\iter(_) := trType || AType::\iter-seps(_,_) := trType) ? muGreaterEqNativeInt(sizeCheck, muCon(1)) : muCon(true); // TODO: more cases?

   if(!usingTypeParams()){
       if(asubtype(subjectType, trType)){
    	   if(isWildCard("<name>") || subjectAssigned){
    	      return trueCont;
    	   }
    	   ppname = prettyPrintName(name);
    	   <fuid, pos> = getVariableScope(ppname, name@\loc);
    	   var = muVar(prettyPrintName(name), fuid, pos, trType[alabel=ppname], patternVariableId());
    	   if(isSameVar(var, subjectExp)){
    	       return trueCont;
    	   } else {
    	       trueBranch = inliningBlocked(var, trueCont) ? muValueBlock(abool(), [muVarInit(var, subjectExp), trueCont])
    	                                                   : muValueBlock(abool(), [muVarDecl(var), inlineVar(var, subjectExp, trueCont)]);
    	       return muIfElse(minSizeCheck, trueBranch, falseCont);
    	   }
    	   //return var == subjectExp ? trueCont : muIfElse(minSizeCheck, trueBranch, falseCont);
       }
       precond = muAndNativeBool(inSignatureSection() ? muValueIsComparable(subjectExp, trType) : muValueIsSubtypeOf(subjectExp, trType), minSizeCheck);
       if(isWildCard("<name>") || subjectAssigned){
          return muIfElse(precond, trueCont, falseCont);
       }
       ppname = prettyPrintName(name);
       <fuid, pos> = getVariableScope(ppname, name@\loc);
       var = muVar(prettyPrintName(name), fuid, pos, trType[alabel=ppname], patternVariableId());
       if(isSameVar(var, subjectExp)){
            return muIfElse(precond, trueCont, falseCont);
       } else {
            trueBranch = inliningBlocked(var, trueCont) ? muValueBlock(abool(), [muVarInit(var, subjectExp), trueCont])
                                                        : muValueBlock(abool(), [muVarDecl(var), inlineVar(var, subjectExp, trueCont)]);
            return muIfElse(precond, trueBranch, falseCont);
       }
       //return var == subjectExp ? muIfElse(precond, trueCont, falseCont)
       //                         : muIfElse(precond, trueBranch, falseCont);
    } else {
    
        if(inSignatureSection()){
            precond = minSizeCheck; //muAndNativeBool(muMatchAndBind(subjectExp, trType), minSizeCheck);
            if(isWildCard("<name>") || subjectAssigned){
                return muIfElse(precond, trueCont, falseCont);
            }
            return muIfElse(precond, trueCont, falseCont);
            //ppname = prettyPrintName(name);
            //<fuid, pos> = getVariableScope(ppname, name@\loc);
            //var = muVar(prettyPrintName(name), fuid, pos, trType[alabel=ppname], patternVariableId());
            //return var == subjectExp ? muIfElse(precond, trueCont, falseCont)
            //                         : muIfElse(precond, muBlock([muVarInit(var, subjectExp), trueCont]), falseCont);
        
        } else {
            precond = muAndNativeBool(muValueIsSubtypeOfInstantiatedType(subjectExp, trType), minSizeCheck);
            if(isWildCard("<name>") || subjectAssigned){
                return muIfElse(precond, trueCont, falseCont);
            }
            ppname = prettyPrintName(name);
            <fuid, pos> = getVariableScope(ppname, name@\loc);
            var = muVar(prettyPrintName(name), fuid, pos, trType[alabel=ppname], patternVariableId());
            if(isSameVar(var, subjectExp)){
                return muIfElse(precond, trueCont, falseCont);
            } else {
                trueBranch = maybeUsedAsExternal(var, trueCont) ? muValueBlock(abool(), [muVarInit(var, subjectExp), trueCont])
                                                                : muValueBlock(abool(), [muVarDecl(var), inlineVar(var, subjectExp, trueCont)]);
                return muIfElse(precond, trueBranch, falseCont);
            }
            //return var == subjectExp ? muIfElse(precond, trueCont, falseCont)
            //                         : muIfElse(precond, trueBranch, falseCont);
        }
    }
}  

// ==== reified type pattern ==================================================
//TODO
MuExp translatePat(p:(Pattern) `type ( <Pattern symbol> , <Pattern definitions> )`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])) {    
    throw "Not implemented";
    //return muApply(mkCallToLibFun("Library","MATCH_REIFIED_TYPE"), [muCon(symbol)]);
}

// ==== call or tree pattern ==================================================

// ---- getBTInfo

str consLabel((Pattern) `<StringLiteral s>`) = "STR";
str consLabel((Pattern) `<QualifiedName s>`) = asJavaName(asUnqualifiedName("<s>"));
str consLabel((Pattern) `<Type tp> <Name nm>`) = asJavaName("<nm>");
default str consLabel(Pattern p) = "GEN";

BTINFO getBTInfo(p:(Pattern) `<Pattern expression> ( <{Pattern ","}* arguments> <KeywordArguments[Pattern] keywordArguments> )`, BTSCOPE btscope, BTSCOPES btscopes) {
    enterCall = "<btscope.enter>_CONS_<consLabel(expression)>";
   
    <btscopeLast, btscopes> = getBTInfo(expression, <enterCall, btscope.resume, btscope.resume>, btscopes);
    for(pat <- arguments){
        <btscopeLast, btscopes> = getBTInfo(pat, btscopeLast, btscopes);
    }
    if(keywordArguments is \default){
        for(kwpat <- keywordArguments.keywordArgumentList){
            <btscopeLast, btscopes> = getBTInfo(kwpat.expression, btscopeLast, btscopes);
        }
    }
    return registerBTScope(p, <enterCall, btscopeLast.resume, btscope.resume>, btscopes);
}

MuExp translatePat(p:(Pattern) `<Pattern expression> ( <{Pattern ","}* arguments> <KeywordArguments[Pattern] keywordArguments> )`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])) {
   //iprintln(btscopes);
   //println("translatePat: <p>, <p@\loc>");
   str fuid = topFunctionScope();
   subjectExpIsVar = isVarOrTmp(subjectExp);
   subject = subjectExpIsVar ? subjectExp : muTmpIValue(nextTmp("subject"), fuid, subjectType);
   contExp = trueCont;
   patType = getType(p);
   expType = getType(expression);
 
   if((KeywordArguments[Pattern]) `<OptionalComma _> <{KeywordArgument[Pattern] ","}+ keywordArgumentList>` := keywordArguments){
        for(kwarg <- keywordArgumentList){
            kwname = prettyPrintName(kwarg.name);
            contExp = translatePat(kwarg.expression, getType(kwarg.expression), muGetKwp(subject, subjectType, kwname), btscopes, contExp, falseCont, restore=restore);                 
        }
   }
   
   list[Pattern] lpats = [pat | pat <- arguments];   //TODO: should be unnnecessary
   MuExp body = contExp;
   MuExp code = muBlock([]);
   nonterminal_get_arg = isNonTerminalAType(patType);
   for(int i <- reverse(index(lpats))){
       MuExp arg_fail = computeFail(p, lpats, i-1, btscopes, falseCont);
      
       MuExp arg = nonterminal_get_arg
                 ? muPrim("nonterminal-get-arg", avalue(), [subjectType, aint()], [subject, muCon(i)], lpats[i]@\loc)
                 : muSubscript(subject, subjectType, muCon(i))
                 ;
 
      // Check the argument type, needed to distinguish argument patterns like mucon({}) vs muCon([])
    
      MuExp arg_val = muTmpIValue(nextTmp("arg<i>_"), fuid, avalue());
      AType arg_type  = getType(lpats[i]);
      // Replace all occurrences of the argument by the var representing its computed value
      body = visit(body){
        case muVar(str name, str fuid, int pos, AType atype, IdRole idRole) => arg_val
             when name == arg_type.alabel, arg_type := atype
      };
      
      body = muBlock([ muVarInit(arg_val, arg),
                       muIfElse(muValueIsComparable(arg_val, arg_type), 
                                translatePat(lpats[i], getType(lpats[i]), arg_val, btscopes, body, arg_fail, restore=restore),
                                arg_fail)
                     ]);
   }
 
   
   subjectInit = subjectExpIsVar ? muBlock([]) : muConInit(subject, subjectExp);
   if(expression is qualifiedName){
      qname = "";
      if(expType.alabel?){
        qname = expType.alabel;
      } else if(overloadedAType(rel[loc, IdRole, AType] overloads) := expType,
                any(<_, _, tp> <- overloads, tp.alabel?)){
        for(<_, _, tp> <- overloads){
            if(tp.alabel?){
                qname = tp.alabel; break;
            }
        }
      } else {
        throw "Cannot get name in call pattern: <expType>";
      }
      fun_name = asUnqualifiedName(prettyPrintName(qname));
      code = muBlock([subjectInit, muIfElse(muHasNameAndArity(subjectType, expType, muCon(fun_name), size(lpats), subject), body, falseCont)]);
      
      return code;
   } else if(expression is literal){ // StringConstant
      fun_name = prettyPrintName("<expression>"[1..-1]); //TODO escapes in string
      code = muBlock([subjectInit, muIfElse(muHasNameAndArity(subjectType, expType, muCon(fun_name), size(lpats), subject), body, falseCont)]);
      return code;
    } else {
     fun_name_subject = muTmpIValue(nextTmp("fun_name_subject"), fuid, expType);
     code = muBlock([subjectInit,
                     muIfElse(muValueIsSubtypeOf(subject, anode([])),
                              muBlock([ muConInit(fun_name_subject, muPrim("get_anode_name", astr(), [anode([])], [subject], getLoc(expression))),
                                        translatePat(expression, expType, fun_name_subject, btscopes, 
                                                     muIfElse(muHasNameAndArity(subjectType, expType, fun_name_subject, size(lpats), subject), body, falseCont),  
                                                     falseCont,
                                                     subjectAssigned=false,
                                                     restore=restore)
                                  ]),
                               falseCont)
                    ]);         
   }
   code = muExists(getEnter(p, btscopes), code);
   return code;
}

 //MuExp translatePatKWArguments((KeywordArguments[Pattern]) ``, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([]))
 //   = trueCont;
 
 //MuExp translatePatKWArguments((KeywordArguments[Pattern]) `<KeywordArguments[Pattern] keywordArguments>`, AType _subjectType, MuExp subjectExp, BTSCOPES _btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false/*, MuExp restore=muBlock([])*/){
 //   code = trueCont;
 //   for(kwarg <- keywordArguments.keywordArgumentList){
 //       kwtype = getType(kwarg.expression);
 //       kwfield = "<kwarg.name>";
 //       code = muIfElse(muHasKwp(subjectExp, kwfield),
 //                       muIfElse(muEqual(muGetKwp(subjectExp, kwtype, kwfield), translate(kwarg.expression)), code, falseCont),
 //                       falseCont);
 //  }
 //  return code;
 //}

// ==== set pattern ===========================================================

// ---- getBTInfo
    
BTINFO getBTInfo(p:(Pattern) `{<{Pattern ","}* pats>}`,  BTSCOPE btscope, BTSCOPES btscopes){
    <fixedLiterals, toBeMatchedPats, fixedVars, fixedMultiVars, leftMostVar> = analyzeSetPattern(p);
   
    enterSet = "<btscope.enter>_SET";
    BTSCOPE btscopeLast = <enterSet, btscope.resume, btscope.resume>;
    for(pat <- toBeMatchedPats){
        <btscopeLast, btscopes> = getBTInfoSet(pat, btscopeLast, btscopes);
    }
    return registerBTScope(p, <enterSet, btscopeLast.resume, btscope.resume>, btscopes);
}

// ---- translate set pattern

MuExp translatePat(p:(Pattern) `{<{Pattern ","}* pats>}`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])) {
    return translateSetPat(p, subjectType, subjectExp, btscopes, trueCont, falseCont/*, subjectAssigned=subjectAssigned*/);
}
// Translate patterns as element of a set pattern

str isLast(bool b) = b ? "LAST_" : "";

BTINFO getBTInfoSet(p:(Pattern) `<QualifiedName name>`, BTSCOPE btscope, BTSCOPES btscopes){
    enter1 = btscope.enter + nameSuffix("VAR", name);
    resume1 = enter1;
    fail1 = btscope.resume;
    return registerBTScope(p, <enter1, resume1, fail1>, btscopes);
}  

MuExp translatePatAsSetElem(p:(Pattern) `<QualifiedName name>`, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont/*, MuExp restore=muBlock([])*/) {
    return translateVarAsSetElem(mkVar(p), isDefinition(p), p@\loc, last, elmType, subject, prevSubject, btscopes, trueCont, falseCont/*, restore=restore*/);
}

BTINFO getBTInfoSet(p:(Pattern) `<Type tp> <Name name>`, BTSCOPE btscope, BTSCOPES btscopes) {
    enter1 = btscope.enter + nameSuffix("VAR", name);
    resume1 = enter1;
    fail1 = btscope.resume;
    return registerBTScope(p, <enter1, resume1, fail1>, btscopes);
}

MuExp translatePatAsSetElem(p:(Pattern) `<Type tp> <Name name>`, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont/*, MuExp restore=muBlock([])*/) {
    return translateVarAsSetElem(mkVar(p), isDefinition(p), p@\loc, last, elmType, subject, prevSubject, btscopes, trueCont, falseCont/*, restore=restore*/);
}

MuExp translateVarAsSetElem(MuExp var, bool isDefinition, loc patloc, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont/*, MuExp restore=muBlock([])*/) {
   fuid = topFunctionScope();
   elem = muTmpIValue(nextTmp("elem"), fuid, elmType);
   varPresent = !isWildCard(var.name);
   needsCheck = !asubtype(elmType, var.atype);
   
   if(!varPresent){
        var.name = var.name + nextTmp();
   }
   
   my_btscope = btscopes[patloc];
   code = muBlock([]);
  
   if(!varPresent && !needsCheck){
       code = muForAll(my_btscope.enter, elem, aset(elmType), prevSubject,
                       muBlock([ muConInit(subject, muPrim("delete", aset(elmType), [aset(elmType), elmType], [prevSubject, elem], patloc)),
                                 trueCont
                               ]),
                       falseCont); //muBlock([]));     
   } else {
	   trueBlock = muBlock([ muConInit(subject, muPrim("delete", aset(elmType), [aset(elmType), elmType], [prevSubject, var], patloc)),
                             trueCont
                           ]);
	   if(isDefinition){
	       needInit = isUsed(var, trueCont) || !last;
	       body = muBlock([ *(needInit  ? [muVarInit(var, elem)] : []),
                            trueBlock
                          ]);
	       if(needsCheck){
    	      body = muIf(muValueIsSubtypeOf(elem, var.atype), body);
    	   }     
    	   code = muForAll(my_btscope.enter, elem, aset(elmType), prevSubject, body, falseCont); //muBlock([]));                           
	   } else {
	       code = muForAll(my_btscope.enter, elem, aset(elmType), prevSubject,
	                       muIfElse(muIsInitialized(var), muIf(muEqual(elem, var), trueBlock),
	                                                      muBlock([ muAssign(var, elem), trueBlock ])),
	                       falseCont); //muBlock([]));
	   }
   }
   
   //if(needsCheck){
   //     code = muIfElse(muValueIsSubtypeOf(prevSubject, aset(var.atype)), code, muFail(getFail(patloc, btscopes)));
  // }
   
   return muIfElse(//last ? muEqualNativeInt(muSize(prevSubject, aset(elmType)), muCon(1)) 
                            muGreaterEqNativeInt(muSize(prevSubject, aset(elmType)), muCon(1)), 
                   code //muBlock([ code, falseCont ])
                   , falseCont
                   );
} 

BTINFO getBTInfoSet(p:(Pattern) `<QualifiedName name>*`, BTSCOPE btscope, BTSCOPES btscopes) {
    enter1 = btscope.enter + nameSuffix("MVAR", name);
    resume1 = enter1;
    fail1 = btscope.resume;
    return registerBTScope(p, <enter1, resume1, fail1>, btscopes);
}

//MuExp translatePatAsSetElem(p:(Pattern) `_*`, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont){
//    return translateMultiVarAsSetElem(mkVar(p), false, p@\loc, last, elmType, subject, prevSubject, btscopes, trueCont, falseCont);
//}

BTINFO getBTInfoSet(p:(Pattern) `*<Name name>`, BTSCOPE btscope, BTSCOPES btscopes) {
    enter1 = btscope.enter + nameSuffix("MVAR", name);
    resume1 = enter1;
    fail1 = btscope.resume;
    return registerBTScope(p, <enter1, resume1, fail1>, btscopes);
}

//MuExp translatePatAsSetElem(p:(Pattern) `*_`, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont){
//    return translateMultiVarAsSetElem(mkVar(p), false, p@\loc, last, elmType, subject, prevSubject, btscopes, trueCont, falseCont);
//}

MuExp translatePatAsSetElem(p:(Pattern) `<QualifiedName name>*`, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont) {
    if("<name>" == "_"){
        return translateMultiVarAsSetElem(mkVar(p), false, p@\loc, last, elmType, subject, prevSubject, btscopes, trueCont, falseCont);
    }
    return translateMultiVarAsSetElem(mkVar(p), isDefinition(name@\loc), p@\loc, last, elmType, subject, prevSubject, btscopes, trueCont, falseCont);  
}

MuExp translatePatAsSetElem(p:(Pattern) `*<Name name>`, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont) {
    if("<name>" == "_"){
        return translateMultiVarAsSetElem(mkVar(p), false, p@\loc, last, elmType, subject, prevSubject, btscopes, trueCont, falseCont);
    }
    return translateMultiVarAsSetElem(mkVar(p), isDefinition(name@\loc), p@\loc, last, elmType, subject, prevSubject, btscopes, trueCont, falseCont); 
}
 
BTINFO getBTInfoSet(p:(Pattern) `*<Type tp> <Name name>`, BTSCOPE btscope, BTSCOPES btscopes) {
    enter1 = btscope.enter + nameSuffix("MVAR", name);
    resume1 = enter1;
    fail1 = btscope.resume;
    return registerBTScope(p, <enter1, resume1, fail1>, btscopes);
} 

//MuExp translatePatAsSetElem(p:(Pattern) `*<Type tp>  _`, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont) {
//   return translateMultiVarAsSetElem(mkVar(p), false, p@\loc, last, elmType, subject, prevSubject, btscopes, trueCont, falseCont);
//}

MuExp translatePatAsSetElem(p:(Pattern) `*<Type tp> <Name name>`, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont) {
   if("<name>" == "_"){
    return translateMultiVarAsSetElem(mkVar(p), false, p@\loc, last, elmType, subject, prevSubject, btscopes, trueCont, falseCont);
   }
   
   return translateMultiVarAsSetElem(mkVar(p), true, p@\loc, last, elmType, subject, prevSubject, btscopes, trueCont, falseCont);
}

MuExp translateMultiVarAsSetElem(MuExp var, bool isDefinition, loc patsrc, bool _last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont) {
   fuid = topFunctionScope();
   elem = muTmpIValue(nextTmp("elem"), fuid, aset(var.atype));
  
   my_btscope = btscopes[patsrc];
   code = muBlock([]);

   varPresent = !isWildCard(var.name); 
   needsCheck = !asubtype(aset(elmType), var.atype);
   
   if(!varPresent){
        var.name = var.name + nextTmp();
   }
  
   if(!varPresent && !needsCheck){
        code = muForAll(my_btscope.enter, elem, aset(elmType), muPrim("subsets", aset(elmType), [aset(elmType)], [prevSubject], patsrc),
                        muBlock([ muConInit(subject, muPrim("subtract", aset(elmType), [aset(elmType), aset(elmType)], [prevSubject, elem], patsrc)),
                                  trueCont
                                ]),
                        falseCont);
                      
    } else {
        if(isDefinition || !varPresent){
            asgSubject =  muBlock([ muConInit(subject, muPrim("subtract", aset(elmType), [aset(elmType), aset(elmType)], [prevSubject, elem], patsrc)),
                                    trueCont
                                  ]);
           
            if(needsCheck){
                asgSubject =  muIf(muValueIsSubtypeOf(var, var.atype), asgSubject);
            }
            code = muForAll(my_btscope.enter, elem, aset(elmType), muPrim("subsets", aset(elmType), [aset(elmType)], [prevSubject], patsrc),
                            muBlock([ muVarInit(var, elem),
                                      asgSubject
                                    ]),
                            falseCont);
        } else {
            trueBlock = muBlock([ muConInit(subject, muPrim("subtract", aset(elmType), [aset(elmType), aset(elmType)], [prevSubject, elem], patsrc)),
                                  trueCont
                                 ]);
            initialized = muTmpBool("initialized", fuid);   
                 
            asgVar = muBlock([ muAssign(var, elem), trueBlock]);
            if(needsCheck){
                asgVar =  muIf(muValueIsSubtypeOf(var, var.atype), asgVar);
            }
            code = muBlock([ muConInit(initialized, muIsInitialized(var)),
                             muForAll(my_btscope.enter, elem, aset(elmType), muPrim("subsets", aset(elmType), [aset(elmType)], [prevSubject], patsrc),
                                      muBlock([ muIfElse(initialized, muIf(muEqual(elem, var),  trueBlock),
                                                                      asgVar)
                                              ]),
                                      falseCont)
                          ]);
        }
    }
    
    //if(needsCheck){
    //    code = muIfElse(muValueIsSubtypeOf(prevSubject, aset(elmType)), 
    //                    code
    //                    , falseCont//, muFail(getFail(my_btscope))
    //                );
    //} 
        
   return code;
}

MuExp translatePatAsSetElem(p:(Pattern) `+<Pattern argument>`, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont) {
  throw "splicePlus pattern <p>";
}   

BTINFO getBTInfoSet(p:(Pattern) `<Name name> : <Pattern pattern>`,  BTSCOPE btscope, BTSCOPES btscopes){
    <btscope, btscopes> = getBTInfo(pattern, btscope, btscopes);
    return registerBTScope(p, btscope, btscopes);
}
MuExp translatePatAsSetElem(p:(Pattern) `<Name name> : <Pattern pattern>`, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont) { 
    str fuid = topFunctionScope();
    elem = muTmpIValue(nextTmp("elem"), fuid, elmType);
    var = mkVar(p);
    varPresent = !isWildCard(var.name);
    needsCheck = !asubtype(elmType, var.atype);
    if(!varPresent){
        var.name = var.name + nextTmp();
    }
    
    my_btscope = btscopes[getLoc(p)];
    asgVar = trueCont;
    if(varPresent){
        asgVar = muBlock([ muAssign(var, elem), trueCont]);
        if(needsCheck){
            asgVar =  muIf(muValueIsSubtypeOf(var, var.atype), asgVar);
        }
    }
    // TODO length check?
    forAll_scope = my_btscope.enter+ "_NAMED_SET_ELM";
    code = muForAll(forAll_scope, elem, aset(elmType), prevSubject,
                    translatePat(p, elmType, elem, btscopes, 
                        muBlock([ muConInit(subject, muPrim("delete", aset(elmType), [aset(elmType), elmType], [prevSubject, elem], p@\loc)),
                                  asgVar
                                ]),            
                        muFail(forAll_scope)
                        ),
                    muBlock([]));
    //return code;
    return redirect(code, my_btscope.enter, forAll_scope);
} 

BTINFO getBTInfoSet(p:(Pattern) `<Type tp> <Name name> : <Pattern pattern>`,  BTSCOPE btscope, BTSCOPES btscopes){
    <btscope, btscopes> = getBTInfo(pattern, btscope, btscopes);
    return registerBTScope(p, btscope, btscopes);
}

MuExp translatePatAsSetElem(p:(Pattern) `<Type tp> <Name name> : <Pattern pattern>`, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont) {
    str fuid = topFunctionScope();
    elem = muTmpIValue(nextTmp("elem"), fuid, elmType);
    var = mkVar(p);
    varPresent = !isWildCard(var.name);
    needsCheck = !asubtype(elmType, var.atype);
    if(!varPresent){
        var.name = var.name + nextTmp();
    }
    
    my_btscope = btscopes[getLoc(p)];
    
    asgVar = muBlock([ muAssign(var, elem), trueCont]);
    if(needsCheck){
        asgVar =  muIf(muValueIsSubtypeOf(var, var.atype), asgVar);
    }
    
    asgVar = trueCont;
    if(varPresent){
        asgVar = muBlock([ muAssign(var, elem), trueCont]);
        if(needsCheck){
            asgVar =  muIf(muValueIsSubtypeOf(var, var.atype), asgVar);
        }
    }
    // TODO length check?
    forAll_scope = my_btscope.enter + "_NAMED_SET_ELM";
    code = muForAll(forAll_scope, elem, aset(elmType), prevSubject,
                    translatePat(p, elmType, elem, btscopes, 
                        muBlock([ muConInit(subject, muPrim("delete", aset(elmType), [aset(elmType), elmType], [prevSubject, elem], p@\loc)),
                                  asgVar
                                ]),            
                        muFail(forAll_scope)
                        ),
                    muBlock([]));
    //return code;
    return redirect(code, my_btscope.enter, forAll_scope);
}  

BTINFO getBTInfoSet(p:(Pattern) `<Pattern expression> ( <{Pattern ","}* arguments> <KeywordArguments[Pattern] keywordArguments> )`, BTSCOPE btscope, BTSCOPES btscopes){ 
    //return getBTInfo(p, btscope, btscopes);
    enter1 = btscope.enter;
    return getBTInfo(p, <enter1, enter1 + "_CONS_<consLabel(expression)>", enter1>, btscopes);
}

BTINFO getBTInfoSet(p:(Pattern) `[<{Pattern ","}* pats>]`, BTSCOPE btscope, BTSCOPES btscopes)
    = getBTInfo(p, btscope, btscopes);
    
BTINFO getBTInfoSet(p:(Pattern) `{<{Pattern ","}* pats>}`, BTSCOPE btscope, BTSCOPES btscopes)
    = getBTInfo(p, btscope, btscopes);
    
BTINFO getBTInfoSet(p:(Pattern) `\<<{Pattern ","}* pats>\>`, BTSCOPE btscope, BTSCOPES btscopes)
    = getBTInfo(p, btscope, btscopes);

default BTINFO getBTInfoSet(Pattern p, BTSCOPE btscope, BTSCOPES btscopes) {
    enter1 = btscope.enter + nextTmp("_DFLT"); // nextTmp("_DFLT_SET_ELM");
    resume1 = enter1;
    fail1 = btscope.resume;
    return registerBTScope(p, <enter1, resume1, resume1>, btscopes);
} 

default MuExp translatePatAsSetElem(Pattern p, bool last, AType elmType, MuExp subject, MuExp prevSubject, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont) {

//TODO: take last element into account
  try {
        pcon = muCon(translatePatternAsConstant(p));
        return muIfElse(muPrim("in", abool(), [elmType, aset(elmType)], [pcon, prevSubject], p@\loc),
                        muBlock([ muConInit(subject, muPrim("delete", aset(elmType), [aset(elmType), elmType], [prevSubject, pcon], p@\loc)),
                                  trueCont ]),
                        falseCont);                            
  } catch: {
        str fuid = topFunctionScope();
        elem = muTmpIValue(nextTmp("elem"), fuid, elmType);
        my_btscope = btscopes[getLoc(p)];
        // TODO length check?
        forAll_scope = my_btscope.enter + nextTmp("_DFLT_SET_ELM");
        code = muForAll(forAll_scope, elem, aset(elmType), prevSubject,
                        translatePat(p, elmType, elem, btscopes, 
                            muBlock([ muConInit(subject, muPrim("delete", aset(elmType), [aset(elmType), elmType], [prevSubject, elem], p@\loc)),
                                      trueCont ]),            
                            muFail(forAll_scope, comment="default set elem")
                            ),
                        falseCont //muBlock([])
                      );
        //return code;
        return redirect(code, my_btscope.enter, forAll_scope);
  }
}

/*
 * Get the name of a pattern at position k, when no name, return "_<k>".
 */
private str getName(Pattern pat, int k){
  if(pat is splice){
     arg = pat.argument;
     return arg is qualifiedName ? prettyPrintName(arg.qualifiedName) : prettyPrintName(arg.name);
  } else if(pat is multiVariable){
    return prettyPrintName(pat.qualifiedName); 
  } else if(pat is qualifiedName){
    return prettyPrintName(pat.qualifiedName);  
  } else if(pat is typedVariable){
    return prettyPrintName(pat.name);
  } else {
    return "_<k>";
  } 
}

private bool isDefinition(Pattern pat){
  if(pat is splice){
     return isDefinition(pat.argument);
  } else if(pat is multiVariable){
    return isWildCard("<pat.qualifiedName>")|| isDefinition(pat.qualifiedName@\loc); 
  } else if(pat is qualifiedName){
    return isWildCard("<pat>") || isDefinition(pat.qualifiedName@\loc);  
  } else if(pat is typedVariable){
    return true;
  } else 
    return false;
}

private bool isDefinedOutsidePat(loc def, Pattern container){
    try {
        defined = getDefinition(def); 
        return !isContainedIn(defined, container@\loc);
    } catch: {
         return false;
    }
}

private bool allVarsDefinedOutsidePat(Pattern pat, Pattern container){
  if(pat is splice){
     return allVarsDefinedOutsidePat(pat.argument, container);
  } else if(pat is multiVariable){
        if(isWildCard("<pat.qualifiedName>")) return false;
        return isDefinedOutsidePat(pat.qualifiedName@\loc, container);
  } else if(pat is qualifiedName){
        if(isWildCard("<pat>")) return false;
        return isDefinedOutsidePat(pat.qualifiedName@\loc, container);  
  } else if(pat is typedVariable){
    return false;
  } else {
    bool found = true;
    visit(pat){
        case (Pattern) `<QualifiedName qualifiedName>`:  found = found && isDefinedOutsidePat(qualifiedName@\loc, container);
        case (Pattern) `<QualifiedName qualifiedName>*`: found = found && isDefinedOutsidePat(qualifiedName@\loc, container);
    }
    return found;
    }
}

private MuExp mkVar(Pattern pat){
   if(pat is splice){
     tp = getType(pat);
     return mkVar(pat.argument);
  } else if(pat is multiVariable){
        if(isWildCard("<pat.qualifiedName>")){
            return muVar("<pat.qualifiedName>", topFunctionScope(), -1, avalue(), patternVariableId());
        } else {
            return mkVar("<pat.qualifiedName>", pat.qualifiedName@\loc);
        }
  } else if(pat is qualifiedName){
        if(isWildCard("<pat>")){
             return muVar("<pat>", topFunctionScope(), -1, avalue(), patternVariableId());
        } else {
            return mkVar("<pat>", pat@\loc);
        }
  } else if(pat is typedVariable){
        if(isWildCard("<pat.name>")){
             return muVar("<pat.name>", topFunctionScope(), -1, getType(pat.name), patternVariableId());
         } else {
            return mkVar("<pat.name>", pat.name@\loc);
         }
  } else if(pat is variableBecomes){
        if(isWildCard("<pat.name>")){
             return muVar("<pat.name>", topFunctionScope(), -1, avalue(), patternVariableId());
        } else {
            return mkVar("<pat.name>", pat.name@\loc);
        }
  } else if(pat is typedVariableBecomes){
        if(isWildCard("<pat.name>")){
            return muVar("<pat.name>", topFunctionScope(), -1, getType(pat.name), patternVariableId());
        } else {
            return mkVar("<pat.name>", pat.name@\loc);
        }
  } else
    throw "mkVar: <pat>";
}

//private MuExp mkVar(Tree pat){
//    if(pat has name){
//        return mkVar("<pat.name>", pat.name@\loc);
//    }
//    throw "mkVar: no name field found in <pat>";
//}

tuple[list[MuExp] literals, list[Pattern] toBeMatched, list[Pattern] vars, list[Pattern] multiVars, int leftMostVar] analyzeSetPattern(p:(Pattern) `{<{Pattern ","}* pats>}`){
   list[Pattern] lpats = [pat | pat <- pats]; // TODO: unnnecessary

   /* collect literals and already defined vars/multivars; also remove patterns with duplicate names */
   fixedLiterals = [];                  // constant elements in the set pattern
   list[Pattern] toBeMatchedPats = [];  // the list of patterns that will ultimately be matched
   fixedVars = [];                      // var pattern elements with already (previosuly) defined value
   fixedMultiVars = [];                 // multi-var pattern elements with already (previosuly) defined value
   int leftMostVar = -1;                // index of leftmost multi-variable
   
    outer: for(int i <- index(lpats)){
              pat = lpats[i];
              str name = getName(pat, i);
              if(!isWildCard(name)){
                  for(int j <- [0 .. i]){
                      if(getName(lpats[j], j) == name){
                         continue outer;
                      }
                  }
              }
              if(pat is literal){
                fixedLiterals += isConstant(pat.literal) ? muCon(getLiteralValue(pat.literal)) : translate(pat.literal);
              } else if(pat is splice || pat is multiVariable){
                if(allVarsDefinedOutsidePat(pat, p)){
                    fixedMultiVars += pat;
                } else {
                    if(leftMostVar == -1) leftMostVar = size(toBeMatchedPats);
                    toBeMatchedPats += pat;
                }
              } else if(pat is qualifiedName){
                if(allVarsDefinedOutsidePat(pat, p)){
                    fixedVars += pat;
               } else {
                    if(leftMostVar == -1) leftMostVar = size(toBeMatchedPats);
                    toBeMatchedPats += pat;               
                }
              } else if(pat is typedVariable){
                    if(leftMostVar == -1) leftMostVar = size(toBeMatchedPats);
                    toBeMatchedPats += pat;
              } else { 
                try {
                    fixedLiterals += muCon(translatePatternAsConstant(pat));
                } catch: {
                    toBeMatchedPats += pat;
                }
              }
           } 
   return <fixedLiterals, toBeMatchedPats, fixedVars, fixedMultiVars, leftMostVar>;
}

/*
 * Translate a set pattern: 
 * - since this is a set, for patterns with the same name, duplicates are removed.
 * - all literal patterns are separated
 * - all other patterns are compiled in order
 * - if the last pattern is a multi-variable it is treated specially.
 * Note: there is an unused optimization here: if the last multi-var in the pattern is followed by other patterns
 * AND these patterns do not refer to that variable, then the multi-var can be moved to the end of the pattern.
*/

MuExp translateSetPat(p:(Pattern) `{<{Pattern ","}* _>}`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont/*, bool subjectAssigned=false*/) {
    //iprintln(btscopes);
    try {
        constantPat = translatePatternAsConstant(p);
        return muIfExp(muEqual(subjectExp, muCon(constantPat)), trueCont, falseCont);
    } catch _: /* not a constant pattern */;
    
   elmType = avalue();
   if(aset(tp) := subjectType && tp != avoid()) elmType = tp;
   typecheckNeeded = !asubtype(getType(p), subjectType);
   my_btscope = btscopes[getLoc(p)];
   //btscope = my_btscope.enter;
   
   fixedLiterals = [];                  // constant elements in the set pattern
   list[Pattern] toBeMatchedPats = [];  // the list of patterns that will ultimately be matched
   list[Pattern] fixedVars = [];        // var pattern elements with already (previosuly) defined value
   list[Pattern] fixedMultiVars = [];   // multi-var pattern elements with already (previosuly) defined value
   int leftMostVar = -1;                // index of leftmost multi-variable
   
   <fixedLiterals, toBeMatchedPats, fixedVars, fixedMultiVars, leftMostVar> = analyzeSetPattern(p);
   rightMostPat = size(toBeMatchedPats) - 1;
   
   str fuid = topFunctionScope();
   subject = muTmpIValue(nextTmp("subject"), fuid, subjectType); // <<< type?
   fixed = muTmpIValue(nextTmp("fixed"), fuid, subjectType);     // <<<
   subjects = [ muTmpIValue(nextTmp("subject"), fuid, subjectType) | int _ <- reverse(index(toBeMatchedPats)) ];
   
   //for(int i <- index(toBeMatchedPats)){
   //     println("<i>: <toBeMatchedPats[i]> =\> <subjects[i]>");
   //}
   
   MuExp fixedParts = muCon({con | muCon(value con) <- fixedLiterals });
    
   for(vp <- fixedVars){
       fixedParts = muPrim("add", aset(elmType), [aset(elmType), elmType], [fixedParts, mkVar(vp)], p@\loc);
   }
   for(vp <- fixedMultiVars){
       fixedParts = muPrim("add", aset(elmType), [aset(elmType), aset(elmType)], [fixedParts, mkVar(vp)], p@\loc);
   }
   subject_minus_fixed = muPrim("subtract", aset(elmType), [aset(elmType), aset(elmType)], [subject, fixed], p@\loc);
   
   MuExp setPatTrueCont =
        isEmpty(subjects) ? ( ( isEmpty(fixedLiterals) && isEmpty(fixedVars) && isEmpty(fixedMultiVars) )
                            ? muIfExp(muEqualNativeInt(muSize(subject, aset(avalue())), muCon(0)), trueCont,  muFail(getFail(my_btscope), comment="set pat1"))
                            : muIfExp(muEqualNativeInt(muSize(subject_minus_fixed, aset(avalue())), muCon(0)), trueCont, muFail(getFail(my_btscope), comment="set pat2"))
                            )
                          : muIfExp(muEqualNativeInt(muSize(subjects[-1], aset(avalue())), muCon(0)), trueCont,  muFail(getResume(my_btscope), comment="set pat3"))
                          ;
   //iprintln(setPatTrueCont);
   for(int i <- reverse(index(toBeMatchedPats))){
      pat = toBeMatchedPats[i];
      isRightMostPat = (i == rightMostPat);
      currentSubject = subjects[i];
      previousSubject = (i == 0) ? subject : subjects[i-1];
      resumePrevious = (i == 0) ? falseCont : muFail(getResume(toBeMatchedPats[i-1], btscopes), comment="set pat4");
      setPatTrueCont = translatePatAsSetElem(pat, isRightMostPat, elmType, currentSubject, previousSubject, btscopes, setPatTrueCont, resumePrevious);
   }
   
   block = muBlock([]);
   if(isEmpty(fixedLiterals) && isEmpty(fixedVars) && isEmpty(fixedMultiVars)){
        block = setPatTrueCont;
   } else {
        block = muBlock([ muConInit(fixed, fixedParts),
                          muIfElse(muPrim("subset", aset(elmType), [aset(elmType), aset(elmType)], [fixed, subject], p@\loc),
                                   muBlock([ *(leftMostVar <= 0 ? [muAssign(subject, subject_minus_fixed)] : [muConInit(subjects[leftMostVar-1], subject)]),
                                             setPatTrueCont]),
                                   muFail(getFail(my_btscope), comment="set pat5"))
                        ]);
   }
   //iprintln(block);
   code = muBlock([ muVarInit(subject, subjectExp),
                    *( typecheckNeeded ? [muIfElse( muValueIsSubtypeOf(subject, subjectType),
                                                    block
                                                    , falseCont
                                                    )
                                         ]
                                       : [ block ])
                   ,  *[] //*( noSequentialExit(block) ? [] : [falseCont] )
                       ]);
    //code = muExists(getEnter(p, btscopes)+"_outer", code);
    return code;
}

// ==== tuple pattern =========================================================

// ---- getBTInfo

BTINFO getBTInfo(p:(Pattern) `\<<{Pattern ","}* pats>\>`, BTSCOPE btscope, BTSCOPES btscopes) {
    enterTuple = "<btscope.enter>_TUPLE";
    BTSCOPE btscopeLast =  <enterTuple, /*btscope.resume,*/ enterTuple, btscope.resume>; //<enterTuple, btscope.resume, btscope.resume>;
    pat_list = [pat | Pattern pat <- pats ];
    for(int i <- index(pat_list)){
        pat = pat_list[i];
        //btscopeLast.enter += "<i>";
        //btscopeLast.resume += "<i>";
        //println("i = <i>: <btscopeLast>");
        <btscopeLast, btscopes> = getBTInfo(pat, btscopeLast, btscopes);
    }
    return registerBTScope(p, <enterTuple, btscopeLast.resume, btscope.resume>, btscopes);
}

// ---- translate tuple pattern

MuExp translatePat(p:(Pattern) `\<<{Pattern ","}* pats>\>`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])) {
    try {
        constantPat = translatePatternAsConstant(p);
        return muIfExp(muEqual(subjectExp, muCon(constantPat)), trueCont, falseCont);
    } catch _: /* not a constant pattern */;
    
    lpats = [pat | pat <- pats];   //TODO: should be unnnecessary
    elmTypes = [getType(pat) | pat <- lpats];
    patType = atuple(atypeList(elmTypes));
    
    str fuid = topFunctionScope();
    subject = muTmpIValue(nextTmp("tuple_subject"), fuid, subjectType);
  
    //iprintln(btscopes);
    
    body = trueCont;
    for(int i <- reverse(index(lpats))){
        body = translatePat(lpats[i], elmTypes[i], muSubscript(subject, subjectType, muCon(i)), btscopes, body, computeFail(p, lpats, i-1, btscopes, falseCont));
    }
    body = muExists(getEnter(p, btscopes), body);
    code = [ muConInit(subject, subjectExp), muIfElse(muHasTypeAndArity(patType, size(lpats), subject), body, falseCont)];
    return muBlock(code);
}

// ==== list pattern ==========================================================

//  List pattern [L0, L1, ..., Ln]
//                                                 +-----------+
//            +----------------------------------->| falseCont |
//            |                                    +-----------+
//            |
//        +--F--+----------------------------------------------+
//        |     |                                              |
//  ----->| L0  R <------+                                     |
//        |     |        |                                     |
//        |-----+        |                                     |
//        |  |        +--F--+-------------------------------+  |
//        |  |        |     |                               |  |
//        |  +------->| L1  R <------+                      |  |
//        |           |     |        |                      |  |
//        |           +-----+        |                      |  |
//        |           |  |           |                      |  |
//        |           |  |    ...    |                      |  |
//        |           |  |        +--F--+----------------+  |  |
//        |           |  |        |     |                |  |  |
//        |           |  +------->| Ln  R <------+       |  |  |
//        |           |           |     |        |       |  |  |
//        |           |           +-----+        |       |  |  |
//        |           |           |  |           |       |  |  |
//        |           |           |  |     +-----F----+  |  |  |
//        |           |           |  +---->| trueCont |  |  |  |
//        |           |           |        +----------+  |  |  |
//        |           |           +----------------------+  |  |
//        |           |                                     |  |
//        |           +-------------------------------------+  |
//        |                                                    |
//        +----------------------------------------------------+    

// ---- getBTInfo 

BTINFO getBTInfo(p:(Pattern) `[<{Pattern ","}* pats>]`,  BTSCOPE btscope, BTSCOPES btscopes){
    enterList = "<btscope.enter>_LIST";
    BTSCOPE btscopeLast = <enterList, btscope.resume, btscope.resume>;
    for(pat <- pats){
        <btscopeLast, btscopes> = getBTInfoList(pat, btscopeLast, btscopes);
    }
    return registerBTScope(p, <enterList, btscopeLast.resume, btscope.resume>, btscopes);
}

str nameSuffix(str s, Name name){
    sname = "<name>";
    return isWildCard(sname) ? "_<s><nextTmp(sname)>" : "_<s><unescapeAndStandardize(sname)>";
}

str nameSuffix(str s, QualifiedName name){
    sname = "<name>";
    return isWildCard(sname) ? "_<s><nextTmp(sname)>" : "_<s><unescapeAndStandardize(sname)>";
}

BTINFO getBTInfoList(p:(Pattern) `<QualifiedName name>`, BTSCOPE btscope, BTSCOPES btscopes){
    //enter1 = btscope.enter + nameSuffix("VAR", name);
    //resume1 = enter1;
    //fail1 = btscope.\fail;
    //return registerBTScope(p, <enter1, resume1, fail1>, btscopes);
    btscope.enter += nameSuffix("VAR", name);
    return registerBTScope(p, btscope, btscopes);
}    
BTINFO getBTInfoList(p:(Pattern) `<Type tp> <Name name>`, BTSCOPE btscope, BTSCOPES btscopes) {
    //enter1 = btscope.enter + nameSuffix("VAR", name);
    //resume1 = enter1;
    //fail1 = btscope.\fail;
    //return registerBTScope(p, <enter1, resume1, fail1>, btscopes);
    btscope.enter += nameSuffix("VAR", name);
    return registerBTScope(p, btscope, btscopes);
}
    
BTINFO getBTInfoList(p:(Pattern) `<Literal lit>`, BTSCOPE btscope, BTSCOPES btscopes)
    = registerBTScope(p, btscope, btscopes);

BTINFO getBTInfoList(p:(Pattern) `<QualifiedName name>*`, BTSCOPE btscope, BTSCOPES btscopes) {
    enter1 = btscope.enter + nameSuffix("MVAR", name);
    resume1 = enter1;
    fail1 = btscope.resume;
    return registerBTScope(p, <enter1, resume1, fail1>, btscopes);
}

BTINFO getBTInfoList(p:(Pattern) `*<Name name>`, BTSCOPE btscope, BTSCOPES btscopes) {
    enter1 = btscope.enter + nameSuffix("MVAR", name);
    resume1 = enter1;
    fail1 = btscope.resume;
    return registerBTScope(p, <enter1, resume1, fail1>, btscopes);
}

BTINFO getBTInfoList(p:(Pattern) `*<Type tp> <Name name>`, BTSCOPE btscope, BTSCOPES btscopes) {
    enter1 = btscope.enter + nameSuffix("MVAR", name);
    resume1 = enter1;
    fail1 = btscope.resume;
    return registerBTScope(p, <enter1, resume1, fail1>, btscopes);
}

default BTINFO getBTInfoList(Pattern p, BTSCOPE btscope, BTSCOPES btscopes) = getBTInfo(p, btscope, btscopes);  

// ---- translate list pattern

MuExp computeFail(Pattern p, list[Pattern] lpats, int i, btscopes, MuExp falseCont){
//println("computeFail: <p>, <i>, <falseCont>");
    //iprintln(btscopes);
    if(i < 0) return falseCont;
    resume_elm = getResume(lpats[i], btscopes);
    fail_p = getFail(p, btscopes);
    res = resume_elm == fail_p ? falseCont : muFail(resume_elm, comment="computeFail");
//    println("computeFail: <p>, <i>, <falseCont> ==\> <res>");
    return res;
}

MuExp computeFail(Tree p, list[Tree] lpats, int i, btscopes, MuExp falseCont){
    //iprintln(btscopes);
    if(i < 0) return falseCont;
    resume_elm = getResume(lpats[i], btscopes);
    fail_p = getFail(p, btscopes);
    return resume_elm == fail_p ? falseCont : muFail(resume_elm, comment="2");
}

MuExp translatePat(p:(Pattern) `[<{Pattern ","}* pats>]`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])){
    return translateListPat(p, subjectType, subjectExp, btscopes, trueCont, falseCont, subjectAssigned=subjectAssigned, restore=restore);
}

MuExp translateListPat(p:(Pattern) `[<{Pattern ","}* pats>]`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])){
    //iprintln(pats);
    //iprintln(btscopes);
    lpats = [pat | Pattern pat <- pats];   //TODO: should be unnnecessary
    lookahead = computeLookahead(lpats);  
   
    npats = size(lpats);
    elmType = avalue();
    if(alist(tp) := subjectType && tp != avoid()){
    	elmType = tp;
    }
    try {
        constantPat = translatePatternAsConstant(p);
        return muIfExp(muEqual(subjectExp, muCon(constantPat)), trueCont, falseCont);
    } catch _: /* not a constant pattern, go on computing its translation */;
    
    str fuid = topFunctionScope();
    subj = nextTmp("subject");
    subject = muTmpIValue(subj, fuid, alist(elmType));
    cursor = muTmpInt(subj + "_cursor", fuid);
    sublen = muTmpInt(subj + "_len", fuid);
    typecheckNeeded = asubtype(getType(p), subjectType);
  
    //println("enter: <getEnter(p, btscopes)>");
    //println("fail: <getFail(p, btscopes)>");
    //println("resume: <getResume(lpats[-1], btscopes)>");
    //println("computeFail: <computeFail(p, lpats, -1, btscopes, falseCont)>");
    
    //trueCont = muIfElse(muEqualNativeInt(cursor, sublen), trueCont, computeFail(p, lpats, -1, btscopes, falseCont));
    
    trueCont = muIfElse(muEqualNativeInt(cursor, sublen), trueCont, muFail(getResume(lpats[-1], btscopes), comment="list match1"));
    for(i <- reverse(index(lpats))){
        trueCont = translatePatAsListElem(lpats[i], lookahead[i], subjectType, subject, sublen, cursor, i, btscopes, 
                                                    trueCont, 
                                                    computeFail(p, lpats, i-1, btscopes, falseCont),
                                                    restore=restore);
    }
    
    body = trueCont;
    <precise, minSize> = computeListMinSize(lpats);
    
    size_test = precise ? muEqualNativeInt(sublen, muCon(minSize)) : muGreaterEqNativeInt(sublen, muCon(minSize));
    
    block = muBlock([ muConInit(sublen, muSize(subject, alist(elmType))),
                      muIfElse(size_test, 
                               body,
                               falseCont
                              )
                    ]);
    code = muBlock([ *(subjectAssigned ? [muVarInit(subject, subjectExp)] : [muConInit(subject, subjectExp)]),   
                     muVarInit(cursor, muCon(0)), 
                     *(typecheckNeeded ? [muIfElse( muValueIsSubtypeOf(subject, subjectType),
                                                    block
                                                    , falseCont
                                                    )]                                                  
                                       : [ block ])
                     ]);
    //code = muExists(getEnter(p, btscopes), code); // <<
    //iprintln(code);
    return code;
}

//bool isMultiVar(p:(Pattern) `<QualifiedName name>*`) = true;
//bool isMultiVar(p:(Pattern) `*<Type tp> <Name name>`) = true;
//bool isMultiVar(p:(Pattern) `*<Name name>`) = true;
//default bool isMultiVar(Pattern p) = false;
//
//bool isAnonymousMultiVar(p:(Pattern) `_*`) = true;
//bool isAnonymousMultiVar(p:(Pattern) `*<Type tp> _`) = true;
//bool isAnonymousMultiVar(p:(Pattern) `*_`) = true;
//default bool isAnonymousMultiVar(Pattern p) = false;
//
//bool isAnonymousVar(p:(Pattern) `_`) = true;
//bool isAnonymousVar(p:(Pattern) `<Type tp> _`) = true;
//default bool isAnonymousVar(Pattern p) = false;

int nIter(p:(Pattern) `<QualifiedName name>*`) = 0;
int nIter(p:(Pattern) `*<Type tp> <Name name>`) = 0;
int nIter(p:(Pattern) `*<Name name>`) = 0;
default int nIter(Pattern p) { throw "Cannot determine iteration count: <p>"; }

// Lookahead information for a specific position in a list pattern
// nElem = the number of pattern elements following this position that are not multivars
// nMultiVar = the number of multivars following this position

alias Lookahead = tuple[int nElem, int nMultiVar];

list[Lookahead] computeLookahead(list[Pattern] pats){
    nElem = 0;
    nMultiVar = 0;
    rprops = for(Pattern p <- reverse([p | Pattern p <- pats])){
                 append <nElem, nMultiVar>;
                 if(isMultiVar(p)) nMultiVar += 1; else nElem += 1;
             };
    return reverse(rprops);
}

str isLast(Lookahead lookahead) = lookahead.nMultiVar == 0 ? "LAST_" : "";

MuExp translatePatAsListElem(p:(Pattern) `<QualifiedName name>`, Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int posInPat, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([])) {
    if(isWildCard("<name>")){
       return muIfElse(muLessNativeInt(cursor, sublen),
                       muBlock([ muIncNativeInt(cursor, muCon(1)), 
                                 trueCont
                               ]),
                       falseCont);
    }
    var = mkVar(prettyPrintName(name), name@\loc);
    if(isDefinition(name@\loc)){
        return muIfElse(muLessNativeInt(cursor, sublen),
                        muBlock([ muVarInit(var, muSubscript(subject, subjectType, cursor)),
                                  muIncNativeInt(cursor, muCon(1)), 
                                  trueCont
                                ]),
                       falseCont);
                  
    } else {
        return muIfElse(muLessNativeInt(cursor, sublen),
                        muIfElse(muIsInitialized(var), 
                                 muIfElse(muEqual(var, muSubscript(subject, subjectType, cursor)),
                                          muBlock([ muIncNativeInt(cursor, muCon(1)), 
                                                    trueCont
                                                  ]),
                                          falseCont),
                                 muBlock([ muAssign(var, muSubscript(subject, subjectType, cursor)), 
                                           muIncNativeInt(cursor, muCon(1)), 
                                           trueCont
                                         ])),
                        falseCont);
    }
} 

MuExp translatePatAsListElem(p:(Pattern) `<Type tp> <Name name>`, Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int posInPat, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([])) {
   trType = translateType(tp);
   lengthCheck = muLessNativeInt(cursor, sublen);
   check = lengthCheck;
   if(!asubtype(subjectType, alist(trType))){
        check = muAndNativeBool(lengthCheck, muValueIsComparable(muSubscript(subject, subjectType, cursor), trType));
   }
   code = muBlock([]);
   if(isWildCard("<name>")){
      code = muIfElse(check, muBlock([ muIncNativeInt(cursor, muCon(1)), trueCont ]),
                             falseCont);
   } else {
       var = mkVar(prettyPrintName(name), name@\loc);
       var.atype = getType(tp);
       
       code = muIfElse(check, muBlock([ muVarInit(var, muSubscript(subject, subjectType, cursor)), muIncNativeInt(cursor, muCon(1)), trueCont ]),
                              falseCont);
   }
   return code;
   //return muBlock([muExists(getEnter(p, btscopes), code), falseCont]);
 } 

MuExp translatePatAsListElem(p:(Pattern) `<Literal lit>`, Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int posInPat, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([])) {
    if(lit is regExp){
        return translateRegExpLiteral(lit.regExpLiteral, subjectType, muSubscript(subject, subjectType, cursor), btscopes, 
               muBlock([ muIncNativeInt(cursor, muCon(1)), 
                         trueCont
                        ]),
               falseCont);
    }
 
    return muIfElse(muAndNativeBool(muLessNativeInt(cursor, sublen), muEqual(translate(lit), muSubscript(subject, subjectType, cursor))), 
                    muBlock([ muIncNativeInt(cursor, muCon(1)), 
                              trueCont
                            ]),
                    falseCont);
}

// Multi variables

bool isUsed(MuExp _var, MuExp _exp){
    return true; // TODO
    //nm = var.name;
    //return /nm := exp; // In some cases, the type in the var can still be a type var, so only look for the var name;
}

MuExp translatePatAsListElem(p:(Pattern) `<QualifiedName name>*`, Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int posInPat, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([])) {
    return translateMultiVarAsListElem(mkVar(p), isDefinition(name@\loc), lookahead, subjectType, subject, sublen, cursor, posInPat, getEnter(p, btscopes), trueCont, falseCont, restore=restore);
}

MuExp translatePatAsListElem(p:(Pattern) `*<Name name>`, Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int posInPat, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([])) {
    return translateMultiVarAsListElem(mkVar(p), isDefinition(name@\loc), lookahead, subjectType, subject, sublen, cursor, posInPat, getEnter(p, btscopes), trueCont, falseCont, restore=restore);
} 

MuExp translatePatAsListElem(p:(Pattern) `*<Type tp> <Name name>`, Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int posInPat, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([])) {
    return translateMultiVarAsListElem(mkVar(p), isDefinition(name@\loc), lookahead, subjectType, subject, sublen, cursor, posInPat, getEnter(p, btscopes), trueCont, falseCont, restore=restore);
}

MuExp translatePatAsListElem(p:(Pattern) `+<Pattern argument>`, Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int posInPat, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([])) {
    throw "splicePlus pattern";
}

// TODO: optimize last multivar in list pattern

MuExp translateMultiVarAsListElem(MuExp var, bool isDefinition, Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int _posInPat, str enter, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([])) {
    fuid =  topFunctionScope();
    v = nextTmp("<unescapeAndStandardize(var.name)>_<abs(var.pos)>");
    startcursor = muTmpInt(v + "_start", fuid);
    savedcursor = muTmpInt(cursor.name + "_saved", fuid);
    len = muTmpInt(v + "_len", fuid);
    prevlen = muTmpInt(v + "_prevlen", fuid);
    //var.atype = alist(avalue()); // = muVar(prettyPrintName("<name>"), fuid, pos, alist(avalue()));
    varPresent =!isWildCard(var.name);
    needsCheck = !asubtype(subjectType, var.atype);
    
    code = muBlock([]);
    if(lookahead.nMultiVar == 0 && !(varPresent && isUsed(var, trueCont) || needsCheck)){
        code = muBlock([ muConInit(startcursor, cursor), 
                         muConInit(len, muSubNativeInt(muSubNativeInt(sublen, startcursor), muCon(lookahead.nElem))),         
                         muAssign(cursor, muAddNativeInt(startcursor, len)),
                         muExists(enter, trueCont),
                         falseCont
                       ]);
                       
    } else {
        if(isDefinition || !varPresent){
           if(needsCheck && !varPresent){
                var.name = var.name + nextTmp();
           }
           asgCursor = muBlock([ muAssign(cursor, muAddNativeInt(startcursor, len)), trueCont ]);
           if(needsCheck){
                asgCursor =  muIf(muValueIsSubtypeOf(var, var.atype), asgCursor);
           }
           code = muBlock([ muConInit(startcursor, cursor),
                            
                            muForRangeInt(enter, len, 0, 1, muSubNativeInt(muSubNativeInt(sublen, startcursor), muCon(lookahead.nElem)), // changed
                                          muBlock([ restore,
                                                    *( (varPresent && isUsed(var, trueCont) || (needsCheck && !varPresent)) ? [muConInit(var, muSubList(subject, startcursor, len))] : [] ),
                                                    asgCursor
                                                  ]),
                                          falseCont)
                          ]);
        } else {
           code = muBlock([ muConInit(startcursor, cursor),
                            muConInit(len, muSubNativeInt(muSubNativeInt(sublen, startcursor), muCon(lookahead.nElem))),   
                            muConInit(prevlen, muSize(var, subjectType)),
                            muIfElse(muGreaterEqNativeInt(len, prevlen), 
                                 muIfElse(muIsInitialized(var),
                                          muIf(muEqual(var, muSubList(subject, startcursor, prevlen)),
                                               muBlock([ muAssign(cursor, muAddNativeInt(startcursor, prevlen)),
                                                         trueCont
                                                       ])),
                                          muBlock([ muAssign(var, muSubList(subject, startcursor, prevlen)),
                                                    muAssign(cursor, muAddNativeInt(startcursor, prevlen)),
                                                    trueCont
                                                  ]))
                                 , falseCont // <<<<
                                 )
                             , falseCont    // <<<<     
                          ]);
        }
   }
    if(needsCheck){
        code = muIf(muValueIsComparable(subject, var.atype), code);
    }
    return muExists(enter, code); 
}

default MuExp translatePatAsListElem(Pattern p, Lookahead lookahead, AType subjectType, MuExp subject, MuExp sublen, MuExp cursor, int posInPat, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, MuExp restore=muBlock([])) {
    try {
        pcon = muCon(translatePatternAsConstant(p));
        return muIfElse(muAndNativeBool(muLessNativeInt(cursor, sublen), muEqual(muSubscript(subject, subjectType, cursor), pcon)),         
                        muBlock([ muAssign(cursor, muAddNativeInt(cursor, muCon(1))),
                                  trueCont ])
                        , falseCont // <<<<
                        );                            
    } catch: {
        if(p is descendant){
            // Make sure to undo cursor updates
            // -2 is ok since p matches at least 1 element (it cannot be a multivar)
            first = true;
            minus_two_delta = -2 * 1;
            trueCont = top-down-break visit(trueCont) { 
                case muForAll( _, _, _, muDescendantMatchIterator(_, _), _, _): {if(first) { first = false; fail;}} // skip nested descandants
                case muForAny( _, _, _, muDescendantMatchIterator(_, _), _, _): {if(first) { first = false; fail;}} // skip nested descandants
                case muBlock([muIncNativeInt(_, muCon(minus_two_delta)), muFail(_)]): {;}    // prevent cascading insertion of decrement ??TOD -2 => -2*delta
                case mf: muFail(_) => muBlock([muIncNativeInt(cursor, muCon(minus_two_delta)), mf]) 
            };
        }
        
        
  
        return translatePat(p, getListElementType(subjectType), muSubscript(subject, subjectType, cursor), btscopes, 
                               muValueBlock(avalue(), [ muIncNativeInt(cursor, muCon(1)), trueCont]), 
                               falseCont, restore=muBlock([restore,muAssign(cursor, muCon(posInPat))])
                           );
   }
}

// -- variable becomes pattern ---------------------------------------

BTINFO getBTInfo(p:(Pattern) `<Name name> : <Pattern pattern>`,  BTSCOPE btscope, BTSCOPES btscopes) {
    <btscope1, btscopes1> = getBTInfo(pattern, btscope, btscopes);
    return registerBTScope(p, btscope1, btscopes1);
}

MuExp translatePat(p:(Pattern) `<Name name> : <Pattern pattern>`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])) {
    if(subjectAssigned){
         return translatePat(pattern, subjectType, subjectExp, btscopes, trueCont, falseCont, subjectAssigned=false, restore=restore);
    } else {
        var = mkVar(prettyPrintName(name), name@\loc);
        asg = isDefinition(name@\loc) ? muVarInit(var, subjectExp) : muAssign(var, subjectExp);
        return translatePat(pattern, subjectType, subjectExp, btscopes, muValueBlock(avalue(), [ asg, trueCont ]), falseCont, subjectAssigned=subjectAssigned, restore=restore);
    }
}

// -- as type pattern ------------------------------------------------

BTINFO getBTInfo(p:(Pattern) `[ <Type tp> ] <Pattern argument>`,  BTSCOPE btscope, BTSCOPES btscopes) {
    <btscope1, btscopes1> = getBTInfo(argument, btscope, btscopes);
    return registerBTScope(p, btscope1, btscopes1);
}

MuExp translatePat(p:(Pattern) `[ <Type tp> ] <Pattern argument>`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])) =
    muIfElse(muValueIsSubtypeOf(subjectExp, translateType(tp)), 
             translatePat(argument, subjectType, subjectExp, btscopes, trueCont, falseCont, subjectAssigned=subjectAssigned, restore=restore),
             falseCont);

// -- descendant pattern ---------------------------------------------

BTINFO getBTInfo(p:(Pattern) `/ <Pattern pattern>`,  BTSCOPE btscope, BTSCOPES btscopes) {
    enter_desc = "<btscope.enter>_DESC<getLoc(p).offset>";
    <pat_btscope, btscopes1> = getBTInfo(pattern, <enter_desc, enter_desc, enter_desc>, btscopes);
    return registerBTScope(p, <enter_desc, enter_desc, btscope.resume>, btscopes1);
}

//MuExp captureExits(MuExp exp, list[str] entered, str succeedLab, str failLab){
//    return visit(exp){
//        case muExists(str enter1, MuExp exp1) => muExists(enter1, captureExits(exp1, enter1 + entered, succeedLab, failLab))
//        case muAll(str enter1, MuExp exp1) => muAll(enter1, captureExits(exp1, enter1 + entered, succeedLab, failLab))
//        case muSucceed(enter1) => muSucceed(succeedLab) when enter1 notin entered
//        case muFail(enter1) => muFail(failLab) when enter1 notin entered
//    }
//}

MuExp translatePat(p:(Pattern) `/ <Pattern pattern>`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont,  bool subjectAssigned=false, MuExp restore=muBlock([])){
	desc_btscope = btscopes[getLoc(p)];
	concreteMatch = concreteTraversalAllowed(pattern, subjectType);

	reachable_syms = { avalue() };
	reachable_prods = {};
	
    if(optimizeVisit()){
	   tc = getTypesAndConstructorNames(pattern);
       <reachable_syms, reachable_prods>  = getReachableTypes(subjectType, tc.constructors, tc.types, concreteMatch);
    }
    descriptor = descendantDescriptor(concreteMatch, reachable_syms, reachable_prods, getReifiedDefinitions());
    fuid = topFunctionScope();
    elmType = ( avoid() | alub(it, sym) | sym <- reachable_syms );
    elem = muTmpIValue(nextTmp("elem"), fuid, elmType);
  
    patType = getType(pattern);
 
    capturedTrueCont = trueCont; //pattern is anti ? captureExits(trueCont, [], desc_btscope.resume, desc_btscope.resume) : trueCont;
    
    myFail = /*pattern is anti ? capturedTrueCont : */muFail(desc_btscope.resume);
    mySucceed = /*pattern is anti ? muFail(desc_btscope.resume) : */capturedTrueCont;
    
    body = translatePat(pattern, avalue(), elem, btscopes, mySucceed, myFail, restore=restore);

    if(!isVoidAType(patType)){
        body = muIfElse(muValueIsComparable(elem, patType), body, myFail);
    }
    
    code = muForAll(desc_btscope.enter, elem, aset(elmType), muDescendantMatchIterator(subjectExp, descriptor), body,  pattern is anti ? trueCont : falseCont);             
    return code;
}

// Strip start if present

AType stripStart(\start(AType s)) = s;
default AType stripStart(AType s) = s;

// Is  a pattern a concretePattern?
// Note that a callOrTree pattern always requires a visit of the production to inspect labeled fields and is etherefore
// NOT a concrete pattern

bool isConcretePattern(Pattern p) {
    tp = getType(p);
    return isSyntaxType(tp) && !(p is callOrTree); // && Symbol::sort(_) := tp;
}  
	
bool isConcreteType(AType subjectType) =
	(  isSyntaxType(subjectType)
	|| asubtype(subjectType, treeType) && subjectType != treeType
	);
	
bool concreteTraversalAllowed(Pattern pattern, AType subjectType) =
    isConcreteType(subjectType) && isConcretePattern(pattern);

// -- anti pattern ---------------------------------------------------
    
BTINFO getBTInfo(p:(Pattern) `! <Pattern pattern>`,  BTSCOPE btscope, BTSCOPES btscopes) {
    //enterAnti = "<btscope.enter>_ANTI";
    //<pat_btscope, btscopes1> = getBTInfo(pattern, <enterAnti, enterAnti, enterAnti>, btscopes);
    //return registerBTScope(p, <enterAnti, pat_btscope.\fail, btscope.resume>, btscopes1);
    
    <btscope1, btscopes1> = getBTInfo(pattern, btscope, btscopes);
    return registerBTScope(p, btscope1, btscopes1);
}

MuExp translatePat(p:(Pattern) `! <Pattern pattern>`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])){
    return muNot(translatePat(pattern, subjectType, subjectExp, btscopes, trueCont, falseCont, restore=restore));
}
// -- typed variable becomes pattern ---------------------------------

BTINFO getBTInfo(p:(Pattern) `<Type tp> <Name name> : <Pattern pattern>`, BTSCOPE btscope, BTSCOPES btscopes) {
        <btscope1, btscopes1> = getBTInfo(pattern, btscope, btscopes);
        return registerBTScope(p, btscope1, btscopes1);
}

MuExp translatePat(p:(Pattern) `<Type tp> <Name name> : <Pattern pattern>`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])) {
    trType = translateType(tp);
  
    if(isWildCard("<name>")){
         trPat = translatePat(pattern, subjectType, subjectExp, btscopes, trueCont, falseCont, subjectAssigned=subjectAssigned, restore=restore);
         // TODO JURGEN: this static subtype test is not correct, the static subjecttype may be \value, but still this code should check
         // whether or not the value is accidentally the right trType!
         return asubtype(subjectType, trType) ? trPat : muIfElse(muValueIsSubtypeOf(subjectExp, trType), trPat, falseCont);
    }
    str fuid = ""; int pos=0;           // TODO: this keeps type checker happy, why?
    <fuid, pos> = getVariableScope(prettyPrintName(name), name@\loc);
    ppname = prettyPrintName(name);
    var = muVar(ppname, fuid, pos, trType/*[alabel=ppname]*/, patternVariableId());
    trueCont2 = trueCont;
    if(!(subjectExp has name) || subjectExp has name && subjectExp.name != var.name){
        trueCont2 =  muValueBlock(avalue(), [ /*subjectAssigned ? muAssign(var, subjectExp) :*/ muVarInit(var, subjectExp), trueCont ]);
    } 
    trPat = translatePat(pattern, subjectType, subjectExp, btscopes, trueCont2, falseCont, subjectAssigned=subjectAssigned, restore=restore);
    return asubtype(subjectType, trType) ? trPat :  muIfElse(muValueIsSubtypeOf(subjectExp, trType), trPat, falseCont);
}

MuExp translatePat(p:(Pattern) `<Concrete con>`, AType subjectType, MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])) {
  return translateConcretePattern(con, getType(con), subjectExp, btscopes, trueCont, falseCont, restore=restore);
} 

// -- default rule for pattern ---------------------------------------

default BTINFO getBTInfo(Pattern p, BTSCOPE btscope, BTSCOPES btscopes)
    = registerBTScope(p, btscope, btscopes);

default MuExp translatePat(Pattern p, AType subjectType,  MuExp subjectExp, BTSCOPES btscopes, MuExp trueCont, MuExp falseCont, bool subjectAssigned=false, MuExp restore=muBlock([])) { 
    //iprintln(p);
    return muValueBlock(avalue(), [muThrow(muCon("could not translate pattern <p>: <p@\loc>"), p@\loc)]); 
}


/*****************************************************************************/
/*                      Constant Patterns                                    */
/* - try to translate a pattern to a constant (and throw an exception when   */
/*   this is impossible                                                      */
/*****************************************************************************/

value getLiteralValue((Literal) `<Literal s>`) =  readTextValueString("<s>") when isConstant(s);

bool isConstant(StringLiteral l) = l is nonInterpolated;
bool isConstant(LocationLiteral l) = l.protocolPart is nonInterpolated && l.pathPart is nonInterpolated;
bool isConstant(RegExpLiteral _)  = false;
default bool isConstant(Literal _) = true;
 
value translatePatternAsConstant(p:(Pattern) `<Literal lit>`) = getLiteralValue(lit) when isConstant(lit);

value translatePatternAsConstant(p:(Pattern) `<Pattern expression> ( <{Pattern ","}* arguments> <KeywordArguments[Pattern] keywordArguments> )`) {
  if(!isEmpty("<keywordArguments>")) throw "Not a constant pattern: <p>";
  if(isADTAType(getType(p))) throw "ADT pattern not considered constant: <p>";
  node_name = expression is literal && expression.literal is string ? "<expression>"[1..-1] : "<expression>";
  return makeNode(node_name, [ translatePatternAsConstant(pat) | Pattern pat <- arguments ]);
}
    
value translatePatternAsConstant(p:(Pattern) `{<{Pattern ","}* pats>}`) {
    res = { translatePatternAsConstant(pat) | Pattern pat <- pats };
    return res;
}

value translatePatternAsConstant(p:(Pattern) `[<{Pattern ","}* pats>]`) = [ translatePatternAsConstant(pat) | Pattern pat <- pats ];

value translatePatternAsConstant(p:(Pattern) `\<<{Pattern ","}* pats>\>`) {
  lpats = [ pat | pat <- pats]; // TODO
  return ( <translatePatternAsConstant(lpats[0])> | it + <translatePatternAsConstant(lpats[i])> | i <- [1 .. size(lpats)] );
}

//value translatePatternAsConstant(list[Pattern] pats)
//    = [ translatePatternAsConstant(pat) | Pattern pat <- pats ];

default value translatePatternAsConstant(Pattern p){
  throw "Not a constant pattern: <p>";
}

value translateConcretePatternAsConstant(p:appl(Production::prod(sort(str A),[Symbol::lit(str litA)],{}),[appl(prod(Symbol::lit(litA),[\char-class([CharRange::range(int charA,charA)])], {}),[Tree::char(charA)])])) {
    return p;
}
value translateConcretePatternAsConstant(p:Production::prod(Symbol::lit(str charA),[\char-class([CharRange::range(int charA,charA)])],{})) {
    return p;
}
value translateConcretePatternAsConstant(p:appl(Production::prod(Symbol::sort(str A), [Symbol::lit(str Alit)], {}),[appl(prod(lit(Alit), [\char-class([CharRange::range(int charA, charA)])], {}), [Tree::char(charA)])])) {
    return p; 
}

value translateConcretePatternAsConstant(p:appl(Production::regular(Symbol::\iter-seps(sort(str A),[Symbol::layouts(str Alayout)])), [arg])){
    translateConcretePatternAsConstant(arg);
    return p;
}

default value translateConcretePatternAsConstant(Pattern p){
    throw "Not a constant pattern: <p>";
}