---
title: "module util::Reflective"
id: Reflective
slug: /Library/util/Reflective
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-0.41.0-RC29</div>

#### Usage

```rascal
import util::Reflective;
```

#### Dependencies
```rascal
import IO;
import List;
import ParseTree;
import String;
import util::FileSystem;
import Message;
import lang::rascal::\syntax::Rascal;
import lang::manifest::IO;
```


## function getLineSeparator {#util-Reflective-getLineSeparator}

Returns the system-dependent line separator string

```rascal
str getLineSeparator()
```

## function evalCommands {#util-Reflective-evalCommands}

```rascal
lrel[str result, str out, str err] evalCommands(list[str] command, loc org)
```

## function parseModule {#util-Reflective-parseModule}

Just parse a module at a given location without any furter processing (i.e., fragment parsing) or side-effects (e.g. module loading)

```rascal
lang::rascal::\syntax::Rascal::Module parseModule(loc location)
```

## function parseModuleWithSpaces {#util-Reflective-parseModuleWithSpaces}

Parse a module (including surounding spaces) at a given location without any furter processing (i.e., fragment parsing) or side-effects (e.g. module loading)

```rascal
start[Module] parseModuleWithSpaces(loc location)
```

## data RascalConfigMode {#util-Reflective-RascalConfigMode}

```rascal
data RascalConfigMode  
     = compiler()
     | interpreter()
     ;
```

## data PathConfig {#util-Reflective-PathConfig}
General configuration (via path references) of a compiler or interpreter.

```rascal
data PathConfig  
     = pathConfig(
        list[loc] srcs = [],  
        list[loc] ignores = [],  
        loc bin = |unknown:///|,
        loc generatedSources = |unknown:///|,
        list[loc] libs = [],          
        list[Message] messages = []
    )
     ;
```


A PathConfig is the result of dependency resolution and other configuration steps. Typically,
IDEs produce the information to fill a PathConfig, such that language tools can consume it
transparantly. A PathConfig is also a log of the configuration process. 

* `srcs` list of root directories to search for source files; to interpret or to compile.
* `ignores` list of directories and files to not compile or not interpret (these are typically subtracted from the `srcs` tree, or skipped when the compiler arives there.)
* `bin` is the target root directory for the output of a compiler. Typically this directory would be linked into a zip or a jar or an executable file later.
* `libs` is a list of binary dependency files (typically jar files or target folders) on other projects, for checking and linking purposes.
* `generatedSources` is where generated (intermediate) source code that has to be compiled further is located.
* `messages` is a list of info, warning and error messages informing end-users about the quality of the configuration process. Typically missing dependencies would be reported here, and clashing versions.

## data RascalManifest {#util-Reflective-RascalManifest}

```rascal
data RascalManifest  
     = rascalManifest(
      str \Project-Name = "Project",
      str \Main-Module = "Plugin",
      str \Main-Function = "main", 
      list[str] Source = ["src"],
      str Bin = "bin",
      list[str] \Required-Libraries = [],
      list[str] \Required-Dependencies = []
    )
     ;
```

## data JavaBundleManifest {#util-Reflective-JavaBundleManifest}

```rascal
data JavaBundleManifest  
     = javaManifest(
      str \Manifest-Version = "",
      str \Bundle-SymbolicName = "",
      str \Bundle-RequiredExecutionEnvironment = "JavaSE-1.8",
      list[str] \Require-Bundle = [],
      str \Bundle-Version = "0.0.0.qualifier",
      list[str] \Export-Package = [],
      str \Bundle-Vendor = "",
      str \Bundle-Name = "",
      list[str] \Bundle-ClassPath = [],
      list[str] \Import-Package = [] 
    )
     ;
```

## function resolveProjectOnClasspath {#util-Reflective-resolveProjectOnClasspath}

```rascal
loc resolveProjectOnClasspath(str projectName)
```

## function resolvedCurrentRascalJar {#util-Reflective-resolvedCurrentRascalJar}

```rascal
loc resolvedCurrentRascalJar()
```

## function metafile {#util-Reflective-metafile}

```rascal
loc metafile(loc l)
```

## function applyManifests {#util-Reflective-applyManifests}

Converts a PathConfig and replaces all references to roots of projects or bundles
  by the folders which are nested under these roots as configured in their respective
  META-INF/RASCAL.MF files.

```rascal
PathConfig applyManifests(PathConfig cfg)
```

## function makeFileName {#util-Reflective-makeFileName}

```rascal
str makeFileName(str qualifiedModuleName, str extension = "rsc")
```

## function getSearchPathLoc {#util-Reflective-getSearchPathLoc}

```rascal
loc getSearchPathLoc(str filePath, PathConfig pcfg)
```

## function getModuleLocation {#util-Reflective-getModuleLocation}

Get the location of a named module, search for `src` in srcs and `tpl` in libs

```rascal
loc getModuleLocation(str qualifiedModuleName,  PathConfig pcfg)
```

## function splitFileExtension {#util-Reflective-splitFileExtension}

```rascal
tuple[str,str] splitFileExtension(str path)
```

## function commonSuffix {#util-Reflective-commonSuffix}

Determine length of common suffix of list of strings

```rascal
int commonSuffix(list[str] dir, list[str] m)
```

## function commonPrefix {#util-Reflective-commonPrefix}

Determine length of common prefix of list of strings

```rascal
int commonPrefix(list[str] rdir, list[str] rm)
```

## function getModuleName {#util-Reflective-getModuleName}

Find the module name corresponding to a given module location via its (src or tpl) location

```rascal
str getModuleName(loc moduleLoc,  PathConfig pcfg)
```

## function getDerivedReadLoc {#util-Reflective-getDerivedReadLoc}

Derive a location from a given module name for reading

```rascal
tuple[bool, loc] getDerivedReadLoc(str qualifiedModuleName, str extension, PathConfig pcfg, set[str] srcExtensions = {"rsc", "mu"}, str rootDir = "")
```


Given a module name, a file name extension, and a PathConfig,
a path name is constructed from the module name + extension.

If a file F with this path exists in one of the directories in the PathConfig,
then the pair <true, F> is returned. Otherwise <false, some error location> is returned.

For a source extension (typically "rsc" or "mu" but this can be configured) srcs is searched, otherwise binPath + libs.

#### Examples



```rascal-shell 
rascal>import util::Reflective;
ok
rascal>getDerivedReadLoc("List", "rsc", pathConfig());
tuple[bool,loc]: <false,|error:///|>
rascal>getDerivedReadLoc("experiments::Compiler::Compile", "rvm", pathConfig());
tuple[bool,loc]: <false,|error:///|>
rascal>getDerivedReadLoc("experiments::Compiler::muRascal2RVM::Library", "mu", pathConfig());
tuple[bool,loc]: <false,|error:///|>
```

#### Benefits


This function is useful for type checking and compilation tasks, when derived information related to source modules has to be read
from locations in different, configurable, directories.

## function getDerivedWriteLoc {#util-Reflective-getDerivedWriteLoc}

Derive a location from a given module name for writing

```rascal
loc getDerivedWriteLoc(str qualifiedModuleName, str extension, PathConfig pcfg, set[str] srcExtensions = {"rsc", "mu"}, str rootDir = "")
```


Given a module name, a file name extension, and a PathConfig,
a path name is constructed from the module name + extension.

For source modules, a writable location cannot be derived.
For other modules, a location for this path in bin will be returned.

#### Examples



```rascal-shell 
rascal>import util::Reflective;
ok
rascal>getDerivedWriteLoc("List", "rvm", pathConfig());
loc: |unknown:///rascal/$List.rvm|
rascal>getDerivedWriteLoc("experiments::Compiler::Compile", "rvm", pathConfig());
loc: |unknown:///rascal/experiments/Compiler/$Compile.rvm|
```


```rascal-shell ,error
rascal>getDerivedWriteLoc("experiments::Compiler::muRascal2RVM::Library", "rsc", pathConfig());
|prompt:///|(0,18,<1,0>,<1,18>): Undeclared variable: getDerivedWriteLoc
Advice: |https://www.rascal-mpl.org/docs/Rascal/Errors/CompileTimeErrors/UndeclaredVariable|
```

#### Benefits


This function is useful for type checking and compilation tasks, when derived information related to source modules has to be written
to locations in separate, configurable, directories.

## function getProjectPathConfig {#util-Reflective-getProjectPathConfig}

```rascal
PathConfig getProjectPathConfig(loc projectRoot, RascalConfigMode mode = compiler())
```

## function inCompiledMode {#util-Reflective-inCompiledMode}

Is the current Rascal code executed by the compiler or the interpreter?

```rascal
bool inCompiledMode()
```

## function diff {#util-Reflective-diff}

Give a textual diff between two values.

```rascal
str diff(value old, value new)
```

## function watch {#util-Reflective-watch}

Watch value val: 
- running in interpreted mode: write val to a file, 
- running in compiled mode: compare val with previously written value

```rascal
&T watch(type[&T] tp, &T val, str name)

&T watch(type[&T] tp, &T val, str name, value suffix)
```

## function getFingerprint {#util-Reflective-getFingerprint}

Compute a fingerprint of a value for the benefit of the compiler and the compiler runtime

```rascal
int getFingerprint(value val, bool concretePatterns)
```

## function getFingerprint {#util-Reflective-getFingerprint}

Compute a fingerprint of a value and arity modifier for the benefit of the compiler and the compiler runtime

```rascal
int getFingerprint(value val, int arity, bool concretePatterns)
```

## function getFingerprintNode {#util-Reflective-getFingerprintNode}

Compute a fingerprint of a complete node for the benefit of the compiler and the compiler runtime

```rascal
int getFingerprintNode(node nd)
```

## function getHashCode {#util-Reflective-getHashCode}

Get the internal hash code of a value. For the benefit of debugging the Rascal implementation.

```rascal
int getHashCode(value v)
```


This function is useless for Rascal programmer's as it is a part of the under-the-hood implementation of values.
You can use a value directly as a lookup key. The internal data-structures probably use this hashCode for
optimal lookups in `O(log(size))`. 

We use this function to diagnose possible performance issues caused by hash collisions.

## function throwNullPointerException {#util-Reflective-throwNullPointerException}

Throw a raw Java NullPointerException, to help simulate an unexpected exception in test scenarios

```rascal
void throwNullPointerException()
```

## function getRascalReservedIdentifiers {#util-Reflective-getRascalReservedIdentifiers}

Return a list of all Rascal reserved identifiers (a.k.a. keywords)

```rascal
set[str] getRascalReservedIdentifiers()
```

## function getRascalVersion {#util-Reflective-getRascalVersion}

```rascal
str getRascalVersion()
```

## function newRascalProject {#util-Reflective-newRascalProject}

Create a folder structure for an empty Rascal project with Maven support

```rascal
void newRascalProject(loc folder, str group="org.rascalmpl", str version="0.1.0-SNAPSHOT")
```

## function pomFile {#util-Reflective-pomFile}

```rascal
loc pomFile(loc folder)
```

## function emptyModule {#util-Reflective-emptyModule}

```rascal
str emptyModule()
```

## function rascalMF {#util-Reflective-rascalMF}

```rascal
str rascalMF(str name)
```

## function newRascalMfFile {#util-Reflective-newRascalMfFile}

Create a new META-INF/RASCAL.MF file.

```rascal
void newRascalMfFile(loc folder, str name=folder.file)
```


The `folder` parameter should point to the root of a project folder.
The name of the project will be derived from the name of that folder
and a META-INF/RASCAL.MF file will be generated and written.

The folder is created if it does not exist already.

## function newRascalPomFile {#util-Reflective-newRascalPomFile}

Create a new pom.xml for a Rascal project

```rascal
void newRascalPomFile(loc folder, str name=folder.file, str group="org.rascalmpl", str version="0.1.0-SNAPSHOT")
```


The `folder` parameter should point to the root of a project folder.
The name of the project will be derived from the name of that folder
and a pom.xml file will be generated and written.  

The folder is created if it does not exist already.

## function pomXml {#util-Reflective-pomXml}

```rascal
str pomXml(str name, str group, str version)
```

