---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-0.41.0-RC29</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://94235faa-59b6-4d5e-836f-2dfc89989fe7|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <7,49,|uuid://3cc867f2-3497-46b3-8b10-c47ab188cabd|>,
  <6,36,|uuid://3c78b717-ffe7-4093-b44a-19a2389a3c7f|>,
  <3,9,|uuid://68646f4d-625d-43e9-90f7-cbdbe47fedd7|>,
  <2,4,|uuid://69f37376-795a-4829-b931-fecff19c2220|>,
  <4,16,|uuid://269e851f-38e7-46e9-989a-f6995f79cc26|>,
  <9,81,|uuid://0dd0c2f8-3a71-4fc8-acf7-5a321011b332|>,
  <1,1,|uuid://c9131b9c-1b61-4a02-a815-87830d6bd77d|>,
  <10,100,|uuid://9d099042-9ae8-430b-b3d3-bd397f936d73|>,
  <5,25,|uuid://661947c7-854d-40e7-aa1b-34b23b935bbe|>,
  <8,64,|uuid://f86ead5d-9eff-44f8-b87c-06b403d1f07e|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://3668c713-90ef-4c67-bbba-56ea6ba57da3|,
  <2,4>:|uuid://30f713d4-88c9-4056-b7bf-a418f45bc70d|,
  <7,49>:|uuid://43025b80-5e06-48d7-b92b-579afa8c3a65|,
  <9,81>:|uuid://9a4151af-d7a6-43e1-a320-7e8b0cf39959|,
  <8,64>:|uuid://e03e253c-6fa5-4010-8d7f-98be623a7c7c|,
  <5,25>:|uuid://f6e8aa5c-bcc8-4241-838c-e4812e4a8113|,
  <4,16>:|uuid://d49e8b8c-407d-4e5e-8b13-bb61a8946fa3|,
  <1,1>:|uuid://3f58071a-563d-44bd-bb28-1e42cf32b0a7|,
  <10,100>:|uuid://42659258-7254-4629-b374-fea58c7bfc32|,
  <3,9>:|uuid://7ad8ee2e-6894-41a1-aaa6-3c1c78e307ed|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

