---
title: "module util::ParseErrorRecovery"
id: ParseErrorRecovery
slug: /Library/util/ParseErrorRecovery
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-0.41.0-RC29</div>

#### Usage

```rascal
import util::ParseErrorRecovery;
```

#### Dependencies
```rascal
import ParseTree;
import String;
```


## function hasParseErrors {#util-ParseErrorRecovery-hasParseErrors}

Check if a parse tree contains any error nodes, the result of error recovery.

Library functions for handling parse trees with error nodes.

```rascal
bool hasParseErrors(Tree tree)
```


When parse functions are called with `errorRecovery=true`, the resulting parse trees can contain error and skipped nodes. For an in-depth description of these nodes see the `ParseTree` module.
Functions in this module can be used to detect, isolate, and analyze these error nodes.
Note that these are fairly low-level functions. Better high-level support for error trees will be added to Rascal and its standard library in the future.
As such, this code should be considered experimental and used with care.

## function findAllParseErrors {#util-ParseErrorRecovery-findAllParseErrors}

Find all error productions in a parse tree.
Note that children of an error tree can contain errors themselves.
The list of errors returned by this methood is created by an outermost visit of the parse tree so if an error tree contains other errors the outermost tree is
returned first.
Often error trees are highly ambiguous and can contain a lot of error trees. This function is primarily used to analyze small examples as calling this function
on a tree with many errors will result in long runtimes and out-of-memory errors.


```rascal
list[Tree] findAllParseErrors(Tree tree)
```

## function getErrorSymbol {#util-ParseErrorRecovery-getErrorSymbol}

Get the symbol (sort) of the failing production

```rascal
Symbol getErrorSymbol(appl(error(Symbol sym, _, _), _))
```

## function getErrorProduction {#util-ParseErrorRecovery-getErrorProduction}

Get the production that failed

```rascal
Production getErrorProduction(appl(error(_, Production prod, _), _))
```

## function getErrorDot {#util-ParseErrorRecovery-getErrorDot}

Get the dot (position in the production) of the failing element in a production

```rascal
int getErrorDot(appl(error(_, _, int dot), _))
```

## function getSkipped {#util-ParseErrorRecovery-getSkipped}

Get the skipped tree

```rascal
Tree getSkipped(appl(error(_, _, _), [*_, skip:appl(skipped(_), _)]))
```

## function getErrorText {#util-ParseErrorRecovery-getErrorText}

Get the text that failed to parse. This is only the text of the part that has been skipped to be able to continue parsing.
If you want the text of the whole error tree, you can just use string interpolation: "<error>".


```rascal
str getErrorText(appl(error(_, _, _), [*_, appl(skipped(_), chars)]))
```

## function disambiguateParseErrors {#util-ParseErrorRecovery-disambiguateParseErrors}

Error recovery often produces ambiguous trees where errors can be recovered in multiple ways.
This filter removes error trees until no ambiguities caused by error recovery are left.
Note that regular ambiguous trees remain in the parse forest unless `allowAmbiguity` is set to false in which case an error is thrown.
This method uses simple and somewhat arbitrary heuristics, so its usefulness is limited.


```rascal
&T<:Tree disambiguateParseErrors(&T<:Tree t, bool allowAmbiguity=true)
```

## function findBestParseErrors {#util-ParseErrorRecovery-findBestParseErrors}

Disambiguate the error ambiguities in a tree and return the list of remaining errors. 
The list is created by an outermost visit of the parse tree so if an error tree contains other errors the outermost tree is returned first.
Error disambiguation is based on heuristics and are therefore unsuitable for many use-cases. We primarily use this functionality
for testing and for presenting recovered errors in VSCode.


```rascal
list[Tree] findBestParseErrors(Tree tree)
```

## function createParseErrorFilter {#util-ParseErrorRecovery-createParseErrorFilter}

Create a parse filter based on `disambiguateErrors` with or without `allowAmbiguity`.

```rascal
Tree(Tree) createParseErrorFilter(bool allowAmbiguity)
```

