---
title: "module util::Sampling"
id: Sampling
slug: /Library/util/Sampling
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-0.41.0-RC29</div>

Utilities to randomly select smaller datasets from larger datasets
#### Usage

```rascal
import util::Sampling;
```

#### Dependencies
```rascal
import util::Math;
import Map;
import List;
import Set;
```

#### Description


Sampling is important when the analysis algorithms do not scale to the size of 
the original corpus, or when you need to train an analysis on a representative
set without overfitting on the entire corpus. These sampling functions all
assume that a uniformly random selection is required.


## function sample {#util-Sampling-sample}

Reduce the arity of a set by selecting a uniformly distributed sample.

```rascal
set[&T] sample(set[&T] corpus, int target)
```


A uniform subset is computed by iterating over the set and skipping every element
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new set of
expected `target` size, but most probably a little smaller or larger.

#### Examples



```rascal-shell 
rascal>import util::Sampling;
ok
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"a","b","c","e","g","j","k"}
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"a","c","k"}
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"a","k"}
```

## function sample {#util-Sampling-sample}

Reduce the length of a list by selecting a uniformly distributed sample.

```rascal
list[&T] sample(list[&T] corpus, int target)
```


The random selection of elements does not change their initial order in the list.
A uniform sublist is computed by iterating over the list and skipping every element
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new list of
expected `target` size, but most probably a little smaller or larger.

#### Examples



```rascal-shell 
rascal>import util::Sampling;
ok
rascal>sample([1..1000], 30)
list[int]: [18,47,71,118,124,138,197,299,358,472,546,595,600,615,815,866,878,883,894,930,949,968,996]
rascal>sample([1..1000], 30)
list[int]: [6,28,43,125,139,154,168,183,187,245,251,300,465,471,486,532,566,648,658,670,688,711,718,757,783,838,843,880,889,892,921,925,947,976]
rascal>sample([1..1000], 30)
list[int]: [48,225,247,264,297,327,348,526,527,530,557,595,596,617,650,680,686,800,811,825,887,956,966,971,996]
```

## function sample {#util-Sampling-sample}

Reduce the size of a map by selecting a uniformly distributed sample.

```rascal
map[&T,&U] sample(map[&T,&U] corpus, int target)
```


A uniform submap is computed by iterating over the map's keys and skipping every key
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new map of
expected `target` size, but most probably a little smaller or larger.

