---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-0.41.0-RC29</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://825e49be-6930-4b3a-8cf2-bad52d51281d|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <2,4,|uuid://bc952bbf-69c6-420e-aa50-7459d3504060|>,
  <8,64,|uuid://42faffcf-7f3a-46fc-b5f7-5c9b88c2848b|>,
  <5,25,|uuid://3900b80d-02e7-4f9c-b31b-536900e80db1|>,
  <3,9,|uuid://23345293-76e8-4d48-a271-b1863c738cb0|>,
  <1,1,|uuid://e56086f5-6c66-48f8-bfdc-6e01cf524f7e|>,
  <4,16,|uuid://2632b7df-892b-4546-a879-3c0a0887dd46|>,
  <10,100,|uuid://a9a4b582-35db-4db6-aaf0-5aee29824b9b|>,
  <6,36,|uuid://bdceab66-1b2a-44f7-8c5f-f1513453d568|>,
  <9,81,|uuid://847d8a48-ccdf-460b-878d-438b49b3d9f8|>,
  <7,49,|uuid://8fa75c8a-b279-49f9-ac4d-b9cbefbb49de|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://70c83e48-27e1-4dc6-9ace-a9f316399769|,
  <2,4>:|uuid://b741910a-ad9b-4f86-8671-18222a25bc10|,
  <7,49>:|uuid://ef84fc64-52db-466b-a3f2-7b5f4357933f|,
  <9,81>:|uuid://3bac2ab0-b47b-460e-9541-b755cebf4958|,
  <8,64>:|uuid://8a226cdd-7cb4-4c5c-864f-06d4cc4a3c8b|,
  <5,25>:|uuid://75e3eabf-93dd-4055-a5fe-2a0e6305dad4|,
  <4,16>:|uuid://7047f70f-8cfe-4272-b586-22aecfaeb3b7|,
  <1,1>:|uuid://45a9a96d-fba9-4d15-99ab-4388db5fad27|,
  <10,100>:|uuid://fc721c94-277c-434d-b98c-5e38ba97029e|,
  <3,9>:|uuid://6bd80436-ad3d-4333-b696-1133d34b6bc0|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

