---
title: "module vis::Graphs"
id: Graphs
slug: /Library/vis/Graphs
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-0.41.0-RC29</div>

Simple data visualization using graphs; based on cytoscape.js
#### Usage

```rascal
import vis::Graphs;
```

#### Dependencies
```rascal
import lang::html::IO;
import lang::html::AST;
import util::IDEServices;
import Content;
import ValueIO;
import Set;
```

#### Description


This modules provides a simple API to create graph visuals for Rascal
(relational) data, based on [Cytoscape.js](https://js.cytoscape.org/). 

This module is quite new and may undergo some tweaks in the coming time.

#### Benefits


* Easy to use for basic graph layouts.


## data CytoGraphConfig {#vis-Graphs-CytoGraphConfig}
Optional configuration attributes for graph style and graph layout

```rascal
data CytoGraphConfig  
     = cytoGraphConfig(
    str title="Graph", 

    NodeLinker[&T]     nodeLinker     = defaultNodeLinker, 
    NodeLabeler[&T]    nodeLabeler    = defaultNodeLabeler,  
    NodeClassifier[&T] nodeClassifier = defaultNodeClassifier, 
    EdgeLabeler[&T]    edgeLabeler    = defaultEdgeLabeler, 
    EdgeClassifier[&T] edgeClassifier = defaultEdgeClassifier,
    
    CytoLayout \layout  = defaultCoseLayout(), 

    CytoStyle nodeStyle      = defaultNodeStyle(), 
    CytoStyle edgeStyle      = defaultEdgeStyle(), 
    list[CytoStyleOf] styles = []
)
     ;
```


These configuration options are used to map input graph data to layout properties
and style properties. 

* title - does what it says
* nodeLinker - makes nodes clickable by providing an editor location
* nodeLabeler - allows simplification or elaboration on node labels beyond their identity string
* nodeClassifier - labels nodes with classes in order to later select them for specific styling
* edgeLabeler - allows simplification or elaboration on edge labels 
* layout - defines and configured the graph layout algorithm
* nodeStyle - defines the default style for all nodes
* edgeStyle - defines the default style for all edges
* style - collects specific styles for specific [Cyto Selector](../..//Library/vis/Graphs.md#vis-Graphs-CytoSelector) edge/node selectors using [Cyto Style Of](../..//Library/vis/Graphs.md#vis-Graphs-CytoStyleOf) tuples.

Typically the functions passed into this configuration are closures that capture and use the original
input data to find out about where to link and how to classify. The `&T` parameter reflects the type of
the original input `Graph[&T]`; so that is the type of the nodes. Often this would be `loc` or `str`.

#### Examples



Let's experiment with a number of styling parameters based on the shape of a graph:

```rascal-shell 
rascal>import vis::Graphs;
ok
```
let's play with the geneology of the "Simpsons"
```rascal-shell
rascal>g = {
|1 >>>>    <"Abraham Simpson", "Homer Simpson">,
|2 >>>>    <"Mona Simpson", "Homer Simpson">,
|3 >>>>    <"Homer Simpson", "Bart Simpson">,
|4 >>>>    <"Homer Simpson", "Lisa Simpson">,
|5 >>>>    <"Homer Simpson", "Maggie Simpson">,
|6 >>>>    <"Marge Simpson", "Bart Simpson">,
|7 >>>>    <"Marge Simpson", "Lisa Simpson">,
|8 >>>>    <"Marge Simpson", "Maggie Simpson">,
|9 >>>>    <"Bart Simpson", "Rod Flanders">,
|10 >>>>    <"Bart Simpson", "Todd Flanders">,
|11 >>>    <"Lisa Simpson", "Bart Simpson">,
|12 >>>    <"Abraham Simpson", "Patty Bouvier">,
|13 >>>    <"Abraham Simpson", "Selma Bouvier">,
|14 >>>    <"Mona Simpson", "Patty Bouvier">,
|15 >>>    <"Mona Simpson", "Selma Bouvier">
|16 >>>};
rel[str,str]: {
  <"Lisa Simpson","Bart Simpson">,
  <"Homer Simpson","Lisa Simpson">,
  <"Homer Simpson","Maggie Simpson">,
  <"Homer Simpson","Bart Simpson">,
  <"Mona Simpson","Homer Simpson">,
  <"Mona Simpson","Patty Bouvier">,
  <"Mona Simpson","Selma Bouvier">,
  <"Marge Simpson","Lisa Simpson">,
  <"Marge Simpson","Maggie Simpson">,
  <"Marge Simpson","Bart Simpson">,
  <"Bart Simpson","Rod Flanders">,
  <"Bart Simpson","Todd Flanders">,
  <"Abraham Simpson","Homer Simpson">,
  <"Abraham Simpson","Patty Bouvier">,
  <"Abraham Simpson","Selma Bouvier">
}
```
visualizing this without styling:
```rascal-shell
rascal>graph(g);
🔗 Serving 'Graph' at |http://localhost:9051/|
```
to style nodes, let's select some special nodes and "classify" them first. We reuse some generic graph analysis tools.
```rascal-shell
rascal>import analysis::graphs::Graph;
ok
rascal>list[str] nodeClassifier(str simpson) = [
|1 >>>>  *["top" | simpson in top(g)],
|2 >>>>  *["bottom" | simpson in bottom(g)]
|3 >>>>];
list[str] (str): function(|prompt:///|(0,113,<1,0>,<4,2>))
```
once classified, we can style each node according to their assigned classes. Nodes can be in more than one class.
```rascal-shell
rascal>styles = [
|1 >>>>    cytoStyleOf( 
|2 >>>>        selector=or([\node(className("top")),\node(className("bottom"))]),
|3 >>>>        style=defaultNodeStyle()[shape=CytoNodeShape::diamond()]
|4 >>>>    )
|5 >>>>];
list[CytoStyleOf]: [cytoStyleOf(
    style=cytoNodeStyle(
      shape=diamond(),
      color="white",
      background-color="blue",
      width="label",
      opacity="1",
      text-valign=center(),
      padding="10pt",
      visibility="visible",
      label="data(label)",
      font-weight=bold(),
      text-halign=center(),
      font-size="20pt"),
    selector=or([
        and([
            node(),
            className("top")
          ]),
        and([
            node(),
            className("bottom")
          ])
      ]))]
```
we pick a sensible layout
```rascal-shell
rascal>lyt = defaultDagreLayout();
CytoLayout: dagreLayout(
  animate=false,
  name=dagre(),
  spacingFactor=1,
  ranker=\network-simplex())
```
we wrap the styling information into a configuration wrapper:
```rascal-shell
rascal>cfg = cytoGraphConfig(nodeClassifier=nodeClassifier, styles=styles, \layout=lyt);
CytoGraphConfig: cytoGraphConfig(
  styles=[cytoStyleOf(
      style=cytoNodeStyle(
        shape=diamond(),
        color="white",
        background-color="blue",
        width="label",
        opacity="1",
        text-valign=center(),
        padding="10pt",
        visibility="visible",
        label="data(label)",
        font-weight=bold(),
        text-halign=center(),
        font-size="20pt"),
      selector=or([
          and([
              node(),
              className("top")
            ]),
          and([
              node(),
              className("bottom")
            ])
        ]))],
  nodeClassifier=function(|prompt:///|(0,113,<1,0>,<4,2>)),
  layout=dagreLayout(
    animate=false,
    name=dagre(),
    spacingFactor=1,
    ranker=\network-simplex()))
```
and now we see the effect:
```rascal-shell
rascal>graph(g, cfg=cfg)
🔗 Serving 'Graph' at |http://localhost:9051/|
```
now let's style some edges:
```rascal-shell
rascal>list[str] edgeClassifier(str from, str to) = ["grandparent" | <from, to > in g o g];
list[str] (str, str): function(|prompt:///|(0,84,<1,0>,<1,84>))
```
add another styling element
```rascal-shell
rascal>styles += [
|1 >>>>    cytoStyleOf( 
|2 >>>>        selector=edge(className("grandparent")),
|3 >>>>        style=defaultEdgeStyle()[\line-style="dashed"]
|4 >>>>    )
|5 >>>>];
list[CytoStyleOf]: [
  cytoStyleOf(
    style=cytoNodeStyle(
      shape=diamond(),
      color="white",
      background-color="blue",
      width="label",
      opacity="1",
      text-valign=center(),
      padding="10pt",
      visibility="visible",
      label="data(label)",
      font-weight=bold(),
      text-halign=center(),
      font-size="20pt"),
    selector=or([
        and([
            node(),
            className("top")
          ]),
        and([
            node(),
            className("bottom")
          ])
      ])),
  cytoStyleOf(
    style=cytoEdgeStyle(
      target-arrow-shape=triangle(),
      width=3,
      line-color="black",
      source-arrow-color="black",
      curve-style=\unbundled-bezier(),
      label="data(label)",
      line-style="dashed",
      target-arrow-color="black",
      color="red",
      opacity="1",
      source-arrow-shape=none(),
      visibility="visible",
      line-opacity="1"),
    selector=and([
        edge(),
        className("grandparent")
      ]))
]
```
and draw again (while adding the grandparent edges too)
```rascal-shell
rascal>graph(g + (g o g), cfg=cytoGraphConfig(nodeClassifier=nodeClassifier, edgeClassifier=edgeClassifier, styles=styles, \layout=lyt))
🔗 Serving 'Graph' at |http://localhost:9051/|
```

## alias NodeLinker {#vis-Graphs-NodeLinker}

A NodeLinker maps node identities to a source location to link to

```rascal
loc (&T _id1)
```

## function defaultNodeLinker {#vis-Graphs-defaultNodeLinker}

The default node linker assumes any loc found in the node identity is a proper link.

```rascal
loc defaultNodeLinker(/loc l)

default loc defaultNodeLinker(&T _)
```

## alias NodeLabeler {#vis-Graphs-NodeLabeler}

A NodeLabeler maps node identies to descriptive node labels

```rascal
str (&T _id2)
```

## function defaultNodeLabeler {#vis-Graphs-defaultNodeLabeler}

The default node labeler searches for any `str`` in the identity, or otherwise a file name of a `loc`

```rascal
str defaultNodeLabeler(/str s)

str defaultNodeLabeler(loc l)

default str defaultNodeLabeler(&T v)
```

## alias NodeClassifier {#vis-Graphs-NodeClassifier}

A NodeClassifier maps node identities to classes that are used later to select specific layout and coloring options.

```rascal
list[str] (&T _id3)
```

## function defaultNodeClassifier {#vis-Graphs-defaultNodeClassifier}

The default classifier produces no classes

```rascal
list[str] defaultNodeClassifier(&T _)
```

## alias EdgeClassifier {#vis-Graphs-EdgeClassifier}

An EdgeClassifier maps edge identities to classes that are used later to select specific layout and coloring options.

```rascal
list[str] (&T _from, &T _to)
```

## function defaultEdgeClassifier {#vis-Graphs-defaultEdgeClassifier}

The default edge classifier produces no classes

```rascal
list[str] defaultEdgeClassifier(&T _, &T _)
```

## alias EdgeLabeler {#vis-Graphs-EdgeLabeler}

An EdgeLabeler maps edge identies to descriptive edge labels.

```rascal
str (&T _source, &T _target)
```

## function defaultEdgeLabeler {#vis-Graphs-defaultEdgeLabeler}

The default edge labeler returns the empty label for all edges.

```rascal
str defaultEdgeLabeler(&T _source, &T _target)
```

## function graph {#vis-Graphs-graph}

A graph plot from a binary list relation.

```rascal
Content graph(lrel[&T x, &T y] v, CytoGraphConfig cfg = cytoGraphConfig())
```

#### Examples



```rascal-shell 
rascal>import vis::Graphs;
ok
rascal>graph([<x,x+1> | x <- [1..100]] + [<100,1>])
🔗 Serving 'Graph' at |http://localhost:9051/|
rascal>graph([<x,x+1> | x <- [1..100]] + [<100,1>], cfg=cytoGraphConfig(\layout=\defaultCircleLayout()))
🔗 Serving 'Graph' at |http://localhost:9051/|
```

Providing locations as node identities automatically transforms them to node links:

```rascal-shell 
rascal>import vis::Graphs;
ok
rascal>import IO;
ok
rascal>d = [<|std:///|, e> | e <- |std:///|.ls];
lrel[loc,loc]: [
  <|std:///|,|std:///Boolean.rsc|>,
  <|std:///|,|std:///Content.rsc|>,
  <|std:///|,|std:///DateTime.rsc|>,
  <|std:///|,|std:///Exception.rsc|>,
  <|std:///|,|std:///Grammar.rsc|>,
  <|std:///|,|std:///IO.rsc|>,
  <|std:///|,|std:///List.rsc|>,
  <|std:///|,|std:///ListRelation.rsc|>,
  <|std:///|,|std:///Location.rsc|>,
  <|std:///|,|std:///Map.rsc|>,
  <|std:///|,|std:///Message.rsc|>,
  <|std:///|,|std:///Messages.class|>,
  <|std:///|,|std:///Node.rsc|>,
  <|std:///|,|std:///ParseTree.rsc|>,
  <|std:///|,|std:///Prelude$1.class|>,
  <|std:///|,|std:///Prelude$2.class|>,
  <|std:///|,|std:///Prelude$3.class|>,
  <|std:///|,|std:///Prelude$4.class|>,
  <|std:///|,|std:///Prelude$Backtrack.class|>,
  <|std:///|,|std:///Prelude$ByteBufferBackedInputStream.class|>,
  <|std:///|,|std:///Prelude$Distance.class|>,
  <|std:///|,|std:///Prelude$Less.class|>,
  <|std:///|,|std:///Prelude$NodeComparator.class|>,
  <|std:///|,|std:///Prelude$ReleasableCallback.class|>,
  <|std:///|,|std:///Prelude$Sorting.class|>,
  <|std:///|,|std:///Prelude.class|>,
  <|std:///|,|std:///Prelude.rsc|>,
  <|std:///|,|std:///Relation.rsc|>,
  <|std:///|,|std:///Set.rsc|>,
  <|std:///|,|std:///String.rsc|>,
  <|std:///|,|std:///Type.class|>,
  <|std:///|,|std:///Type.rsc|>,
  <|std:///|,|std:///ValueIO.rsc|>,
  <|std:///|,|std:///analysis|>,
  <|std:///|,|std:///index.md|>,
  <|std:///|,|std:///lang|>,
  <|std:///|,|std:///resource|>,
  <|std:///|,|std:///util|>,
  <|std:///|,|std:///vis|>
]
rascal>d += [<e,f> | <_, e> <- d, isDirectory(e), f <- e.ls];
lrel[loc,loc]: [
  <|std:///|,|std:///Boolean.rsc|>,
  <|std:///|,|std:///Content.rsc|>,
  <|std:///|,|std:///DateTime.rsc|>,
  <|std:///|,|std:///Exception.rsc|>,
  <|std:///|,|std:///Grammar.rsc|>,
  <|std:///|,|std:///IO.rsc|>,
  <|std:///|,|std:///List.rsc|>,
  <|std:///|,|std:///ListRelation.rsc|>,
  <|std:///|,|std:///Location.rsc|>,
  <|std:///|,|std:///Map.rsc|>,
  <|std:///|,|std:///Message.rsc|>,
  <|std:///|,|std:///Messages.class|>,
  <|std:///|,|std:///Node.rsc|>,
  <|std:///|,|std:///ParseTree.rsc|>,
  <|std:///|,|std:///Prelude$1.class|>,
  <|std:///|,|std:///Prelude$2.class|>,
  <|std:///|,|std:///Prelude$3.class|>,
  <|std:///|,|std:///Prelude$4.class|>,
  <|std:///|,|std:///Prelude$Backtrack.class|>,
  <|std:///|,|std:///Prelude$ByteBufferBackedInputStream.class|>,
  <|std:///|,|std:///Prelude$Distance.class|>,
  <|std:///|,|std:///Prelude$Less.class|>,
  <|std:///|,|std:///Prelude$NodeComparator.class|>,
  <|std:///|,|std:///Prelude$ReleasableCallback.class|>,
  <|std:///|,|std:///Prelude$Sorting.class|>,
  <|std:///|,|std:///Prelude.class|>,
  <|std:///|,|std:///Prelude.rsc|>,
  <|std:///|,|std:///Relation.rsc|>,
  <|std:///|,|std:///Set.rsc|>,
  <|std:///|,|std:///String.rsc|>,
  <|std:///|,|std:///Type.class|>,
  <|std:///|,|std:///Type.rsc|>,
  <|std:///|,|std:///ValueIO.rsc|>,
  <|std:///|,|std:///analysis|>,
  <|std:///|,|std:///index.md|>,
  <|std:///|,|std:///lang|>,
  <|std:///|,|std:///resource|>,
  <|std:///|,|std:///util|>,
  <|std:///|,|std:///vis|>,
  <|std:///analysis|,|std:///analysis/clustering|>,
  <|std:///analysis|,|std:///analysis/diff|>,
  <|std:///analysis|,|std:///analysis/flow|>,
  <|std:///analysis|,|std:///analysis/formalconcepts|>,
  <|std:///analysis|,|std:///analysis/grammars|>,
  <|std:///analysis|,|std:///analysis/graphs|>,
  <|std:///analysis|,|std:///analysis/linearprogramming|>,
  <|std:///analysis|,|std:///analysis/m3|>,
  <|std:///analysis|,|std:///analysis/statistics|>,
  <|std:///lang|,|std:///lang/aterm|>,
  <|std:///lang|,|std:///lang/aut|>,
  <|std:///lang|,|std:///lang/box|>,
  <|std:///lang|,|std:///lang/c90|>,
  <|std:///lang|,|std:///lang/csv|>,
  <|std:///lang|,|std:///lang/diff|>,
  <|std:///lang|,|std:///lang/dimacs|>,
  <|std:///lang|,|std:///lang/dot|>,
  <|std:///lang|,|std:///lang/html|>,
  <|std:///lang|,|std:///lang/html5|>,
  <|std:///lang|,|std:///lang/java|>,
  <|std:///lang|,|std:///lang/javascript|>,
  <|std:///lang|,|std:///lang/json|>,
  <|std:///lang|,|std:///lang/manifest|>,
  <|std:///lang|,|std:///lang/paths|>,
  <|std:///lang|,|std:///lang/pico|>,
  <|std:///lang|,|std:///lang/rascal|>,
  <|std:///lang|,|std:///lang/rsf|>,
  <|std:///lang|,|std:///lang/sdf2|>,
  <|std:///lang|,|std:///lang/sexp|>,
  <|std:///lang|,|std:///lang/smtlib2|>,
  <|std:///lang|,|std:///lang/std|>,
  <|std:///lang|,|std:///lang/uri|>,
  <|std:///lang|,|std:///lang/xforms|>,
  <|std:///lang|,|std:///lang/xml|>,
  <|std:///lang|,|std:///lang/yaml|>,
  <|std:///resource|,|std:///resource/jdbc|>,
  <|std:///util|,|std:///util/Benchmark.class|>,
  <|std:///util|,|std:///util/Benchmark.rsc|>,
  <|std:///util|,|std:///util/Clipboard.class|>,
  <|std:///util|,|std:///util/Clipboard.rsc|>,
  <|std:///util|,|std:///util/Eval$EvaluatorInterruptTimer.class|>,
  <|std:///util|,|std:///util/Eval$RascalRuntime.class|>,
  <|std:///util|,|std:///util/Eval.class|>,
  <|std:///util|,|std:///util/Eval.rsc|>,
  <|std:///util|,|std:///util/FileSystem.rsc|>,
  <|std:///util|,|std:///util/Highlight.rsc|>,
  <|std:///util|,|std:///util/IDEServices.rsc|>,
  <|std:///util|,|std:///util/IDEServicesLibrary.class|>,
  <|std:///util|,|std:///util/Math.class|>,
  <|std:///util|,|std:///util/Math.rsc|>,
  <|std:///util|,|std:///util/Maybe.rsc|>,
  <|std:///util|,|std:///util/Memo.rsc|>,
  <|std:///util|,|std:///util/Monitor.class|>,
  <|std:///util|,|std:///util/Monitor.rsc|>,
  <|std:///util|,|std:///util/ObjectReader.class|>,
  <|std:///util|,|std:///util/ParseErrorRecovery$ScoredTree.class|>,
  <|std:///util|,|...
rascal>graph(d, \layout=defaultDagreLayout());
🔗 Serving 'Graph' at |http://localhost:9051/|
```
here we adapt the node labeler to show only the last file name in the path of the location:
```rascal-shell
rascal>graph(d, \layout=defaultDagreLayout(), cfg=cytoGraphConfig(nodeLabeler=str (loc l) { return l.file; }));
🔗 Serving 'Graph' at |http://localhost:9051/|
```

## function graph {#vis-Graphs-graph}

A graph plot from a ternary list relation where the middle column is the edge label.

```rascal
Content graph(lrel[&T x, &L edge, &T y] v, CytoGraphConfig cfg=cytoGraphConfig())
```

#### Examples



```rascal-shell 
rascal>import vis::Graphs;
ok
rascal>graph([<x,2*x+1,x+1> | x <- [1..100]] + [<100,101,1>])
🔗 Serving 'Graph' at |http://localhost:9051/|
```

## function graph {#vis-Graphs-graph}

A graph plot from a binary relation.

```rascal
Content graph(rel[&T x, &T y] v, CytoGraphConfig cfg=cytoGraphConfig())
```

#### Examples



```rascal-shell 
rascal>import vis::Graphs;
ok
rascal>graph({<x,x+1> | x <- [1..100]} + {<100,1>})
🔗 Serving 'Graph' at |http://localhost:9051/|
```

## function graph {#vis-Graphs-graph}

A graph plot from a ternary relation where the middle column is the edge label.

```rascal
Content graph(rel[&T x, &L edge, &T y] v, CytoGraphConfig cfg=cytoGraphConfig())
```

#### Examples



```rascal-shell 
rascal>import vis::Graphs;
ok
rascal>graph({<x,2*x+1,x+1> | x <- [1..100]} + {<100,101,1>})
🔗 Serving 'Graph' at |http://localhost:9051/|
```

## function cytoscape {#vis-Graphs-cytoscape}

This core workhorse mixes the graph data with the configuration to obtain visualizable CytoScape.js data-structure.

Produces an overall cytoscape.js wrapper which is sent as JSON to the client side.

```rascal
Cytoscape cytoscape(list[CytoData] \data, CytoGraphConfig cfg=cytoGraphConfig())
```


This data-structure is serialized to JSON and communicated directly to initialize cytoscape.js.
The serialization is done by the generic [IO](../..//Library/lang/json/IO.md) library under the hood of a [Webserver](../..//Library/util/Webserver.md).

## function graphData {#vis-Graphs-graphData}

Turns a `rel[loc from, loc to]` into a graph

```rascal
list[CytoData] graphData(rel[loc x, loc y] v, CytoGraphConfig cfg=cytoGraphConfig())
```

## function graphData {#vis-Graphs-graphData}

Turns any `rel[&T from, &T to]` into a graph

```rascal
default list[CytoData] graphData(rel[&T x, &T y] v, CytoGraphConfig cfg=cytoGraphConfig())
```

## function graphData {#vis-Graphs-graphData}

Turns any `lrel[loc from, &L edge, loc to]` into a graph

```rascal
list[CytoData] graphData(lrel[loc x, &L edge, loc y] v, CytoGraphConfig cfg=cytoGraphConfig())
```

## function graphData {#vis-Graphs-graphData}

Turns any `lrel[&T from, &L edge, &T to]` into a graph

```rascal
default list[CytoData] graphData(lrel[&T x, &L edge, &T y] v, CytoGraphConfig cfg=cytoGraphConfig())
```

## function graphData {#vis-Graphs-graphData}

Turns any `lrel[loc from, loc to]` into a graph

```rascal
list[CytoData] graphData(lrel[loc x, loc y] v, CytoGraphConfig cfg=cytoGraphConfig())
```

## function graphData {#vis-Graphs-graphData}

Turns any `lrel[&T from, &T to]` into a graph

```rascal
default list[CytoData] graphData(lrel[&T x, &T y] v, CytoGraphConfig cfg=cytoGraphConfig())
```

## function graphData {#vis-Graphs-graphData}

Turns any `rel[loc from, &L edge, loc to]` into a graph

```rascal
list[CytoData] graphData(rel[loc x, &L edge, loc y] v, CytoGraphConfig cfg=cytoGraphConfig())
```

## function graphData {#vis-Graphs-graphData}

Turns any `rel[&T from, &L edge, &T to]` into a graph

```rascal
default list[CytoData] graphData(rel[&T x, &L edge, &T y] v, CytoGraphConfig cfg=cytoGraphConfig())
```

## data CytoNodeShape {#vis-Graphs-CytoNodeShape}

```rascal
data CytoNodeShape  
     = \ellipse()
     | \triangle()
     | \round-triangle()
     | \rectangle()
     | \round-rectangle()
     | \bottom-round-rectangle()
     | \cut-rectangle()
     | \barrel()
     | \rhomboid()
     | \diamond()
     | \round-diamond()
     | \pentagon()
     | \round-pentagon()
     | \hexagon()
     | \round-hexagon()
     | \concave-hexagon()
     | \heptagon()
     | \round-heptagon()
     | \octagon()
     | \round-octagon()
     | \star()
     | \tag()
     | \round-tag()
     | \vee()
     | \polygon()
     ;
```

## data Cytoscape {#vis-Graphs-Cytoscape}
Overall cytoscape.js object for sending to the client side.

```rascal
data Cytoscape  
     = cytoscape(
        list[CytoData] elements = [],
        list[CytoStyleOf] style=[],
        CytoLayout \layout = cytolayout()
    )
     ;
```

## data CytoData {#vis-Graphs-CytoData}

```rascal
data CytoData  
     = cytodata(CytoElement \data, list[str] classes=[])
     ;
```

## data CytoElement {#vis-Graphs-CytoElement}

```rascal
data CytoElement  
     = \node(str id, str label=id, str editor="|none:///|")
     | \edge(str source, str target, str id="<source>-<target>", str label="")
     ;
```

## data CytoHorizontalAlign {#vis-Graphs-CytoHorizontalAlign}

```rascal
data CytoHorizontalAlign  
     = left()
     | center()
     | right()
     | auto()
     ;
```

## data CytoVerticalAlign {#vis-Graphs-CytoVerticalAlign}

```rascal
data CytoVerticalAlign  
     = top()
     | center()
     | bottom()
     | auto()
     ;
```

## data CytoArrowHeadStyle {#vis-Graphs-CytoArrowHeadStyle}

```rascal
data CytoArrowHeadStyle  
     = triangle()
     | \triangle-tee()
     | \circle-triangle()
     | \triangle-cross()
     | \triangle-backcurve()
     | vee()
     | tee()
     | square()
     | circle()
     | diamond()
     | chevron()
     | none()
     ;
```

## data CytoTextWrap {#vis-Graphs-CytoTextWrap}

```rascal
data CytoTextWrap  
     = none()
     | wrap()
     | ellipses()
     ;
```

## data CytoCurveStyle {#vis-Graphs-CytoCurveStyle}

```rascal
data CytoCurveStyle  
     = bezier()
     | \unbundled-bezier()
     | straight()
     | segments()
     | \straight-triangle()
     | taxi()
     | haystack()
     ;
```

## data CytoStyleOf {#vis-Graphs-CytoStyleOf}

```rascal
data CytoStyleOf  
     = cytoStyleOf(
        CytoSelector selector = \node(),
        CytoStyle style = cytoNodeStyle()
    )
     ;
```

## function cytoNodeStyleOf {#vis-Graphs-cytoNodeStyleOf}

```rascal
CytoStyleOf cytoNodeStyleOf(CytoStyle style)
```

## function cytoEdgeStyleOf {#vis-Graphs-cytoEdgeStyleOf}

```rascal
CytoStyleOf cytoEdgeStyleOf(CytoStyle style)
```

## function defaultNodeStyle {#vis-Graphs-defaultNodeStyle}

Instantiates a default node style

```rascal
CytoStyle defaultNodeStyle()
```


Because the JSON writer can not instantiate default values for keyword fields,
we have to do it manually here.

## function defaultEdgeStyle {#vis-Graphs-defaultEdgeStyle}

Instantiates a default edge style

```rascal
CytoStyle defaultEdgeStyle()
```


Because the JSON writer can not instantiate default values for keyword fields
we have to do it manually here.

## data CytoFontWeight {#vis-Graphs-CytoFontWeight}

```rascal
data CytoFontWeight  
     = normal()
     | lighter()
     | bold()
     | bolder()
     ;
```

## data CytoStyle {#vis-Graphs-CytoStyle}

```rascal
data CytoStyle  
     = cytoNodeStyle(
        str visibility          = "visible", /* hidden, collapse */
        str opacity             = "1",
        str width               = "label",
        str padding             = "10pt",
        str color               = "white",
        str \text-opacity       = "1",
        str \font-family        = "",
        str \font-size          = "12pt",
        str \font-style         = "",
        CytoFontWeight \font-weight = normal(),
        str \background-color   = "blue",
        str label               = "data(label)",
        CytoNodeShape shape     = CytoNodeShape::ellipse(),
        CytoHorizontalAlign \text-halign = CytoHorizontalAlign::center(),
        CytoVerticalAlign \text-valign = CytoVerticalAlign::\top(),
        CytoTextWrap \text-wrap = CytoTextWrap::none(),
        str \text-max-width     = "100px",
        CytoHorizontalAlign \text-justification = CytoHorizontalAlign::center(),
        int \line-height        = 1
    )
     | cytoEdgeStyle(
        str visibility          = "visible", /* hidden, collapse */
        str opacity             = "1",
        str \line-opacity       = "1",
        int width               = 3,
        str \line-color         = "black",
        str \line-style         = "solid", /* dotted, dashed */
        str color               = "red",
        str \target-arrow-color = "black",
        str \source-arrow-color = "black",
        CytoArrowHeadStyle \target-arrow-shape = CytoArrowHeadStyle::triangle(),
        CytoArrowHeadStyle \source-arrow-shape = CytoArrowHeadStyle::none(),
        CytoCurveStyle \curve-style            = CytoCurveStyle::\unbundled-bezier(),
        int \source-text-offset = 1,
        int \target-text-offset = 1,
        str label               = "data(label)"
    )
     ;
```

## data CytoSelector {#vis-Graphs-CytoSelector}
A combinator language that translates down to strings in JSON

```rascal
data CytoSelector  
     = \node()
     | \edge()
     | \id(str id)
     | \and(list[CytoSelector] conjuncts)
     | \or(list[CytoSelector] disjuncts)
     | \equal(str field, str \value)
     | \equal(str field, int limit)
     | \greater(str field, int limit)
     | \less(str field, int limit)
     | \greaterEqual(str field, int limit)
     | \lessEqual(str field, int limit)
     | \className(str)
     ;
```


* For field names you can use the names, or the dot notation for array indices and fields of objects: `"labels.0"`, `"name.first"`.
* `and` and `or` can not be nested; this will lead to failure to select anything at all. The or must be outside and the and must be inside.
* `node()` selects all nodes
* `edge()` selects all edges

## function \node {#vis-Graphs-\node}

Short-hand for a node with a single condition

```rascal
CytoSelector \node(CytoSelector condition)
```

## function \edge {#vis-Graphs-\edge}

Short-hand for a node with a single condition

```rascal
CytoSelector \edge(CytoSelector condition)
```

## function more {#vis-Graphs-more}

Utility to generate class attributes with multiple names consistently.

```rascal
str more(set[str] names)
```

## function formatCytoSelector {#vis-Graphs-formatCytoSelector}

Serialize a ((CytoSelector)) to string for client side expression.

```rascal
str formatCytoSelector(\node())

str formatCytoSelector(\edge())

str formatCytoSelector(\id(str i))

str formatCytoSelector(and(list[CytoSelector] cjs))

str formatCytoSelector(or(list[CytoSelector] cjs))

str formatCytoSelector(className(str class))

str formatCytoSelector(equal(str field, str val))

str formatCytoSelector(equal(str field, int lim))

str formatCytoSelector(greater(str field, int lim))

str formatCytoSelector(greaterEqual(str field, int lim))

str formatCytoSelector(lessEqual(str field, int lim))

str formatCytoSelector(less(str field, int lim))
```

## data CytoLayoutName {#vis-Graphs-CytoLayoutName}
Choice of different node layout algorithms.

```rascal
data CytoLayoutName  
     = grid()
     | circle()
     | breadthfirst()
     | cose()
     | dagre()
     ;
```


The different algorithms use heuristics to find a layout
that shows the structure of a graph best. Different types
of graph data call for different algorithms:
* `grid` is best when there are very few edges or when edges are not important. The edge relation
is not used at all for deciding where each node will end up. Grid 
is typically used for an initial exploration of the graph. It is very fast.
* `circle` puts all nodes on the edge of a circle and draws edges between them. The order on the
circle is arbitrary. This layout fails on larger collections of nodes because the points on the 
circle will become really small and indistinguishable. However for graphs with less than 100 nodes 
it provides a quick and natural overview.
* `breadthfirst` computes a breadthfirst spanning tree, and uses path length to decide on which
layer each node will reside. Cross-edges (between branches) and back-edges are allowed but if there
are many the graph will be messy. So this layout is best when you have a mostly hierarchical graph.
Examples are flow charts and dependency graphs.
* `cose` is a so-called "force-directed" layout. The edges become springs that both push nodes
apart as well as pull them together. Nodes drag on the surface but have an initial momentum such
that they can find a spot on the plain. This layout is very natural for scale-free networks such
as biological organisms, friends graphs and software ecosystems.

## function hierarchical {#vis-Graphs-hierarchical}

An alias for dagre layout for documentation purposes.

```rascal
CytoLayoutName hierarchical()
```


Dagre is a hierarchical graph layout.

## data CytoLayout {#vis-Graphs-CytoLayout}

```rascal
data CytoLayout (CytoLayoutName name = dagre(), bool animate=false) 
     = cytolayout()
     | breadthfirstLayout(
        CytoLayoutName name = CytoLayoutName::breadthfirst(),
        num spacingFactor= 1,
        list[str] roots = [],
        bool circle=false,
        bool grid=!circle,
        bool directed=false
    )
     | gridLayout(
        CytoLayoutName name = CytoLayoutName::grid(),
        int rows=2,
        int cols=2,
        bool avoidOverlap=true,
        num spacingFactor=.1
    )
     | circleLayout(
        CytoLayoutName name = CytoLayoutName::circle(),
        bool avoidOverlap=true,
        num spacingFactor=.1
    )
     | coseLayout(
        CytoLayoutName name = cose()
    )
     | dagreLayout(
        CytoLayoutName name = dagre(),
        num spacingFactor = .1,
        DagreRanker ranker = \network-simplex() // network-simples tight-tree, or longest-path
    )
     ;
```

## data DagreRanker {#vis-Graphs-DagreRanker}

```rascal
data DagreRanker  
     = \network-simplex()
     | \tight-tree()
     | \longest-path()
     ;
```

## function defaultCoseLayout {#vis-Graphs-defaultCoseLayout}

```rascal
CytoLayout defaultCoseLayout()
```

## function defaultCircleLayout {#vis-Graphs-defaultCircleLayout}

```rascal
CytoLayout defaultCircleLayout(bool avoidOverlap=true, num spacingFactor=.1)
```

## function defaultGridLayout {#vis-Graphs-defaultGridLayout}

```rascal
CytoLayout defaultGridLayout(int rows=2, int cols=rows, bool avoidOverlap=true, num spacingFactor=.1)
```

## function defaultBreadthfirstLayout {#vis-Graphs-defaultBreadthfirstLayout}

```rascal
CytoLayout defaultBreadthfirstLayout(num spacingFactor=.1, bool circle=false, bool grid=!circle, bool directed=false)
```

## function defaultDagreLayout {#vis-Graphs-defaultDagreLayout}

```rascal
CytoLayout defaultDagreLayout(num spacingFactor=1)
```

## function graphServer {#vis-Graphs-graphServer}

this is the main server generator for any graph value

```rascal
Response (Request) graphServer(Cytoscape ch)
```


Given a Graph value this server captures the value and serves it
as a JSON value to the HTML client generated by [plotHTML](../..//Library/vis/Graphs.md#vis-Graphs-plotHTML).

## function plotHTML {#vis-Graphs-plotHTML}

default HTML wrapper for a cytoscape.js graph

```rascal
HTMLElement plotHTML()
```


This client features:
* cytoscape.js loading with cytoscape-dagre and dagre present.
* fetching of graph data via `http://localhost/cytoscape` URL
* clickable links in every node that has an 'editor' data field that holds a `loc`, via the `http://localhost/editor?src=loc` URL
* full screen graph view

This client mirrors the server defined by [Graph Server](../..//Library/vis/Graphs.md#vis-Graphs-graphServer).

