---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://70dbc0a7-3755-4c95-a40a-15c0959640bf|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <4,16,|uuid://a8a60973-1c53-424a-9307-fb508306436c|>,
  <3,9,|uuid://45d1cbd0-2d86-4009-91d4-4dfcb16b8a7f|>,
  <5,25,|uuid://608409b4-3f17-49da-8d22-89663a2a4d70|>,
  <7,49,|uuid://f8df9894-fda6-4bee-bfaa-76efca01662c|>,
  <9,81,|uuid://804f1ddb-5c11-4a35-b335-9475f4b9bea9|>,
  <10,100,|uuid://e4d0a872-8f3b-4051-895a-9da417c33e73|>,
  <6,36,|uuid://42d934ac-fe12-4a46-aedc-d05b1082739c|>,
  <8,64,|uuid://bd8c3c0c-6857-4048-a65f-a1cfb0024e9b|>,
  <2,4,|uuid://028732b0-9d87-495c-8570-57a5454ab343|>,
  <1,1,|uuid://6f2955f5-7124-43bf-afc4-f3141663c8f1|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://ba21a859-e6df-4a58-8ecb-93af53dd3772|,
  <2,4>:|uuid://e3f4e79d-7be8-4c3d-b69b-89aed2e26778|,
  <7,49>:|uuid://5ca6b0bc-77d6-4b86-82e5-c93a6699ef72|,
  <9,81>:|uuid://3de426f4-79ef-4348-845d-bbedaab113fc|,
  <8,64>:|uuid://793ece3a-3a72-4f12-a17b-5bb6d99cd273|,
  <5,25>:|uuid://4580377c-3ae7-4610-9536-b687cee13cca|,
  <4,16>:|uuid://eec1af1f-0bff-4bd8-8dea-748cbc7e6082|,
  <1,1>:|uuid://1cbfe00f-3c1c-4fae-a949-d69471fadb1d|,
  <10,100>:|uuid://8618f029-7a24-4001-a09e-571bf58b54ae|,
  <3,9>:|uuid://c32a26cb-0aa7-4046-838f-86e9725805ce|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

