module lang::rascal::tests::library::analysis::diff::edits::HiFiTreeDiffTests

extend analysis::diff::edits::ExecuteTextEdits;
extend analysis::diff::edits::HiFiLayoutDiff;
extend analysis::diff::edits::HiFiTreeDiff;
extend lang::pico::\syntax::Main;

import IO;
import ParseTree;
import String;

public str simpleExample
    = "begin
      '  declare
      '    a : natural,
      '    b : natural;
      '  a := a + b;
      '  b := a - b;
      '  a := a - b
      'end    
      '";

public str ifThenElseExample
    = "begin
      '  declare
      '    a : natural;
      '  if a then 
      '    a := 10
      '  else
      '    if a then 
      '      a := 11 
      '    else 
      '      a := 12 
      '    fi
      '  fi
      'end
      '";

@synopsis{Specification of what it means for `treeDiff` to be syntactically correct}
@description{
TreeDiff is syntactically correct if:
* The tree after rewriting _matches_ the tree after applying the edits tot the source text and parsing that.
* Note that _matching_ ignores case-insensitive literals and layout, indentation and comments
}
bool editsAreSyntacticallyCorrect(type[&T<:Tree] grammar, str example, Tree(Tree) transform, list[TextEdit](Tree, Tree) diff) {
    orig        = parse(grammar, example);
    transformed = transform(orig);
    edits       = diff(orig, transformed);
    edited      = executeTextEdits(example, edits);

    try {
        if (transformed := parse(grammar, edited)) {
            return true;
        }
        else {
            println("The edited result is not the same:");
            println(edited);
            println("As the transformed:");
            println(transformed);
            return false;
        }
    }
    catch ParseError(loc l): {
        println("<transform> caused a parse error <l> in:");
        println(edited);
        return false;
    }
}

@synopsis{Extract the leading spaces of each line of code}
list[str] indentationLevels(str example)
    = [ i | /^<i:[\ ]*>[^\ ]*/ <- split("\n", example)];

@synopsis{In many cases, but not always, treeDiff maintains the indentation levels}
@description{
Typically when a rewrite does not change the lines of code count, 
and when the structure of the statements remains comparable, treeDiff
can guarantee that the indentation of a file remains unchanged, even if
significant changes to the code have been made.
}
@pitfalls{
* This specification is not true for any transformation. Only apply it to 
a test case if you can expect indentation-preservation for _the entire file_.
}
bool editsMaintainIndentationLevels(type[&T<:Tree] grammar, str example, Tree(Tree) transform, list[TextEdit](Tree, Tree) diff) {
    orig        = parse(grammar, example);
    transformed = transform(orig);
    edits       = diff(orig, transformed);
    edited      = executeTextEdits(example, edits);
    
    return indentationLevels(example) == indentationLevels(edited);
}

Tree identity(Tree x) = x;

start[Program] swapAB(start[Program] p) = visit(p) {
    case (Id) `a` => (Id) `b`
    case (Id) `b` => (Id) `a`
};

start[Program] swapIfBranches(start[Program] p) = visit(p) {
    case (Statement) `if <Expression e> then <{Statement ";"}* thenBranch> else <{Statement ";"}* elseBranch> fi`
      => (Statement) `if <Expression e> then 
                     '  <{Statement ";"}* elseBranch> 
                     'else 
                     '  <{Statement ";"}* thenBranch>
                     'fi`
};

start[Program] naturalToString(start[Program] p) = visit(p) {
    case (Type) `natural` => (Type) `string`
};

start[Program] addDeclarationToEnd(start[Program] p) = visit(p) {
    case (Program) `begin declare <{IdType ","}* decls>; <{Statement  ";"}* body> end`
        => (Program) `begin
                     '  declare
                     '    <{IdType ","}* decls>,
                     '    c : natural;
                     '  <{Statement  ";"}* body>
                     'end`
};

start[Program] addDeclarationToStart(start[Program] p) = visit(p) {
    case (Program) `begin declare <{IdType ","}* decls>; <{Statement  ";"}* body> end`
        => (Program) `begin
                     '  declare
                     '    c : natural,
                     '    <{IdType ","}* decls>;
                     '  <{Statement  ";"}* body>
                     'end`
};

start[Program] addDeclarationToMiddle(start[Program] p) = visit(p) {
    case (Program) `begin declare <{IdType ","}* pre>, <IdType middle>, <{IdType ","}* post>; <{Statement  ";"}* body> end`
        => (Program) `begin
                     '  declare
                     '    <{IdType ","}* pre>,
                     '    <IdType middle>,
                     '    middle : natural,
                     '    <{IdType ","}* post>;
                     '  <{Statement  ";"}* body>
                     'end`
};

start[Program](start[Program]) indent(str indentation = "  ", bool indentFirstLine = true) {
    return start[Program](start[Program] p) {
        return parse(#start[Program], indent(indentation, "<p>", indentFirstLine=indentFirstLine));
    };
}

start[Program] insertSpacesInDeclaration(start[Program] p) = visit(p) {
    case (IdType) `<Id id> : <Type t>`
        => (IdType) `<Id id>  :  <Type t>`
};

test bool nulTestWithId() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, identity, treeDiff);

test bool simpleSwapper() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, swapAB, treeDiff)
    && editsMaintainIndentationLevels(#start[Program], simpleExample, swapAB, treeDiff);

test bool addDeclarationToEndTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, addDeclarationToEnd, treeDiff);

test bool addDeclarationToStartTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, addDeclarationToStart, treeDiff);

test bool addDeclarationToMiddleTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, addDeclarationToMiddle, treeDiff);

test bool addDeclarationToStartAndMiddleTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, addDeclarationToStart o addDeclarationToMiddle, treeDiff);

test bool addDeclarationToMiddleAndEndTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, addDeclarationToMiddle o addDeclarationToEnd, treeDiff);

test bool addDeclarationToStartAndEndTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, addDeclarationToStart o addDeclarationToEnd, treeDiff);

test bool addDeclarationToStartMiddleAndEndTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, addDeclarationToStart o addDeclarationToMiddle o addDeclarationToEnd, treeDiff);

test bool addDeclarationToEndAndSwapABTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, addDeclarationToEnd o swapAB, treeDiff);

test bool addDeclarationToStartAndSwapABTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, addDeclarationToStart o swapAB, treeDiff);

test bool addDeclarationToStartAndEndAndSwapABTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, addDeclarationToStart o addDeclarationToEnd o swapAB, treeDiff);

test bool naturalToStringTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, naturalToString, treeDiff);

test bool naturalToStringAndAtoBTest() 
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, naturalToString o swapAB, treeDiff);

test bool swapBranchesTest()
    = editsAreSyntacticallyCorrect(#start[Program], ifThenElseExample, swapIfBranches, treeDiff);
    
test bool nulTestWithIdLayout()
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, identity, layoutDiff)
    && editsMaintainIndentationLevels(#start[Program], simpleExample, identity, layoutDiff);

test bool indentAllLayout()
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, indent(), layoutDiff)
    && !editsMaintainIndentationLevels(#start[Program], simpleExample, indent(), layoutDiff);

test bool insertSpacesInDeclarationLayout()
    = editsAreSyntacticallyCorrect(#start[Program], simpleExample, insertSpacesInDeclaration, layoutDiff)
    && editsMaintainIndentationLevels(#start[Program], simpleExample, insertSpacesInDeclaration, layoutDiff);

