---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://d9f90f9a-4931-4e62-8797-e3144f8cdf07|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <8,64,|uuid://1ca54078-926a-4d9c-8e21-00438977950f|>,
  <4,16,|uuid://de57a22b-8f14-4f1a-bce6-de2a38a9f738|>,
  <6,36,|uuid://86db87f5-92f8-4d1f-90fd-b63a3c890cd1|>,
  <5,25,|uuid://629df23c-583f-4e04-b8e3-f454bc3f0ae1|>,
  <10,100,|uuid://9d6aa238-1fa4-4242-812b-8829041b4cf1|>,
  <2,4,|uuid://4f1b1ebe-c27e-4c20-ba12-cbd871541d69|>,
  <9,81,|uuid://647dfbb9-2c1f-44a6-90b3-a9e7fcf90b95|>,
  <7,49,|uuid://87a51bba-6931-45a7-b88b-143f4844691e|>,
  <3,9,|uuid://1ddd2e54-6f26-4eab-8b98-a1e4f4b7dbdb|>,
  <1,1,|uuid://2c34c0f0-ed30-4d74-a92e-ba4d7015aaff|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://6ad898a4-a52a-4273-b33c-d65f6c39f756|,
  <2,4>:|uuid://fa188f12-eb0d-4e5c-8d6e-f30ac6fdc4e4|,
  <7,49>:|uuid://9bb808f9-8b3b-4d0d-a857-ccd652432a05|,
  <9,81>:|uuid://78c68d94-c6d9-4cfd-a69b-875d620b873a|,
  <8,64>:|uuid://e4d718e7-60e2-4a6c-98c3-8261036d2368|,
  <5,25>:|uuid://68e88a52-7799-4815-b739-09955a8bf951|,
  <4,16>:|uuid://0b9df685-da2c-4836-b6d6-0045a6bc64fd|,
  <1,1>:|uuid://b7005552-fd5e-4462-86dc-f6f4f391519e|,
  <10,100>:|uuid://42138f26-4141-4329-a374-976de957fb4a|,
  <3,9>:|uuid://86d1a100-e3e8-4de7-9df2-450de5cf6b65|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

