---
title: "module String"
id: String
slug: /Library//String
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

Library functions for strings.
#### Usage

```rascal
import String;
```

#### Dependencies
```rascal
extend Exception;
import List;
import ParseTree;
```

#### Description


The following library functions are defined for strings:
* [arbString](..//Library/String.md#String-arbString)
* [capitalize](..//Library/String.md#String-capitalize)
* [center](..//Library/String.md#String-center)
* [charAt](..//Library/String.md#String-charAt)
* [chars](..//Library/String.md#String-chars)
* [contains](..//Library/String.md#String-contains)
* [deescape](..//Library/String.md#String-deescape)
* [endsWith](..//Library/String.md#String-endsWith)
* [escape](..//Library/String.md#String-escape)
* [findAll](..//Library/String.md#String-findAll)
* [findFirst](..//Library/String.md#String-findFirst)
* [findLast](..//Library/String.md#String-findLast)
* [format](..//Library/String.md#String-format)
* [fromBase32](..//Library/String.md#String-fromBase32)
* [fromBase64](..//Library/String.md#String-fromBase64)
* [indent](..//Library/String.md#String-indent)
* [isEmpty](..//Library/String.md#String-isEmpty)
* [isValidCharacter](..//Library/String.md#String-isValidCharacter)
* [left](..//Library/String.md#String-left)
* [replaceAll](..//Library/String.md#String-replaceAll)
* [replaceFirst](..//Library/String.md#String-replaceFirst)
* [replaceLast](..//Library/String.md#String-replaceLast)
* [reverse](..//Library/String.md#String-reverse)
* [rexpMatch](..//Library/String.md#String-rexpMatch)
* [right](..//Library/String.md#String-right)
* [size](..//Library/String.md#String-size)
* [split](..//Library/String.md#String-split)
* [squeeze](..//Library/String.md#String-squeeze)
* [startsWith](..//Library/String.md#String-startsWith)
* [stringChar](..//Library/String.md#String-stringChar)
* [stringChars](..//Library/String.md#String-stringChars)
* [substitute](..//Library/String.md#String-substitute)
* [substring](..//Library/String.md#String-substring)
* [toBase32](..//Library/String.md#String-toBase32)
* [toBase64](..//Library/String.md#String-toBase64)
* [toInt](..//Library/String.md#String-toInt)
* [toLocation](..//Library/String.md#String-toLocation)
* [toLowerCase](..//Library/String.md#String-toLowerCase)
* [toReal](..//Library/String.md#String-toReal)
* [toUpperCase](..//Library/String.md#String-toUpperCase)
* [trim](..//Library/String.md#String-trim)
* [uncapitalize](..//Library/String.md#String-uncapitalize)
* [wrap](..//Library/String.md#String-wrap)


## function center {#String-center}

Center a string in given space.

```rascal
str center(str s, int n)

str center(str s, int n, str pad)
```


*  Center string `s` in string of length `n` using spaces.
*  Center string `s` in string of length `n` using `pad` as padding character.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>center("abc", 10);
str: "   abc    "
───
   abc    
───
rascal>center("abc", 10, "x");
str: "xxxabcxxxx"
───
xxxabcxxxx
───
```

## function charAt {#String-charAt}

Return character in a string by its index position.

```rascal
int charAt(str s, int i) throws IndexOutOfBounds
```


Return the character at position `i` in string `s` as integer character code.
Also see [StringChar](..//Library/String.md#String-stringChar) that converts character codes back to string.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>charAt("abc", 0);
int: 97
rascal>stringChar(charAt("abc", 0));
str: "a"
───
a
───
```

## function chars {#String-chars}

Return characters of a string.

```rascal
list[int] chars(str s)
```


Return a list of the characters of `s` as integer character codes.
Also see [StringChars](..//Library/String.md#String-stringChars) that converts character codes back to string.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>chars("abc");
list[int]: [97,98,99]
rascal>stringChars(chars("abc")) == "abc";
bool: true
```

## function contains {#String-contains}

Check that a string contains another string.

```rascal
bool contains(str input, str find)
```


Check whether the string `find` occurs as substring in the string `subject`.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>contains("abracadabra", "bra");
bool: true
rascal>contains("abracadabra", "e");
bool: false
```

## function deescape {#String-deescape}

Replace escaped characters by the escaped character itself (using Rascal escape conventions).

```rascal
str deescape(str s)
```

## function endsWith {#String-endsWith}

Check whether a string ends with a given substring.

```rascal
bool endsWith(str subject, str suffix)
```


Yields `true` if string `subject` ends with the string `suffix`.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>endsWith("Hello.rsc", ".rsc");
bool: true
```

## function escape {#String-escape}

Replace single characters in a string.

```rascal
str escape(str subject, map[str,str] mapping)
```


Return a copy of `subject` in which each single character key in replacements
has been replaced by its associated value.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>import IO;
ok
rascal>escape("abracadabra", ("a" : "AA", "c" : "C"));
str: "AAbrAACAAdAAbrAA"
───
AAbrAACAAdAAbrAA
───
rascal>L = escape("\"Good Morning\", he said", ("\"": "\\\""));
str: "\\\"Good Morning\\\", he said"
───
\"Good Morning\", he said
───
rascal>println(L);
\"Good Morning\", he said
ok
```

## function findAll {#String-findAll}

Find all occurrences of a string in another string.

```rascal
list[int] findAll(str subject, str find)
```


Find all occurrences of string `find` in string `subject`.
The result is a (possible empty) list of positions where `find` matches.

See also [Find First](..//Library/String.md#String-findFirst) and [Find Last](..//Library/String.md#String-findLast).

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>findAll("abracadabra", "a");
list[int]: [0,3,5,7,10]
rascal>findAll("abracadabra", "bra");
list[int]: [1,8]
rascal>findAll("abracadabra", "e");
list[int]: []
```

## function findFirst {#String-findFirst}

Find the first occurrence of a string in another string.

```rascal
int findFirst(str subject, str find)
```


Find the first occurrence of string `find` in string `subject`.
The result is either a position in `subject` or `-1` when `find` is not found.

Also see [Find All](..//Library/String.md#String-findAll) and [Find Last](..//Library/String.md#String-findLast).

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>findFirst("abracadabra", "a");
int: 0
rascal>findFirst("abracadabra", "bra");
int: 1
rascal>findFirst("abracadabra", "e");
int: -1
```

## function findLast {#String-findLast}

Find the last occurrence of a string in another string.

```rascal
int findLast(str subject, str find)
```


Find the last occurrence of string `find` in string `subject`.
The result is either a position in `subject` or `-1` when `find` is not found.

Also see [Find All](..//Library/String.md#String-findAll) and [Find First](..//Library/String.md#String-findFirst).

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>findLast("abracadabra", "a");
int: 10
rascal>findLast("abracadabra", "bra");
int: 8
rascal>findLast("abracadabra", "e");
int: -1
```

## function isEmpty {#String-isEmpty}

Check whether a string is empty.

```rascal
bool isEmpty(str s)
```


Returns `true` if string `s` is empty.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>isEmpty("");
bool: true
rascal>isEmpty("abc");
bool: false
```

## function arbString {#String-arbString}

Generate a arbitrary string.

```rascal
str arbString(int n)
```


Returns a string of maximum `n` length, with arbitrary characters.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>arbString(3);
str: ""
───

───
rascal>arbString(10);
str: ""
───

───
```

## function left {#String-left}

Left alignment of string in given space.

```rascal
str left(str s, int n)

str left(str s, int n, str pad)
```


*  Left align string `s` in string of length `n` using spaces.
*  Left align string `s` in string of length `n` using `pad` as pad character.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>left("abc", 10);
str: "abc       "
───
abc       
───
rascal>left("abc", 10, "x");
str: "abcxxxxxxx"
───
abcxxxxxxx
───
```

## function replaceAll {#String-replaceAll}

Replace all occurrences of a string in another string.

```rascal
str replaceAll(str subject, str find, str replacement)
```


Return a copy of `subject` in which all occurrences of `find` (if any) have been replaced by `replacement`.
Also see [Replace First](..//Library/String.md#String-replaceFirst) and [Replace Last](..//Library/String.md#String-replaceLast).

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>replaceAll("abracadabra", "a", "A");
str: "AbrAcAdAbrA"
───
AbrAcAdAbrA
───
rascal>replaceAll("abracadabra", "ra", "RARA");
str: "abRARAcadabRARA"
───
abRARAcadabRARA
───
rascal>replaceAll("abracadabra", "cra", "CRA");
str: "abracadabra"
───
abracadabra
───
```

#### Pitfalls


Note that `find` is a string (as opposed to, for instance, a regular expression in Java).

## function replaceFirst {#String-replaceFirst}

Replace the first occurrence of a string in another string.

```rascal
str replaceFirst(str subject, str find, str replacement)
```


Return a copy of `subject` in which the first occurrence of `find` (if it exists) has been replaced by `replacement`.
Also see [Replace All](..//Library/String.md#String-replaceAll) and [Replace Last](..//Library/String.md#String-replaceLast).

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>replaceFirst("abracadabra", "a", "A");
str: "Abracadabra"
───
Abracadabra
───
rascal>replaceFirst("abracadabra", "ra", "RARA");
str: "abRARAcadabra"
───
abRARAcadabra
───
rascal>replaceFirst("abracadabra", "cra", "CRA");
str: "abracadabra"
───
abracadabra
───
```

#### Pitfalls


Note that `find` is a string (as opposed to, for instance, a regular expression in Java).

## function replaceLast {#String-replaceLast}

Replace the last occurrence of a string in another string.

```rascal
str replaceLast(str subject, str find, str replacement)
```


Return a copy of `subject` in which the last occurrence of `find` (if it exists) has been replaced by `replacement`.
Also see [Replace First](..//Library/String.md#String-replaceFirst) and [Replace Last](..//Library/String.md#String-replaceLast).

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>replaceLast("abracadabra", "a", "A");
str: "abracadabrA"
───
abracadabrA
───
rascal>replaceLast("abracadabra", "ra", "RARA");
str: "abracadabRARA"
───
abracadabRARA
───
rascal>replaceLast("abracadabra", "cra", "CRA");
str: "abracadabra"
───
abracadabra
───
```

#### Pitfalls


Note that `find` is a string (as opposed to, for instance, a regular expression in Java).

## function reverse {#String-reverse}

Return a string with all characters in reverse order.

```rascal
str reverse(str s)
```


Returns string with all characters of string `s` in reverse order.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>reverse("abc");
str: "cba"
───
cba
───
```

## function right {#String-right}

Right alignment of a string value in a given space.

Right align s in string of length n using space.

```rascal
str right(str s, int n)

str right(str s, int n, str pad)
```


*  Right align string `s` in string of length `n` using spaces.
*  Right align string `s` in string of length `n` using `pad` as pad character.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>right("abc", 10);
str: "       abc"
───
       abc
───
rascal>right("abc", 10, "x");
str: "xxxxxxxabc"
───
xxxxxxxabc
───
```

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>right("abc", 10);
str: "       abc"
───
       abc
───
rascal>right("abc", 10, "x");
str: "xxxxxxxabc"
───
xxxxxxxabc
───
```

## function size {#String-size}

Determine length of a string value.

```rascal
int size(str s)
```


Returns the length (number of characters) in string `s`.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>size("abc");
int: 3
rascal>size("");
int: 0
```

## function startsWith {#String-startsWith}

Check whether a string starts with a given prefix.

```rascal
bool startsWith(str subject, str prefix)
```


Yields `true` if string `subject` starts with the string `prefix`.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>startsWith("Hello.rsc", "Hell");
bool: true
```

## function stringChar {#String-stringChar}

Convert a character code into a string.

```rascal
str stringChar(int char) throws IllegalArgument
```

## function stringChars {#String-stringChars}

Convert a list of character codes into a string.

```rascal
str stringChars(list[int] chars) throws IllegalArgument
```

## function isValidCharacter {#String-isValidCharacter}

Check that a given integer value is a valid Unicode code point.

```rascal
bool isValidCharacter(int ch)
```

## function substring {#String-substring}

Extract a substring from a string value.

```rascal
str substring(str s, int begin)

str substring(str s, int begin, int end)
```


*  Yields substring of string `s` from index `begin` to the end of the string.
*  Yields substring of string `s` from index `begin` to (but not including) index `end`.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>substring("abcdef", 2);
str: "cdef"
───
cdef
───
rascal>substring("abcdef", 2, 4);
str: "cd"
───
cd
───
```

## function toInt {#String-toInt}

Convert a string value to integer.

```rascal
int toInt(str s) throws IllegalArgument

int toInt(str s, int r) throws IllegalArgument
```


*  Converts string `s` to integer. 
*  Convert string `s` to integer using radix `r`.


Throws `IllegalArgument` when `s` cannot be converted.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>toInt("11");
int: 11
rascal>toInt("11", 8);
int: 9
```
Now try an erroneous argument:

```rascal-shell ,continue,error
rascal>toInt("abc");
|std:///String.rsc|(10801,456,<430,0>,<450,52>): IllegalArgument("abc","For input string: \"abc\"")
	at *** somewhere ***(|std:///String.rsc|(10801,456,<430,0>,<450,52>))
	at toInt(|prompt:///|(6,5,<1,6>,<1,11>))
```

## function toLowerCase {#String-toLowerCase}

Convert the characters in a string value to lower case.

```rascal
str toLowerCase(str s)
```


Convert all characters in string `s` to lowercase. Also see [To Upper Case](..//Library/String.md#String-toUpperCase).

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>toLowerCase("AaBbCc123");
str: "aabbcc123"
───
aabbcc123
───
```

## function toReal {#String-toReal}

Convert a string value to real.

```rascal
real toReal(str s)
```


Converts string `s` to a real. Throws `IllegalArgument` when `s` cannot be converted.

#### Examples



```rascal-shell ,error
rascal>import String;
ok
rascal>toReal("2.5e-3");
real: 0.0025
rascal>toReal("123");
real: 123.
rascal>toReal("abc");
|std:///String.rsc|(11678,320,<471,0>,<484,31>): IllegalArgument()
	at *** somewhere ***(|std:///String.rsc|(11678,320,<471,0>,<484,31>))
	at toReal(|prompt:///|(7,5,<1,7>,<1,12>))
```

## function toUpperCase {#String-toUpperCase}

Convert the characters in a string value to upper case.

```rascal
str toUpperCase(str s)
```


Converts all characters in string `s` to upper case.

Also see [To Lower Case](..//Library/String.md#String-toLowerCase).

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>toUpperCase("AaBbCc123");
str: "AABBCC123"
───
AABBCC123
───
```

## function trim {#String-trim}

Returns string with leading and trailing whitespace removed.

```rascal
str trim(str s)
```

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>trim("  jelly
|1 >>>>beans  ");
str: "jelly\nbeans"
───
jelly
beans
───
```

## function squeeze {#String-squeeze}

Squeeze repeated occurrences of characters.

```rascal
str squeeze(str src, str charSet)
```


Squeeze repeated occurrences in `src` of characters in `charSet` removed.
See http://commons.apache.org/lang/api-2.6/index.html?org/apache/commons/lang/text/package-summary.html[Apache]
for the allowed syntax in `charSet`.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>squeeze("hello", "el");
str: "helo"
───
helo
───
```
the other squeeze function uses character class types instead:
```rascal-shell
rascal>squeeze("hello", "el") == squeeze("hello", #[el]);
bool: true
```

## function squeeze {#String-squeeze}

Squeeze repeated occurrences of characters.

```rascal
str squeeze(str src, type[&CharClass] _:type[![]] _)
```


Squeeze repeated occurrences in `src` of characters, if they are a member of `&CharClass`, removed.

* `src` is any string
* `&CharClass` is a reified character class type such as `[a-z]` (a type that is a subtype of the class of all characters `![]`)
* To pass in a char-class type used the type reifier operator: `#[a-z]` or `#![]`

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>squeeze("hello", #[el]);
str: "helo"
───
helo
───
```

#### Benefits


* to squeeze all characters use the universal character class: `#![]` (the negation of the empty class).
* this function is type-safe; you can only pass in correct reified character classes like `#[A-Za-z]`.

#### Pitfalls


* `![]` excludes the 0'th unicode character, so we can not squeeze the unicode codepoint `0` using this function. 
If you really need to squeeze 0 then it's best to write your own:
```rascal
visit (x) { 
  case /<dot:.>+/ => "\a00" when dot == "\a00" 
}
````
* Do not confuse the character `0` (codepoint 48) with the zero codepoint: `#[0] != #[\a00]`

## function split {#String-split}

Split a string into a list of strings based on a literal separator.

```rascal
list[str] split(str sep, str src)
```

## function capitalize {#String-capitalize}

```rascal
str capitalize(str src)
```

## function uncapitalize {#String-uncapitalize}

```rascal
str uncapitalize(str src)
```

## function toBase64 {#String-toBase64}

Base-64 encode the characters of a string.

```rascal
str toBase64(str src, str charset=DEFAULT_CHARSET, bool includePadding=true)
```


   Convert the characters of a string to a list of bytes using UTF-8 encoding and then encode these bytes using base-64 encoding
   as defined by RFC 4648: https://www.ietf.org/rfc/rfc4648.txt.

## function fromBase64 {#String-fromBase64}

Decode a base-32 encoded string.

```rascal
str fromBase64(str src, str charset=DEFAULT_CHARSET)
```


  Convert a base-32 encoded string to bytes and then convert these bytes to a string using the specified cahracter set.
  The base-32 encoding used is defined by RFC 4648: https://www.ietf.org/rfc/rfc4648.txt.

## function toBase32 {#String-toBase32}

Base-32 encode the characters of a string.

```rascal
str toBase32(str src, str charset=DEFAULT_CHARSET, bool includePadding=true)
```


   Convert the characters of a string to a list of bytes using UTF-8 encoding and then encode these bytes using base-32 encoding
   as defined by RFC 4648: https://www.ietf.org/rfc/rfc4648.txt.

## function fromBase32 {#String-fromBase32}

Decode a base-32 encoded string.

```rascal
str fromBase32(str src, str charset=DEFAULT_CHARSET)
```


  Convert a base-32 encoded string to bytes and then convert these bytes to a string using the specified cahracter set.
  The base-32 encoding used is defined by RFC 4648: https://www.ietf.org/rfc/rfc4648.txt.

## function wrap {#String-wrap}

Word wrap a string to fit in a certain width.

```rascal
str wrap(str src, int wrapLength)
```


Inserts newlines in a string in order to fit the string in a certain width. It only breaks on spaces (' ').

## function format {#String-format}

```rascal
str format(str s, str dir, int n, str pad)
```

## function rexpMatch {#String-rexpMatch}

Determine if a string matches the given (Java-syntax) regular expression.

```rascal
bool rexpMatch(str s, str re)
```

## function toLocation {#String-toLocation}

Convert a string value to a (source code) location.

```rascal
loc toLocation(str s)
```


*  Converts string `s` to a location.
*  If the scheme is not provided, it is assumed to be `cwd`.

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>toLocation("http://grammarware.net");
loc: |http://grammarware.net|
rascal>toLocation("document.xml");
loc: |cwd:///document.xml|
```

## function substitute {#String-substitute}

Substitute substrings in a string based on a substitution map from location to string.

```rascal
str substitute(str src, map[loc,str] s)
```

#### Examples



```rascal-shell 
rascal>import String;
ok
rascal>substitute("abc", (|stdin:///|(1,1): "d"))
str: "adc"
───
adc
───
```

## function indent {#String-indent}

Indent a block of text

```rascal
str indent(str indentation, str content, bool indentFirstLine=false)
```


Every line in `content` will be indented using the characters
of `indentation`.

#### Benefits


* This operation executes in constant time, independent of the size of the content
or the indentation.
* Indent is the identity function if `indentation == ""`

#### Pitfalls


* This function works fine if `indentation` is not spaces or tabs; but it does not make much sense. 

