---
title: "module util::Benchmark"
id: Benchmark
slug: /Library/util/Benchmark
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

Functions for time measurement and benchmarking.
#### Usage

```rascal
import util::Benchmark;
```



The `Benchmark` library provides the following functions:
* [benchmark](../..//Library/util/Benchmark.md#util-Benchmark-benchmark)
* [cpuTime](../..//Library/util/Benchmark.md#util-Benchmark-cpuTime)
* [cpuTimeOf](../..//Library/util/Benchmark.md#util-Benchmark-cpuTimeOf)
* [gc](../..//Library/util/Benchmark.md#util-Benchmark-gc)
* [getFreeMemory](../..//Library/util/Benchmark.md#util-Benchmark-getFreeMemory)
* [getMaxFreeMemory](../..//Library/util/Benchmark.md#util-Benchmark-getMaxFreeMemory)
* [getMaxMemory](../..//Library/util/Benchmark.md#util-Benchmark-getMaxMemory)
* [getMilliTime](../..//Library/util/Benchmark.md#util-Benchmark-getMilliTime)
* [getNanoTime](../..//Library/util/Benchmark.md#util-Benchmark-getNanoTime)
* [getTotalMemory](../..//Library/util/Benchmark.md#util-Benchmark-getTotalMemory)
* [getUsedMemory](../..//Library/util/Benchmark.md#util-Benchmark-getUsedMemory)
* [heapDump](../..//Library/util/Benchmark.md#util-Benchmark-heapDump)
* [realTime](../..//Library/util/Benchmark.md#util-Benchmark-realTime)
* [realTimeOf](../..//Library/util/Benchmark.md#util-Benchmark-realTimeOf)
* [systemTime](../..//Library/util/Benchmark.md#util-Benchmark-systemTime)
* [systemTimeOf](../..//Library/util/Benchmark.md#util-Benchmark-systemTimeOf)
* [userTime](../..//Library/util/Benchmark.md#util-Benchmark-userTime)
* [userTimeOf](../..//Library/util/Benchmark.md#util-Benchmark-userTimeOf)


## function heapDump {#util-Benchmark-heapDump}

Write a JVM heap dump to a file.

```rascal
void heapDump(loc file, bool live=true)
```


* The file parameter has to be of the `file` scheme.
* The live parameter restricts the dump to only live objects.

## function getFreeMemory {#util-Benchmark-getFreeMemory}

Returns the free memory of the current JVM

```rascal
int getFreeMemory()
```


This returns the number of bytes that can be allocated
still against the current result of [Get Total Memory](../..//Library/util/Benchmark.md#util-Benchmark-getTotalMemory).

## function getTotalMemory {#util-Benchmark-getTotalMemory}

Returns the current total memory allocated by the current JVM

```rascal
int getTotalMemory()
```


This returns the number of bytes currently allocated for use by the JVM.
The number can change over time but it's never higher than [Get Max Memory](../..//Library/util/Benchmark.md#util-Benchmark-getMaxMemory)`

## function getMaxMemory {#util-Benchmark-getMaxMemory}

Returns the maximum amount of memory that is available to the current JVM

```rascal
int getMaxMemory()
```

Amount returned in bytes.

## function getUsedMemory {#util-Benchmark-getUsedMemory}

Returns the amount of memory that is currently in use by the programs running on this JVM

```rascal
int getUsedMemory()
```

Amount returned in bytes.

## function getMaxFreeMemory {#util-Benchmark-getMaxFreeMemory}

Returns the amount of memory that is yet available, in principle, on the current JVM

```rascal
int getMaxFreeMemory()
```

Amount returned in bytes.

## function cpuTime {#util-Benchmark-cpuTime}

CPU time in nanoseconds (10^-9^ sec)

```rascal
int cpuTime()
```


* Current cpu time in __nanoseconds__ (10^-9^ sec) since the start of the thread that runs the code that calls this function.
* This number has nanoseconds resolution, but not necessarily nanosecond accuracy.

#### Examples



```rascal-shell 
rascal>import util::Benchmark;
ok
rascal>int fac(int n) {
|1 >>>>  if (n <= 1) return 1;
|2 >>>>  else return n * fac(n - 1);
|3 >>>>}
int (int): function(|prompt:///|(0,72,<1,0>,<4,1>))
```
Here we measure time by using separate calls to `cpuTime` before and after a call to `fac`.

```rascal-shell ,continue
rascal>before = cpuTime();
int: 38350250549
rascal>fac(50);
int: 30414093201713378043612608166064768844377641568960512000000000000
rascal>cpuTime() - before;
int: 1829829
```

See also [Cpu Time Of](../..//Library/util/Benchmark.md#util-Benchmark-cpuTimeOf) for a more convenient way of measuring the time spent during a block of code.

#### Pitfalls


* The timings shown above may be significantly influenced by the documentation compilation process

## function realTime {#util-Benchmark-realTime}

Returns wall clock time in _milliseconds_ since the Unix epoch

```rascal
int realTime()
```


Returns the difference, measured in milliseconds, between the current time and midnight, January 1, 1970 UTC

#### Pitfalls


* The actual accuracy of the time may be not as good as a millisecond. This depends on OS and hardware specifics.
   * Note that the resolution is _milliseconds_ here, while [Cpu Time](../..//Library/util/Benchmark.md#util-Benchmark-cpuTime) produces nanosecond resolution.

## function getNanoTime {#util-Benchmark-getNanoTime}

Return nanoseconds clock time of the JVM's high resolution clock.

```rascal
int getNanoTime()
```


See `System.nanoTime` Java documentation. An excerpt:

> Returns the current value of the running Java Virtual Machine's 
> high-resolution time source, in nanoseconds.
> This method can only be used to measure elapsed time and is
> not related to any other notion of system or wall-clock time.
> The value returned represents nanoseconds since some fixed but
> arbitrary <i>origin</i> time (perhaps in the future, so values
> may be negative).  The same origin is used by all invocations of
> this method in an instance of a Java virtual machine; other
> virtual machine instances are likely to use a different origin.
     
> This method provides nanosecond precision, but not necessarily
> nanosecond resolution (that is, how frequently the value changes).

## function getMilliTime {#util-Benchmark-getMilliTime}

Synonym for ((realTime))

```rascal
int getMilliTime()
```

## function cpuTimeOf {#util-Benchmark-cpuTimeOf}

Measure the exact running time of a block of code, using ((cpuTime)).

```rascal
int cpuTimeOf(void () block)
```

## function systemTime {#util-Benchmark-systemTime}

System time in nanoseconds (10^-9^ sec).

```rascal
int systemTime()
```


Returns the CPU time that the current thread has executed in system mode in nanoseconds.

* Current system time in nanoseconds (10^-9^ sec) since the start of the thread that runs the code that calls this function.
* The returned value is of nanoseconds precision but not necessarily nanoseconds accuracy.
* CPU time is the number of CPU cycles times the OS-registered clock speed.
* The other [CPU time](../..//Library/util/Benchmark.md#util-Benchmark-cpuTime), next to [System time](../..//Library/util/Benchmark.md#util-Benchmark-systemTime) is spent in [User time](../..//Library/util/Benchmark.md#util-Benchmark-userTime).

#### Examples



```rascal-shell ,continue
rascal>import util::Benchmark;
ok
```

Here we measure time by using separate calls to `sytemTime` before and after a call to `fac`.


```rascal-shell ,continue
rascal>before = systemTime();
int: 330882370
rascal>fac(50);
int: 30414093201713378043612608166064768844377641568960512000000000000
rascal>systemTime() - before;
int: 1494178
```

## function systemTimeOf {#util-Benchmark-systemTimeOf}

Measure the exact running time of a block of code, using ((systemTime)).

```rascal
int systemTimeOf(void () block)
```

#### Examples



```rascal-shell ,continue
rascal>import util::Benchmark;
ok
rascal>int fac(int n) = n <= 1 ? 1 : n * fac(n - 1);
int (int): function(|prompt:///|(0,45,<1,0>,<1,45>))
rascal>systemTimeOf(
|1 >>>>   void() { 
|2 >>>>      fac(50); 
|3 >>>>   } 
|4 >>>>);
int: 240251
```

## function userTime {#util-Benchmark-userTime}

User time in nanoseconds (10^-9^ sec)

```rascal
int userTime()
```


Returns the CPU time that the current thread has executed in user mode in nanoseconds.

* The returned value is of nanoseconds precision but not necessarily nanoseconds accuracy.
* As distinguished from [Now](../..//Library/DateTime.md#DateTime-now) which returns the wall clock time since the Unix epoch.
* CPU time is the number of CPU cycles times the OS-registered clock speed.
* The other [CPU time](../..//Library/util/Benchmark.md#util-Benchmark-cpuTime), next to [user time](../..//Library/util/Benchmark.md#util-Benchmark-userTime) is spent in [system time](../..//Library/util/Benchmark.md#util-Benchmark-systemTime).

#### Examples



```rascal-shell 
rascal>import util::Benchmark;
ok
rascal>int fac(0) = 1;
int (int): function(|prompt:///|(0,15,<1,0>,<1,15>))
rascal>default int fac(int n) = n * fac(n - 1);
int (int): function(|prompt:///|(0,40,<1,0>,<1,40>))
```
Here we measure time by using separate calls to `userTime` before and after a call to `fac`.

```rascal-shell ,continue
rascal>before = userTime();
int: 38090000000
rascal>fac(50);
int: 30414093201713378043612608166064768844377641568960512000000000000
rascal>userTime() - before;
int: 0
```

## function userTimeOf {#util-Benchmark-userTimeOf}

Measure the exact running time of a block of code in nanoseconds, doc combined with previous function.

```rascal
int userTimeOf(void () block)
```

## function realTimeOf {#util-Benchmark-realTimeOf}

Measure the exact running time of a block of code in milliseconds, doc included in previous function.

```rascal
int realTimeOf(void () block)
```

#### Pitfalls


* watch out this is measured in milliseconds, not nanoseconds

## function benchmark {#util-Benchmark-benchmark}

Utility to measure and compare the execution time a set of code blocks

```rascal
map[str,num] benchmark(map[str, void()] Cases)

map[str,num] benchmark(map[str, void()] Cases, int (void ()) duration)
```


Given is a map that maps strings (used as label to identify each case) to void-closures that execute the code to be benchmarked.
An optional `duration` argument can be used to specify the function to perform the actual measurement. By default the function [Real Time Of](../..//Library/util/Benchmark.md#util-Benchmark-realTimeOf) is used. A map of labels and durations is returned.

#### Examples



```rascal-shell 
rascal>import util::Benchmark;
ok
rascal>int fac(int n) {
|1 >>>>  if (n <= 1) return 1;
|2 >>>>  else return n * fac(n - 1);
|3 >>>>}
int (int): function(|prompt:///|(0,72,<1,0>,<4,1>))
```

We measure two calls to the factorial function with arguments `100`, respectively, `200` 
(using by default [Real Time](../..//Library/util/Benchmark.md#util-Benchmark-realTime) that returns milliseconds):

```rascal-shell ,continue
rascal>benchmark(
|1 >>>>   ("fac100" : void() {
|2 >>>>                  fac(100);
|3 >>>>               }, 
|4 >>>>   "fac200" :  void() {
|5 >>>>                  fac(200);
|6 >>>>               }) 
|7 >>>>   );
map[str, num]: ("fac100":0,"fac200":1)
```

We can do the same using [User Time](../..//Library/util/Benchmark.md#util-Benchmark-userTime) that returns nanoseconds:

```rascal-shell ,continue
rascal>benchmark( 
|1 >>>>   ("fac100" : void() {
|2 >>>>                  fac(100);
|3 >>>>            }, 
|4 >>>>   "fac200" : void() {
|5 >>>>                  fac(200);
|6 >>>>            })
|7 >>>>   , userTimeOf);
map[str, num]: ("fac100":0,"fac200":0)
```

## function gc {#util-Benchmark-gc}

```rascal
void gc()
```


This function tries to trigger a garbage collection. It may be useful to call this function
just before measuring the efficiency of a code block, in order to factor out previous effects
on the heap.

#### Benefits


* This helps avoiding to restart the JVM, and optionally warming it up, for each individual measurement.
* Long running terminals can be rejuvenated on demand by a call to [Gc](../..//Library/util/Benchmark.md#util-Benchmark-gc).

#### Pitfalls


* Although a GC cycle is triggered by this function, it guarantees nothing about the effect of this cycle in terms of completeness or precision in removing garbage from the heap.
* GC only works for real garbage. So if there is an unrelated accidental memory leak somewhere, it may better to start a fresh JVM to measure the current functionality under scrutiny.

