---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://4e1edb1b-8f04-4315-b951-a9f72d6ce69d|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <1,1,|uuid://7c7eb8fa-6b36-4b1d-a8c2-9c8b9e088d2f|>,
  <10,100,|uuid://f7ff55ce-2134-4da3-a137-59eaf09e3f04|>,
  <7,49,|uuid://0c365593-9148-4c3a-b16b-8960dd84f672|>,
  <2,4,|uuid://2174f0b8-f34b-4c87-aef9-4129d27ce2a1|>,
  <8,64,|uuid://acd3410b-fad2-48ab-ae38-0e3fc90e71ac|>,
  <9,81,|uuid://c9016a81-ba22-4abd-98c1-436264ae1b4a|>,
  <3,9,|uuid://5f9524b8-067d-464f-aeef-459692b8944c|>,
  <4,16,|uuid://2481dece-a5b7-4ba1-bea6-7fe3869efade|>,
  <6,36,|uuid://e88739a0-3054-40de-93a1-fa99ee0f8a92|>,
  <5,25,|uuid://27d4716f-c37b-4e27-926f-f0de13cf5008|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://13b1393a-6a30-4ba6-aa2b-7cd63518c324|,
  <2,4>:|uuid://041508bf-6fb8-450e-a280-d5e1cb21c4d5|,
  <7,49>:|uuid://77d62900-015e-4cfb-9a11-8f70609c2bf5|,
  <9,81>:|uuid://a2c54e48-26ed-4966-9399-c845f49c000a|,
  <8,64>:|uuid://099e130d-8b6e-4a69-a8fd-412ce0df5108|,
  <5,25>:|uuid://6f429e67-506f-4877-8ac1-ca7c4accf573|,
  <4,16>:|uuid://069ee85f-d706-4390-a76d-8e8ac3b30597|,
  <1,1>:|uuid://d053f6ce-e1e1-495f-a7cc-1eb59c2d56cb|,
  <10,100>:|uuid://661f0c40-6c83-464c-88ac-6070164b9087|,
  <3,9>:|uuid://b0ebcd61-8fd9-4529-94da-14bc9fccdc04|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

