---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://dd068a5c-690e-4d92-85ba-28806c522c0f|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <3,9,|uuid://d90659b0-2512-4c45-a7a9-8addda86a2eb|>,
  <6,36,|uuid://fd22c6b8-7dab-4cf6-9884-a74c553c03f5|>,
  <7,49,|uuid://ad74ac2a-5030-4708-b6a5-66aee26a8ff2|>,
  <1,1,|uuid://23e4c070-17cc-48bf-868c-5af0d32b0d30|>,
  <9,81,|uuid://d99bd16f-70c6-4873-9358-c25a9418db4c|>,
  <2,4,|uuid://19ea4d45-07d0-4b6c-8ee9-f0579a845a2a|>,
  <4,16,|uuid://a37a4d62-1273-4c8c-bc4a-611041a71627|>,
  <5,25,|uuid://7eeb0f00-c93b-4306-9c7b-f57380ca773c|>,
  <10,100,|uuid://3f282e46-5766-454f-ae7c-403f5ee3b1c3|>,
  <8,64,|uuid://90a61808-92af-40aa-8659-8dbe9efd89a5|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://29a78c25-3c3d-49f8-bdc1-2f768425e9bb|,
  <2,4>:|uuid://a947472b-4f26-4d9a-ae0f-e8dde265f3e8|,
  <7,49>:|uuid://e9ad32f7-3b81-4216-9321-05e49f49d6ce|,
  <9,81>:|uuid://61e822b2-edfa-40a1-bcef-1ee8465daa3b|,
  <8,64>:|uuid://016305bd-c0cd-4cce-ac84-5a7e8fd67ed1|,
  <5,25>:|uuid://00e46fd1-d8e8-433a-81d2-db4025d9e24e|,
  <4,16>:|uuid://5f60bbdd-8691-443d-9b18-68bdd3665e6c|,
  <1,1>:|uuid://23ec5b0b-1143-4182-912e-33a42364d2c0|,
  <10,100>:|uuid://4d80f928-6c09-45df-bb2e-bf1f5c5771ef|,
  <3,9>:|uuid://9119463e-d2bf-48a0-927c-5d73559553be|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

