---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://ebc24e82-a9b6-4555-b9b3-02310b98fb2c|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <4,16,|uuid://1612106a-e640-4320-90bc-046de00c02b3|>,
  <7,49,|uuid://99cade87-f475-4124-9cd3-1902477ba8de|>,
  <8,64,|uuid://ef90268f-90f9-4f9c-aeaf-a89fe217594c|>,
  <1,1,|uuid://4a3aed36-dc75-4631-90e7-8ddc9ad1b026|>,
  <3,9,|uuid://01228447-4cea-4462-841c-016bb2596cb1|>,
  <6,36,|uuid://65965b90-2860-45cb-ae09-871d292400fa|>,
  <9,81,|uuid://1cb56049-61b3-4927-a93e-5ea9adb64a30|>,
  <5,25,|uuid://92cb9cb0-7b86-4ca0-97e2-5fceaf249893|>,
  <10,100,|uuid://6b9e308e-e032-4f83-98ee-e110088a6c71|>,
  <2,4,|uuid://ba61a3d4-93d1-40ed-8c9f-0761e152c13f|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://7c7b910c-435f-435a-a8b3-1f7dd15aae24|,
  <2,4>:|uuid://ff065cd4-f376-4850-b47b-4086b7b57515|,
  <7,49>:|uuid://9972ae8f-43e4-49f4-a871-b7ff09a9a374|,
  <9,81>:|uuid://8ee17afa-8aa6-4dfe-8a2f-c29034a83e06|,
  <8,64>:|uuid://226c9082-b24a-4efd-b390-032b577f415d|,
  <5,25>:|uuid://ab17501f-d08c-4a4c-a134-4834ca060183|,
  <4,16>:|uuid://1ef2db1e-717d-410a-8828-97560f5f4806|,
  <1,1>:|uuid://a46dc31a-8e18-4cf9-90af-0ef037030427|,
  <10,100>:|uuid://fbe0b405-fd2e-426f-90df-e0a8b2ca1f92|,
  <3,9>:|uuid://d927a3d4-d316-49f9-af4d-c83673952b4f|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

