---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://74b4edab-bd26-49b3-a65d-702a5023cd50|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <10,100,|uuid://9d2ddf37-c39c-4f42-94d9-9e8b6cd7504b|>,
  <4,16,|uuid://613a595b-5d50-4999-9c77-627b636394b7|>,
  <8,64,|uuid://e9be261d-c3fe-4fc3-8f0d-71c0c9c81323|>,
  <6,36,|uuid://cecd5627-400c-4c36-ba7a-75b8c5bcc3b3|>,
  <5,25,|uuid://2e35f41f-1d78-432a-8f0e-506b3e71c52a|>,
  <9,81,|uuid://08f60d67-6509-4376-a852-72f7d1059dfe|>,
  <1,1,|uuid://ef7aee70-2f2c-4edc-9497-1150e980e7ad|>,
  <7,49,|uuid://505ebf50-29f0-4765-9bde-cec621fe7fa9|>,
  <3,9,|uuid://f14a3051-03cf-4d22-9343-52c99bc138aa|>,
  <2,4,|uuid://4b740dfe-1ef3-45b2-8f3a-018540bf7388|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://742bdbbd-cb30-40ae-8739-cb59da16e263|,
  <2,4>:|uuid://6848f5b9-6ab4-4ba0-b809-7d47a9a9fb92|,
  <7,49>:|uuid://cc9a0b60-8d5a-4a29-b9d5-93d6677bf476|,
  <9,81>:|uuid://7256d8cc-dcc2-49a8-a75c-cd67554c02b8|,
  <8,64>:|uuid://6ca4f229-ff33-40a5-8045-52f56878bcb6|,
  <5,25>:|uuid://60a79ff7-1c1b-41c8-a158-420ce1a67744|,
  <4,16>:|uuid://c4017ede-8462-427d-85de-cf9791aeb3b9|,
  <1,1>:|uuid://37baeda6-c4bf-4fb5-81d8-fc0b5c40c472|,
  <10,100>:|uuid://d50eada8-0f9b-4a98-9aa2-a58a0b7be0f2|,
  <3,9>:|uuid://d66986de-8e56-4399-a640-c192ab2877d7|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

