---
title: "module util::Sampling"
id: Sampling
slug: /Library/util/Sampling
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

Utilities to randomly select smaller datasets from larger datasets
#### Usage

```rascal
import util::Sampling;
```

#### Dependencies
```rascal
import util::Math;
import Map;
import List;
import Set;
```

#### Description


Sampling is important when the analysis algorithms do not scale to the size of 
the original corpus, or when you need to train an analysis on a representative
set without overfitting on the entire corpus. These sampling functions all
assume that a uniformly random selection is required.


## function sample {#util-Sampling-sample}

Reduce the arity of a set by selecting a uniformly distributed sample.

```rascal
set[&T] sample(set[&T] corpus, int target)
```


A uniform subset is computed by iterating over the set and skipping every element
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new set of
expected `target` size, but most probably a little smaller or larger.

#### Examples



```rascal-shell 
rascal>import util::Sampling;
ok
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"b","c","e","k"}
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"a","f","k"}
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"e","f","g","h","k"}
```

## function sample {#util-Sampling-sample}

Reduce the length of a list by selecting a uniformly distributed sample.

```rascal
list[&T] sample(list[&T] corpus, int target)
```


The random selection of elements does not change their initial order in the list.
A uniform sublist is computed by iterating over the list and skipping every element
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new list of
expected `target` size, but most probably a little smaller or larger.

#### Examples



```rascal-shell 
rascal>import util::Sampling;
ok
rascal>sample([1..1000], 30)
list[int]: [13,31,92,143,182,192,207,208,214,277,282,291,319,332,337,343,353,370,457,510,522,526,542,564,598,599,651,670,713,735,757,802,831,834,936,962,966,989]
rascal>sample([1..1000], 30)
list[int]: [88,190,193,356,386,390,393,447,499,506,527,641,666,674,700,738,751,943,986]
rascal>sample([1..1000], 30)
list[int]: [38,70,73,122,226,384,391,455,456,497,522,561,572,574,628,738,759,770,791,885,886,902,911,915,961,992,994]
```

## function sample {#util-Sampling-sample}

Reduce the size of a map by selecting a uniformly distributed sample.

```rascal
map[&T,&U] sample(map[&T,&U] corpus, int target)
```


A uniform submap is computed by iterating over the map's keys and skipping every key
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new map of
expected `target` size, but most probably a little smaller or larger.

