---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://26287004-64d5-48b3-bf61-718cb64f473c|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <8,64,|uuid://04283a12-8466-4c64-ab41-5f862adb97ab|>,
  <3,9,|uuid://3738b7f4-6807-4c59-9ace-30e2ca7d16e6|>,
  <6,36,|uuid://cfa91313-0916-40b3-812c-f523fd927978|>,
  <5,25,|uuid://1ceef4ec-6fa9-49d7-a98e-b66cd7815b92|>,
  <7,49,|uuid://5f36f0dd-3b89-40a0-b788-c86521dca452|>,
  <1,1,|uuid://958f511b-881f-412e-9b87-3c8eb01c5823|>,
  <2,4,|uuid://f3d11ec7-2758-46ba-b2f5-81d2080f9da7|>,
  <9,81,|uuid://8a83abc0-73ae-4854-a349-51d59944d73d|>,
  <10,100,|uuid://61142a79-b8a3-4074-bf76-a071f339bda9|>,
  <4,16,|uuid://e226369c-ddd4-49b5-b220-c2c2ffce7bcb|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://cfa275c3-8cdf-4b4a-a961-e784a9dfa9a2|,
  <2,4>:|uuid://e163e224-6e9b-40a8-94c5-188b4718094c|,
  <7,49>:|uuid://0193c223-c83b-4974-a757-411a49daaaab|,
  <9,81>:|uuid://9687f9ef-3172-4369-ac86-061b8ccfc35c|,
  <8,64>:|uuid://6ce32296-ec63-485c-83f0-8993691294bd|,
  <5,25>:|uuid://f22eb361-8db8-4a8b-bc13-6fb336ca7562|,
  <4,16>:|uuid://0f2f5f1a-25d7-4589-9d3c-c81d0f3bf1d6|,
  <1,1>:|uuid://b5fc9141-1840-4214-8476-751e91ac2ba6|,
  <10,100>:|uuid://957da024-6458-485f-8c3f-bc98eac90103|,
  <3,9>:|uuid://0e18c7c3-a416-4dda-adff-f5906babef0d|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

