---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://b450f908-0884-4f34-ab5c-b186a644fc94|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <3,9,|uuid://8737dbc5-245c-4f75-9782-4f2d01c627f4|>,
  <2,4,|uuid://6c174a48-f84d-49f2-a7e9-79dfdd5aff91|>,
  <1,1,|uuid://9f4f6761-d440-4acc-bcab-af88956f6c64|>,
  <4,16,|uuid://c47736dc-9bae-413b-8f96-711b2a69b36e|>,
  <8,64,|uuid://5ddf6b48-5ba4-4c9c-8aca-6b66984549bd|>,
  <5,25,|uuid://a708d3ec-a8b7-469b-8390-224e11823610|>,
  <7,49,|uuid://bfbe752a-479b-42ff-ae55-e47674372d58|>,
  <6,36,|uuid://1291cbb2-7131-4ce5-8182-50a06825028f|>,
  <9,81,|uuid://34fe185c-bb74-4dc5-9214-e2965d0ff48b|>,
  <10,100,|uuid://75c36358-dff6-42fa-824b-c62544f4e0a2|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://e8a6a0aa-fc5f-40c3-a947-4c9fd2137e2a|,
  <2,4>:|uuid://4a268f8f-aa61-4ab1-9b08-feafc9f8f499|,
  <7,49>:|uuid://71bf4383-965e-4f6a-9e8b-5fd3ca746bf3|,
  <9,81>:|uuid://fc187b0f-352f-49f3-affa-cd4588b7943c|,
  <8,64>:|uuid://0685573f-1679-46e4-a4a2-939ae65e1607|,
  <5,25>:|uuid://b8830f01-9dd8-43e0-92ef-7df15d3d3c98|,
  <4,16>:|uuid://5f974fd5-ae2e-4c82-b693-e87753827cac|,
  <1,1>:|uuid://8c35ba98-51c7-4c16-8724-fa8baa410f71|,
  <10,100>:|uuid://d741d8b3-6e0c-4c09-9b13-82f49d6d125b|,
  <3,9>:|uuid://f385a413-00b2-4e0c-bf39-db701418674d|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

