/*
 * Copyright (c) 2025, Swat.engineering
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package org.rascalmpl.util.maven;

import java.io.PrintWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpClient.Version;
import java.nio.file.Path;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import org.apache.maven.model.Model;
import org.apache.maven.model.building.DefaultModelBuilderFactory;
import org.apache.maven.model.building.DefaultModelBuildingRequest;
import org.apache.maven.model.building.ModelBuilder;
import org.apache.maven.model.building.ModelBuildingException;
import org.apache.maven.model.building.ModelBuildingRequest;
import org.apache.maven.model.building.ModelBuildingResult;
import org.apache.maven.model.building.ModelCache;
import org.apache.maven.model.building.ModelProblem;
import org.apache.maven.model.building.ModelSource;
import org.apache.maven.model.building.ModelSource2;
import org.apache.maven.model.resolution.ModelResolver;
import org.apache.maven.model.resolution.UnresolvableModelException;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.rascalmpl.uri.URIUtil;
import org.rascalmpl.values.IRascalValueFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;

import io.usethesource.vallang.IListWriter;
import io.usethesource.vallang.ISourceLocation;
import io.usethesource.vallang.IValue;
import io.usethesource.vallang.IValueFactory;

public class MavenParser {
    private static final IValueFactory VF = IRascalValueFactory.getInstance();

    private final Path projectPom;
    private final ISourceLocation projectPomLocation;
    private final ModelBuilder builder;
    private final HttpClient httpClient;
    private final ModelCache modelCache;
    private final List<IValue> settingMessages;

    private final SimpleResolver rootResolver;

    public MavenParser(Path projectPom) {
        this(MavenSettings.readSettings(), projectPom);
    }

    /*package*/ MavenParser(MavenSettings settings, Path projectPom) {
        this(settings, projectPom, settings.getLocalRepository());
    }

    /*package*/ MavenParser(MavenSettings settings, Path projectPom, Path rootMavenRepo) {
        this.projectPom = projectPom;
        try {
            this.projectPomLocation = URIUtil.createFileLocation(projectPom);
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException("Project pom is an illegal path", e);
        }

        settingMessages = new ArrayList<>();

        var proxySelector = new MavenProxySelector(settings.getProxies(), settingMessages);

        builder = new DefaultModelBuilderFactory().newInstance();
        httpClient = HttpClient.newBuilder()
            .version(Version.HTTP_2) // upgrade where possible
            .connectTimeout(Duration.ofSeconds(10))
            .proxy(proxySelector)
            .build();

        modelCache = new CaffeineModelCache();

        rootResolver = SimpleResolver.createRootResolver(rootMavenRepo, httpClient, settings.getMirrors(), settings.getServerSettings());
    }

    public Artifact parseProject() throws ModelResolutionError {
        var request = new DefaultModelBuildingRequest()
            .setPomFile(projectPom.toFile());

        var resolver = rootResolver.createChildResolver();

        var messages = VF.listWriter();
        messages.appendAll(settingMessages);

        var model = getBestModel(projectPomLocation, request, resolver, messages);
        if (model == null) {
            throw new ModelResolutionError(messages);
        }

        Artifact result = null;
        if (model.getGroupId() != null && model.getVersion() != null) {
            result = Artifact.build(model, null, true, projectPom, projectPomLocation, "", Collections.emptySet(), messages, resolver);
        }

        if (result == null) {
            return Artifact.unresolved(new ArtifactCoordinate(Objects.requireNonNullElse(model.getGroupId(), ""), model.getArtifactId(), Objects.requireNonNullElse(model.getVersion(), ""), ""), null, messages);
        }
        return result;
    }

    /*package*/ @Nullable Artifact parseArtifact(ArtifactCoordinate coordinate, Set<ArtifactCoordinate.WithoutVersion> exclusions, Dependency origin, SimpleResolver originalResolver) {
        var messages = VF.listWriter();
        try {
            var modelSource = originalResolver.resolveModel(coordinate);
            var pomLocation = calculateLocation(modelSource);
            var pomPath = Path.of(pomLocation.getURI());

            var resolver = originalResolver.createChildResolver();

            // we need to use the original resolver to be able to resolve parent poms
            var workspaceResolver = new SimpleWorkspaceResolver(originalResolver, builder, this);

            var request = new DefaultModelBuildingRequest()
                .setModelSource(modelSource)
                .setWorkspaceModelResolver(workspaceResolver); // only for repository poms do we setup this extra resolver to help find parent poms

            var model = getBestModel(pomLocation, request, resolver, messages);
            if (model == null) {
                return Artifact.unresolved(coordinate, origin, messages);
            }

            return Artifact.build(model, origin, false, pomPath, pomLocation, coordinate.getClassifier(), exclusions, messages, resolver);
        } catch (UnresolvableModelException e) {
            messages.append(MavenMessages.error("Could not resolve " + coordinate + ". " + e.getMessage(), origin));
            return Artifact.unresolved(coordinate, origin, messages);
        }
    }

    private static ISourceLocation calculateLocation(ModelSource source) {
        try {
            URI loc;
            if (source instanceof ModelSource2) {
                loc = ((ModelSource2)source).getLocationURI();
            }
            else {
                loc = new URI(source.getLocation());
            }
            return VF.sourceLocation(URIUtil.fixUnicode(loc));
        }
        catch (URISyntaxException e) {
            return URIUtil.unknownLocation();
        }
    }


    private @Nullable Model getBestModel(ISourceLocation pom, ModelBuildingRequest request, ModelResolver resolver, IListWriter messages) {
        try {
            var result = buildModel(request, resolver);
            translateProblems(result.getProblems(), pom, messages);
            return result.getEffectiveModel();
        } catch (ModelBuildingException be) {
            translateProblems(be.getProblems(), pom, messages);
            return be.getModel();
        } 
    }

    private void translateProblems(List<ModelProblem> problems, ISourceLocation loc, IListWriter messages) {
        for (var problem : problems) {
            String message = problem.getMessage();
            // If maven has no coordinates to report, it uses -1, -1 for line and column
            int line = Math.max(0, problem.getLineNumber());
            int column = Math.max(0, problem.getColumnNumber());
            switch (problem.getSeverity()) {
                case ERROR: // fall through
                case FATAL: messages.append(MavenMessages.error(message, loc, line, column)); break;
                case WARNING: messages.append(MavenMessages.warning(message, loc, line, column)); break;
                default: throw new UnsupportedOperationException("Missing case: " + problem.getSeverity());
            }
        }
    }

    public Model buildEffectiveModel(ModelBuildingRequest request, ModelResolver resolver) throws ModelBuildingException {
        return buildModel(request, resolver).getEffectiveModel();
    }


    private ModelBuildingResult buildModel(ModelBuildingRequest request, ModelResolver resolver) throws ModelBuildingException {
        request.setModelResolver(resolver)
            .setModelCache(modelCache)
            .setLocationTracking(true)
            .setValidationLevel(ModelBuildingRequest.VALIDATION_LEVEL_MINIMAL)
            .setSystemProperties(System.getProperties());
        return builder.build(request);
    }

    private static final class CaffeineModelCache implements ModelCache {
        private static final class Key {
            private final String groupId; 
            private final String artifactId;
            private final String version;
            private final String tag;

            public Key(String groupId, String artifactId, String version, String tag) {
                this.groupId = groupId;
                this.artifactId = artifactId;
                this.version = version;
                this.tag = tag;
            }

            @Override
            public int hashCode() {
                return groupId.hashCode()
                    + (artifactId.hashCode() * 7)
                    + (version.hashCode() * 11)
                    + (tag.hashCode() * 13)
                    ;
            }

            @Override
            public boolean equals(Object obj) {
                if (this == obj)
                    return true;
                if (!(obj instanceof Key))
                    return false;
                Key other = (Key) obj;
                return groupId.equals(other.groupId)
                    && artifactId.equals(other.artifactId)
                    && version.equals(other.version)
                    && tag.equals(other.tag);
            }
        }

        private final Cache<Key, Object> modelCache = Caffeine.newBuilder()
            .maximumSize(100)
            .build();


        @Override
        public void put(String groupId, String artifactId, String version, String tag, Object data) {
            modelCache.put(new Key(groupId, artifactId, version, tag), data);
        }

        @Override
        public Object get(String groupId, String artifactId, String version, String tag) {
            return modelCache.getIfPresent(new Key(groupId, artifactId, version, tag));
        }
    }
    

    private static void test(Path target) throws ModelResolutionError {
        var start = System.currentTimeMillis();
        var parser = new MavenParser(new MavenSettings(), target);
        var project = parser.parseProject();
        var stop = System.currentTimeMillis();
        var out = new PrintWriter(System.out);
        project.report(out);
        out.printf("It took %d ms to resolve root artifact%n", stop - start);
        start = System.currentTimeMillis();
        var deps = project.resolveDependencies(Scope.COMPILE, parser);
        stop = System.currentTimeMillis();
        out.println(deps);
        out.printf("It took %d ms to resolve dependencies%n", stop - start);
        out.flush();
    }

    public static void main(String[] args) {
        try {
            test(Path.of("C:/Users/Davy/swat.engineering/rascal/rascal/test/org/rascalmpl/util/maven/poms/multi-module/example-core/pom.xml"));

            System.out.println("******");
            System.out.println("******");
            System.out.println("******");

            test(Path.of("pom.xml").toAbsolutePath());
        } catch (Throwable t) {
            System.err.println("Caught: " +t);
            t.printStackTrace();
        }
    }

}
