---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://077cd3d2-1ba3-4b87-b3c7-2a9018b816a4|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <5,25,|uuid://4c2e4aab-4609-429c-9661-41e7bfee486c|>,
  <10,100,|uuid://ee1cf0d0-86c7-45ce-9345-0bc36c988fa9|>,
  <7,49,|uuid://62c368db-fa14-4144-a0a6-59fe9c665b67|>,
  <6,36,|uuid://98fa836d-6c18-4d42-9352-d01157fe60c3|>,
  <8,64,|uuid://4311c4d7-1181-4099-a588-ab06880c0a14|>,
  <2,4,|uuid://34034410-5608-4ca7-b4d9-d224dc7d2339|>,
  <3,9,|uuid://c17320c7-b4d9-426d-b451-2a90ffe90593|>,
  <4,16,|uuid://cfeeb221-8cd9-4585-b1ed-aeb17cd0295e|>,
  <1,1,|uuid://d0825842-8cb8-4d38-809c-84cab457f7ce|>,
  <9,81,|uuid://bd5d42b1-5fc8-4c52-af83-7cb48e88581a|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://faf0ee8d-a600-461d-9fb4-0999c61de578|,
  <2,4>:|uuid://6bf56222-85a4-4178-9e3b-f70ee9b6314a|,
  <7,49>:|uuid://04c1cb75-e0ce-4d92-a595-047eb75fab07|,
  <9,81>:|uuid://74a68ea2-9cc7-4d2f-8bf6-8d8d8fe23615|,
  <8,64>:|uuid://69fae0bc-3821-474e-9f8f-60fc9ed5a8af|,
  <5,25>:|uuid://2d04b5a2-0de7-4eae-9819-0dcf589235f5|,
  <4,16>:|uuid://918e1861-7e80-4189-b9ba-96889c385e6e|,
  <1,1>:|uuid://f24c584e-99ac-4360-8ccd-e7ce743bab26|,
  <10,100>:|uuid://3797243a-1064-4b02-94c5-9b076e26a8cc|,
  <3,9>:|uuid://f4387375-c7fa-41d6-b008-2f84cedaa976|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

