---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://8bf77871-cf4a-44f8-bd09-db30e4e6e905|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <3,9,|uuid://7251e5a2-34bc-4dbe-abc4-a42e34b7f41a|>,
  <1,1,|uuid://e42a46bb-82ef-4023-a912-61757f282cf5|>,
  <10,100,|uuid://b7c11965-afb0-4e4e-8d4a-f3e37df9cf45|>,
  <4,16,|uuid://0d1b1f48-62db-4423-b263-d87bf80ecc1d|>,
  <2,4,|uuid://3d7cda51-0eae-4b27-a9cb-3e9c365fc94a|>,
  <5,25,|uuid://6243f585-2f41-4241-a904-9321ababb953|>,
  <9,81,|uuid://3b3c21a3-238d-42df-bdb7-bee9f4e60072|>,
  <7,49,|uuid://99b1185b-d0dc-4da1-bfeb-543eb7f4ac6c|>,
  <6,36,|uuid://8453d879-396e-4e51-ba85-72d22a73c4b5|>,
  <8,64,|uuid://548a5f38-dc1f-4f94-8313-204879b07f23|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://71646033-6b0a-440f-a0f7-f004cf48a4ac|,
  <2,4>:|uuid://c1d1c413-b28f-49ab-97e5-cec77163dee8|,
  <7,49>:|uuid://6fb08105-75e4-4466-a57a-94891bbcefeb|,
  <9,81>:|uuid://1af2392b-f8f2-4b5d-9067-493ad6c4dbe1|,
  <8,64>:|uuid://127aa10a-6052-44f3-9f93-575816d36521|,
  <5,25>:|uuid://0888e166-dcd8-46cf-9b36-41c3a5b63802|,
  <4,16>:|uuid://d528dd2f-10e7-4b1b-881b-7000d5294067|,
  <1,1>:|uuid://a964ba8c-5585-44eb-ab3a-6265564ed1cd|,
  <10,100>:|uuid://99c72bc1-3da4-4b19-86a6-3266e056f21b|,
  <3,9>:|uuid://a61baf66-03e0-4a66-ba55-52ae86d54621|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

