---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://8b6e802d-e1b7-47e4-bc7b-2b7339f7897a|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <2,4,|uuid://e09aa9a7-e24b-4550-b619-7ce50e606a70|>,
  <10,100,|uuid://6f0cdd6f-4908-4e87-81b4-d45b3bec5970|>,
  <7,49,|uuid://408a512e-d635-42cc-b6bb-38c2550a852c|>,
  <1,1,|uuid://c795813c-122e-40d7-8252-00adf725e0ff|>,
  <8,64,|uuid://93a8d323-df63-4346-b830-560f1c2592b6|>,
  <3,9,|uuid://48824d3d-e3ba-4be9-810c-2520d4d697e9|>,
  <6,36,|uuid://cdba68cd-9e67-40ab-a1f0-caa03c4a3ea2|>,
  <9,81,|uuid://8622b579-78e4-4554-8d0b-ddea3fb4f3f3|>,
  <4,16,|uuid://43a07ee3-cf89-4266-a2b0-ae2f7bb718e5|>,
  <5,25,|uuid://27d452b1-0b0a-45c5-9b2b-622331de5333|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://040f19af-7958-4965-91d2-3a74f24c75f2|,
  <2,4>:|uuid://b3475fd4-5a6c-46bf-847e-b720e79ea527|,
  <7,49>:|uuid://3973c8a8-3fe7-4f10-91e3-333abdc54bef|,
  <9,81>:|uuid://08c69345-fa42-41fe-98e8-365792f5cc30|,
  <8,64>:|uuid://c52c7b34-1b49-413f-82bc-64bf09ca99d3|,
  <5,25>:|uuid://a572f317-d5e9-410e-8602-c17b583421b3|,
  <4,16>:|uuid://cab4cf01-3464-4fd4-bd89-8424254a6c90|,
  <1,1>:|uuid://e0f61bc2-7be1-41d4-91a3-18334bf4ae69|,
  <10,100>:|uuid://4d13ad1c-1c9e-4436-adcf-c0518a6c717f|,
  <3,9>:|uuid://8a0b93b4-3bf3-48ae-a147-4ce55d35cb30|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

