---
title: "module util::Validator"
id: Validator
slug: /Library/util/Validator
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

Generic validator function that can convert values of the `node` type to instances of abstract `data` type constructors.
#### Usage

```rascal
import util::Validator;
```

#### Dependencies
```rascal
import Type;
import Node;
import List;
import Exception;
import IO;
```

#### Description


The intended use-case is to read structured data externally, say an XML or JSON or YAML file, as generic `node` values and then
to use the `validate` function to map the untyped representation to a typed representation, if it can be validated accordingly.


## data RuntimeException {#util-Validator-RuntimeException}

```rascal
data RuntimeException  
     = invalid(str \type, value v, list[value] path=[])
     ;
```

## data RuntimeException {#util-Validator-RuntimeException}

```rascal
data RuntimeException  
     = none()
     ;
```

## function validate {#util-Validator-validate}

The general and simple validation case is when a value's run-time type already matches the expected static type

```rascal
&T validate(type[&T] expected, value v, list[value] path=[], bool relaxed=false)
```

## function validate {#util-Validator-validate}

To validate nodes we can try whether or not it can be matched to a constructor of a defined data type with the same name and (resp. validating) children.

```rascal
&T validate(type[&T] expected, node v, list[value] path = [], bool relaxed=false)
```

## function validate {#util-Validator-validate}

if a (sub)value can not be validated we report the expected type, the not-matching value and the path that led us there

```rascal
default &T validate(type[&T] expected, value v, list[value] path=[], bool relaxed=false)
```

# Tests
## test simpleInt {#util-Validator-simpleInt}

```rascal
test bool simpleInt() {
    value x = 1;
  
    return int _ := validate(#int, x);
}
```

## test defaultNode {#util-Validator-defaultNode}

```rascal
test bool defaultNode() {
    value x = "hello"();
    
    return node _ := validate(#node, x);
}
```

## test adtTest {#util-Validator-adtTest}

```rascal
test bool adtTest() {
   value x = "invalid"("XXX", [[[]]],path=[1,0,0]);
   
   return RuntimeException _ := validate(#RuntimeException, x);
}
```

## test adtRelaxedTest {#util-Validator-adtRelaxedTest}

```rascal
test bool adtRelaxedTest() {
   value x = "object"("XXX", [[[]]],path=[1,0,0]);
   
   return RuntimeException _ := validate(#RuntimeException, x, relaxed=true);
}
```

## test adtTestFail {#util-Validator-adtTestFail}

```rascal
test bool adtTestFail() {
  value x = "invali"("XXX", [[[]]],path=[1,0,0]);
  
  try {
      validate(#RuntimeException, x);
      return false;
  }
  catch invalid(_,_) : 
      return true;
  
}
```

## test adtTestFailNested {#util-Validator-adtTestFailNested}

```rascal
test bool adtTestFailNested() {
  value x = "invalid"(2, [[[]]],path=[1,0,0]);
  
  try {
      validate(#RuntimeException, x);
      return false;
  }
  catch invalid(_,_) : 
      return true;
  
}
```

## test adtTestFailKeyword {#util-Validator-adtTestFailKeyword}

```rascal
test bool adtTestFailKeyword() {
  value x = "invalid"("hello", [[[]]],path="[1,0,0]");
  
  try {
      validate(#RuntimeException, x);
      return false;
  }
  catch invalid(_,_) : 
      return true;
  
}
```

