---
title: "module lang::paths::Unix"
id: Unix
slug: /Library/lang/paths/Unix
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

Defines the syntax of filesystem and network drive paths on DOS and Windows Systems.
#### Usage

```rascal
import lang::paths::Unix;
```

#### Dependencies
```rascal
import ParseTree;
```

#### Description


This syntax definition of POSIX paths and file names, including some of the conventions 
with ~ for the home folder and . and .. for relative directories.

The main function of this module, [Parse Unix Path](../../..//Library/lang/paths/Unix.md#lang-paths-Unix-parseUnixPath):
* faithfully maps any syntactically correctly Unix paths to syntactically correct `loc` values.
* throws a ParseError if the path does not comply. 
* ensures that if the file exists on system A, then the `loc` representation
resolves to the same file on system A via any [IO](../../..//Library/IO.md) function. 
* and nothing more. No normalization, no interpretatioon of `.` and `..`, no changing of cases. 
This is left to downstream processors of `loc` values, if necessary. The current transformation
is purely syntactical, and tries to preserve the semantics of the path as much as possible.

#### Pitfalls


* the `~` notation is typically a feature of the shell and not of system paths. However it is so commonly
used to refer to the  home directories of users that we've added an interpretation here with the `home:///` scheme.
* otherwise, the path syntax may be different from what you have to type in _bash_ or _zsh_. This is because shells
need to reserve characters, like spaces, for different purposes (commandline argument separation). The 
current definition is about the path notation that shells like _zsh_ and _bash_, and other programs, have to pass into the string arguments of
OS features, after their own concatenation, splicing, variable expansion, de-escaping and unquoting routines have finished.. 


## syntax UnixPath {#lang-paths-Unix-UnixPath}

```rascal
lexical UnixPath
    = absolute: Slashes UnixFilePath?
    | relative: UnixFilePath 
    | home    : "~" (Slashes UnixFilePath)?
    | user    : "~" UserName uname (Slashes UnixFilePath)?
    ;
```

## syntax UserName {#lang-paths-Unix-UserName}

```rascal
lexical UserName = ![/~]+;
```

## syntax PathChar {#lang-paths-Unix-PathChar}

```rascal
lexical PathChar = ![/];
```

## syntax PathSegment {#lang-paths-Unix-PathSegment}

```rascal
lexical PathSegment
    = current: "."
    | parent : ".."
    | pname  : (PathChar \ "~" PathChar*) \ ".." \ "." \ "~"
    ;
```

## syntax Slashes {#lang-paths-Unix-Slashes}

```rascal
lexical Slashes = Slash+ !>> [/];
```

## syntax Slash {#lang-paths-Unix-Slash}

```rascal
lexical Slash = [/];
```

## syntax UnixFilePath {#lang-paths-Unix-UnixFilePath}

```rascal
lexical UnixFilePath = {PathSegment Slashes}+ segments Slashes?;
```

## function parseUnixPath {#lang-paths-Unix-parseUnixPath}

Convert a Unix path literal to a source location URI

```rascal
loc parseUnixPath(str input, loc src=|unknown:///|)
```


1. parses the path using the grammar for [Unix Path](../../..//Library/lang/paths/Unix.md#lang-paths-Unix-UnixPath)
2. takes the _literal_ name components using string interpolation `"<segment>"`. This means no decoding/encoding happens at all while extracting
hostname, share name and path segment names. Also all superfluous path separators are skipped.
3. uses `loc + str` path concatenation with its builtin character encoding to construct the URI. Also
the right path separators are introduced. 

This conversion supports generic Unix path syntax, including:
* Absolute: `/usr/local/bin`
* Relative: `hello.txt`
* Home: `~/hello.txt`
* User: `~userName\hello.txt`

## function mapPathToLoc {#lang-paths-Unix-mapPathToLoc}

Root is a special case

```rascal
loc mapPathToLoc((UnixPath) `<Slashes _>`)
```

## function mapPathToLoc {#lang-paths-Unix-mapPathToLoc}

Absolute: given the drive and relative to its root.

```rascal
loc mapPathToLoc((UnixPath) `<Slashes _><UnixFilePath path>`)
```

## function mapPathToLoc {#lang-paths-Unix-mapPathToLoc}

Relative: relative to the current working directory.

```rascal
loc mapPathToLoc((UnixPath) `<UnixFilePath path>`)
```

## function mapPathToLoc {#lang-paths-Unix-mapPathToLoc}

Home relative: relative to the current users home directory

```rascal
loc mapPathToLoc((UnixPath) `~<Slash _><UnixFilePath path>`)
```

## function mapPathToLoc {#lang-paths-Unix-mapPathToLoc}

Home relative: relative to the current users home directory

```rascal
loc mapPathToLoc((UnixPath) `~`)
```

## function mapPathToLoc {#lang-paths-Unix-mapPathToLoc}

User relative: relative to any specific user's home directory

```rascal
loc mapPathToLoc((UnixPath) `~<UserName uname><Slash _><UnixFilePath path>`)
```

## function mapPathToLoc {#lang-paths-Unix-mapPathToLoc}

User relative: relative to any specific user's home directory

```rascal
loc mapPathToLoc((UnixPath) `~<UserName uname>`)
```

## function appendPath {#lang-paths-Unix-appendPath}

```rascal
loc appendPath(loc root, UnixFilePath path)
```

# Tests
## test root {#lang-paths-Unix-root}

```rascal
test bool root()
    = parseUnixPath("/") == |file:///|;
```

## test absolutePath {#lang-paths-Unix-absolutePath}

```rascal
test bool absolutePath()
    = parseUnixPath("/usr/local/bin")
    == |file:///usr/local/bin|;
```

## test relativePath {#lang-paths-Unix-relativePath}

```rascal
test bool relativePath()
    = parseUnixPath(".bash_rc")
    == |cwd:///.bash_rc|;
```

## test homePath {#lang-paths-Unix-homePath}

```rascal
test bool homePath()
    = parseUnixPath("~/.bash_profile")
    == |home:///.bash_profile|;
```

## test userPath {#lang-paths-Unix-userPath}

```rascal
test bool userPath()
    = parseUnixPath("~root/.bash_profile")
    == |home:///../root/.bash_profile|;
```

