---
title: "module util::Sampling"
id: Sampling
slug: /Library/util/Sampling
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

Utilities to randomly select smaller datasets from larger datasets
#### Usage

```rascal
import util::Sampling;
```

#### Dependencies
```rascal
import util::Math;
import Map;
import List;
import Set;
```

#### Description


Sampling is important when the analysis algorithms do not scale to the size of 
the original corpus, or when you need to train an analysis on a representative
set without overfitting on the entire corpus. These sampling functions all
assume that a uniformly random selection is required.


## function sample {#util-Sampling-sample}

Reduce the arity of a set by selecting a uniformly distributed sample.

```rascal
set[&T] sample(set[&T] corpus, int target)
```


A uniform subset is computed by iterating over the set and skipping every element
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new set of
expected `target` size, but most probably a little smaller or larger.

#### Examples



```rascal-shell 
rascal>import util::Sampling;
ok
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"c","e","f","h","k"}
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"a","b","h","i"}
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"a","i","k"}
```

## function sample {#util-Sampling-sample}

Reduce the length of a list by selecting a uniformly distributed sample.

```rascal
list[&T] sample(list[&T] corpus, int target)
```


The random selection of elements does not change their initial order in the list.
A uniform sublist is computed by iterating over the list and skipping every element
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new list of
expected `target` size, but most probably a little smaller or larger.

#### Examples



```rascal-shell 
rascal>import util::Sampling;
ok
rascal>sample([1..1000], 30)
list[int]: [1,116,230,242,295,296,300,306,318,330,386,424,457,479,550,553,563,568,604,636,639,739,790,796,801,836,886,924,948,991]
rascal>sample([1..1000], 30)
list[int]: [28,48,52,83,84,87,101,107,109,167,180,225,226,247,277,285,307,320,354,370,417,423,523,546,559,602,612,625,653,685,745,747,784,813,822,829,835,843,849,871,966,982]
rascal>sample([1..1000], 30)
list[int]: [5,19,71,118,119,120,147,158,197,273,343,379,392,469,520,525,569,617,646,687,695,703,707,718,733,745,746,782,783,861,917,921,958]
```

## function sample {#util-Sampling-sample}

Reduce the size of a map by selecting a uniformly distributed sample.

```rascal
map[&T,&U] sample(map[&T,&U] corpus, int target)
```


A uniform submap is computed by iterating over the map's keys and skipping every key
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new map of
expected `target` size, but most probably a little smaller or larger.

