---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://3c888c93-5034-4521-bf12-79fe7a01fe14|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <5,25,|uuid://3aa18fba-fc5f-4c1c-b594-beb0baac843b|>,
  <10,100,|uuid://868d6b4f-b5de-480d-88d0-3968cd8597e1|>,
  <2,4,|uuid://612c6798-6ca1-440a-bc78-aba4c072583d|>,
  <9,81,|uuid://04c76cb8-1e83-4e74-8a27-fbc8a157a008|>,
  <6,36,|uuid://42787470-17e6-4907-83df-0e911610bc36|>,
  <1,1,|uuid://0fc02966-e26a-4f5e-8ac8-0917796fc652|>,
  <4,16,|uuid://28ad0b05-b6ff-486e-acdb-1bc6ed0039ae|>,
  <7,49,|uuid://80362ba6-6b41-45c0-8015-36b22b454121|>,
  <8,64,|uuid://3a7d7130-9a08-4d17-b8dd-91f9a2f65a91|>,
  <3,9,|uuid://0dada745-4924-4ec5-bd42-86c5fdb2a359|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://a6bd2249-c094-42c6-b335-239a7bb92296|,
  <2,4>:|uuid://23545271-5309-4b74-abcb-8984ab8d3215|,
  <7,49>:|uuid://c008b63f-dd8f-4666-884d-96b0927311de|,
  <9,81>:|uuid://6b917086-eb28-455f-b476-6d099c8491be|,
  <8,64>:|uuid://e70fa22a-d176-43e8-a7e5-88a9c883e630|,
  <5,25>:|uuid://63e4ae4e-9645-4a30-ae5e-ec185a9c77bc|,
  <4,16>:|uuid://b7236ac8-dbc1-4706-8258-23ac6844738f|,
  <1,1>:|uuid://03ffd253-4e22-49fb-9623-de427c7350e0|,
  <10,100>:|uuid://72d337c7-98ac-4ea2-be23-77783ffa23ad|,
  <3,9>:|uuid://fb940add-0722-43fa-9ccd-30a0f273fa7b|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

