---
title: "module Content"
id: Content
slug: /Library//Content
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

Content provides access to the content server of the Rascal terminal for viewing interactive HTML output.
#### Usage

```rascal
import Content;
```

#### Dependencies
```rascal
import lang::json::IO;
```


## data Content {#Content-Content}
Content wraps the HTTP Request/Response API to support interactive visualization types
on the terminal.

```rascal
data Content  
     = content(str id, Response (Request) callback, str title = id, ViewColumn viewColumn = normalViewColumn(1))
     | content(Response response, str title="*static content*", ViewColumn viewColumn = normalViewColumn(1))
     ;
```


Values wrapped in a `Content` wrapper will be displayed by interactive
Rascal applications such as the IDE, the REPL terminal and the documentation pages. 


For example, a piece of html can be displayed directly like such:

```rascal-shell ,continue
rascal>html("\<a href=\"http://www.rascal-mpl.org\"\>Rascal homepage\</a\>")
```
![image](/assets/Library/Content_screenshot_19.png)
```rascal-shell
```

In its most general form, `Content` is an HTTP(s) webserver callback, such that you might deliver
any kind of content, based on any kind of request. If you produce a `Content` value
which processes requests dynamically, subsequent interaction from the web browser will be 
processed as well. So using the `Content` wrapper you can start an interactive user experience
in the browser directly from the REPL. 

Content values stay plugged into the application server that is hidden in the REPL 
environment until they have not been used for at least 30 minutes. If you want the same
interaction back after 30 minutes of non-usage, you have to produce another Content value 
on the commandline. 

When you are happy with the interaction, or you want a permanent visualization which is not
garbage collected after 30 minutes, you can consider wrapping the same callback in
a webserver using the [serve](..//Library/util/Webserver.md#util-Webserver-serve) function.

## function html {#Content-html}

Directly serve a static html page

```rascal
Content html(str html)
```

## function file {#Content-file}

Directly serve the contents of a file

```rascal
Content file(loc src)
```

## function plainText {#Content-plainText}

Directly serve the contents of a string as plain text

```rascal
Content plainText(str text)
```

## alias Body {#Content-Body}

```rascal
value (type[value] expected)
```

## data Request {#Content-Request}
Request values represent what a browser is asking for, most importantly the URL path.

```rascal
data Request (map[str, str] headers = (), map[str, str] parameters = (), map[str,str] uploads = ()) 
     = get (str path)
     | put (str path, Body content)
     | post(str path, Body content)
     | delete(str path)
     | head(str path)
     ;
```


A request value also contains the full HTTP headers, the URL parameters as a `map[str,str]`
and possibly uploaded content, also coded as a map[str,str]. From the constructor type,
`put` or `get` you can see what kind of HTTP request it was.

#### Pitfalls


* Note that `put` and `post` have not been implemented yet in the REPL server.

## data Response {#Content-Response}
A response encodes what is send back from the server to the browser client.

```rascal
data Response  
     = response(Status status, str mimeType, map[str,str] header, str content)
     | fileResponse(loc file, str mimeType, map[str,str] header)
     | jsonResponse(Status status, map[str,str] header, value val, str dateTimeFormat = "yyyy-MM-dd\'T\'HH:mm:ss\'Z\'", JSONFormatter[value] formatter = str (value _) { fail; }, bool explicitConstructorNames=false, bool explicitDataTypes=false)
     ;
```


The three kinds of responses, encode either content that is already a `str`,
some file which is streamed directly from its source location or a jsonResponse
which involves a handy, automatic, encoding of Rascal values into json values.

## function response {#Content-response}

Utility to quickly render a string as HTML content

```rascal
Response response(str content, map[str,str] header = ())
```

## function response {#Content-response}

Utility to quickly report an HTTP error with a user-defined message

```rascal
Response response(Status status, str explanation, map[str,str] header = ())
```

## function plain {#Content-plain}

Utility to quickly make a plaintext response.

```rascal
Response plain(str text)
```

## function response {#Content-response}

Utility to serve a file from any source location.

```rascal
Response response(loc f, map[str,str] header = ())
```

## function response {#Content-response}

Utility to quickly serve any rascal value as a json text.

```rascal
default  Response response(value val, map[str,str] header = ())
```

#### Benefits


This comes in handy for asynchronous HTTP requests from Javascript clients. Rascal Values are
fully transformed to their respective JSON serialized form before being communicated over HTTP.

## function response {#Content-response}

Utility to quickly serve any rascal value as a json text, formatting data-types on-the-fly using a `formatter` function

```rascal
Response response(value val, JSONFormatter[value] formatter, map[str,str] header = ())
```

#### Benefits


Fast way of producing JSON strings for embedded DSLs on the Rascal side.

## data Status {#Content-Status}
Encoding of HTTP status

```rascal
data Status  
     = ok()
     | created()
     | accepted()
     | noContent()
     | partialContent()
     | redirect()
     | notModified()
     | badRequest()
     | unauthorized()
     | forbidden()
     | notFound()
     | rangeNotSatisfiable()
     | internalError()
     ;
```

## alias ViewColumn {#Content-ViewColumn}

Hint the IDE where to open the next web view or editor

```rascal
int
```


The `viewColumn`  decides where in the IDE a web client or editor is opened,
_if_ the current IDE honors this parameter. 

There are _9_ possible
view columns: 1, 2, 3, 4, 5, 6, 7, 8, 9. 

Next to this:
* view column `-1` is converted to the _currently active_ view column before the editor is opened.
* view column `-2` is chosen to be a view column _beside_ (to the right) of the currently active view column.

All other view column integers are ignored and interpreted as `-1`.

## function activeViewColumn {#Content-activeViewColumn}

```rascal
ViewColumn activeViewColumn()
```

## function besideViewColumn {#Content-besideViewColumn}

```rascal
ViewColumn besideViewColumn()
```

## function normalViewColumn {#Content-normalViewColumn}

```rascal
ViewColumn normalViewColumn(int v)
```

