---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://d1a72db4-c259-4a6b-90a3-4580e1ab1a7c|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <10,100,|uuid://364a5b97-2dd2-4aa6-9f55-f261fd2e1d5d|>,
  <3,9,|uuid://333bcd98-0bc4-4310-8b92-a6dfb84ede1b|>,
  <8,64,|uuid://121de58f-91db-4447-bfb0-74c4e7db5fbd|>,
  <7,49,|uuid://f1d7d92f-d5dc-48d7-a87d-4d260e4cb5cc|>,
  <6,36,|uuid://8420a8b1-656a-47ab-bcb1-ab71b3384ec3|>,
  <2,4,|uuid://668c2849-d140-4234-bd33-cb891356640e|>,
  <9,81,|uuid://5d1f04ef-c2cb-4d09-a10f-058e58197a2a|>,
  <5,25,|uuid://c1c3260b-c79c-4ec3-abbd-7ce546e05e9b|>,
  <4,16,|uuid://cd4e700b-934e-4704-81dd-d2bf052f3a88|>,
  <1,1,|uuid://b858458c-cd58-4e30-ade7-fa34739fba4d|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://ec7a91e7-d3e0-4b85-b36e-4bd679b8f1f2|,
  <2,4>:|uuid://4a8b4ba6-8fd3-429c-a733-f51da77a0dd2|,
  <7,49>:|uuid://9f8c2a96-274c-4586-bbcd-13b133c47b39|,
  <9,81>:|uuid://0b80518b-0118-4ea6-851b-b72105122b15|,
  <8,64>:|uuid://7ac4dddd-c6bb-44bb-a2cd-adbe469346e1|,
  <5,25>:|uuid://5276391c-f3d6-45c0-939d-3d74d72a5c9f|,
  <4,16>:|uuid://e54153ea-fe32-4bc1-91d2-e22d36f0dd67|,
  <1,1>:|uuid://f9f3f92c-ea8e-49fe-b3a8-c8f60873d84a|,
  <10,100>:|uuid://461a0b8d-a1c3-458a-8843-1bcb9b1f2fa9|,
  <3,9>:|uuid://c5f231b7-f1ac-4c0e-aa84-8c40b43dd09d|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

