---
title: "module Map"
id: Map
slug: /Library//Map
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

Library functions for maps.
#### Usage

```rascal
import Map;
```



The following library functions are defined for maps:
* [delete](..//Library/Map.md#Map-delete)
* [domain](..//Library/Map.md#Map-domain)
* [domainR](..//Library/Map.md#Map-domainR)
* [domainX](..//Library/Map.md#Map-domainX)
* [getOneFrom](..//Library/Map.md#Map-getOneFrom)
* [invert](..//Library/Map.md#Map-invert)
* [invertUnique](..//Library/Map.md#Map-invertUnique)
* [isEmpty](..//Library/Map.md#Map-isEmpty)
* [itoString](..//Library/Map.md#Map-itoString)
* [mapper](..//Library/Map.md#Map-mapper)
* [range](..//Library/Map.md#Map-range)
* [rangeR](..//Library/Map.md#Map-rangeR)
* [rangeX](..//Library/Map.md#Map-rangeX)
* [size](..//Library/Map.md#Map-size)
* [toList](..//Library/Map.md#Map-toList)
* [toRel](..//Library/Map.md#Map-toRel)
* [toString](..//Library/Map.md#Map-toString)


## function delete {#Map-delete}

Delete a key from a map.

```rascal
map[&K,&V] delete(map[&K,&V] m, &K k)
```


Returns the map `m` minus the key `k`.

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>delete(("apple":1,"pear":2), "apple");
map[str, int]: ("pear":2)
```

## function domain {#Map-domain}

Determine the domain (set of keys) of a map.

```rascal
set[&K] domain(map[&K, &V] M)
```


Returns the domain (set of keys) of map `M`.

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>domain(("apple": 1, "pear": 2));
set[str]: {"pear","apple"}
```

## function domainR {#Map-domainR}

Map restricted to certain keys.

```rascal
map[&K, &V] domainR(map[&K, &V] M, set[&K] S)
```


Return the map `M` restricted to pairs with key in `S`.

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>domainR(("apple": 1, "pear": 2, "orange": 3), {"apple", "pear"});
map[str, int]: ("pear":2,"apple":1)
```

## function domainX {#Map-domainX}

Map with certain keys excluded.

```rascal
map[&K, &V] domainX(map[&K, &V] M, set[&K] S)
```


Return the map `M` restricted to pairs with key not in `S`.

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>domainX(("apple": 1, "pear": 2, "orange": 3), {"apple", "pear"});
map[str, int]: ("orange":3)
```

## function getOneFrom {#Map-getOneFrom}

Get a n arbitrary key from a map.

```rascal
&K getOneFrom(map[&K, &V] M)
```


Returns an arbitrary key of map `M`.

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>getOneFrom(("apple": 1, "pear": 2, "pineapple": 3));
str: "pear"
───
pear
───
rascal>getOneFrom(("apple": 1, "pear": 2, "pineapple": 3));
str: "pineapple"
───
pineapple
───
rascal>getOneFrom(("apple": 1, "pear": 2, "pineapple": 3));
str: "pear"
───
pear
───
```

## function invert {#Map-invert}

Invert the (key,value) pairs in a map.

```rascal
map[&V, set[&K]] invert(map[&K, &V] M)
```


Returns inverted map in which each value in the old map `M` is associated with a set of key values from the old map.
Also see [Invert Unique](..//Library/Map.md#Map-invertUnique).

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>invert(("apple": 1, "pear": 2, "orange": 1));
map[int, set[str]]: (
  1:{"orange","apple"},
  2:{"pear"}
)
```

## function invertUnique {#Map-invertUnique}

Invert the (key,value) pairs in a map.

```rascal
map[&V, &K] invertUnique(map[&K, &V] M)
```


Returns a map with key and value inverted; the result should be a map.
If the initial map contains duplicate values,
the `MultipleKey` exception is raised since
an attempt is made to create a map where more than one 
value would be associated with the same key.

Also see [Invert](..//Library/Map.md#Map-invert) and [Exception](..//Library/Exception.md).

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>invertUnique(("apple": 1, "pear": 2, "orange": 3));
map[int, str]: (1:"apple",3:"orange",2:"pear")
```
Here is an examples that generates an exception:

```rascal-shell ,continue,error
rascal>invertUnique(("apple": 1, "pear": 2, "orange": 1));
|std:///Map.rsc|(2659,708,<109,0>,<130,54>): MultipleKey(1,"apple","orange")
	at *** somewhere ***(|std:///Map.rsc|(2659,708,<109,0>,<130,54>))
	at invertUnique(|prompt:///|(47,1,<1,47>,<1,48>))
```

## function isEmpty {#Map-isEmpty}

Test whether a map is empty.

```rascal
bool isEmpty(map[&K, &V] M)
```


Returns `true` if map `M` is empty, and `false` otherwise.

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>isEmpty(());
bool: true
rascal>isEmpty(("apple": 1, "pear": 2, "orange": 3));
bool: false
```

## function mapper {#Map-mapper}

Apply a function to all (key, value) pairs in a map.

```rascal
map[&L, &W] mapper(map[&K, &V] M, &L (&K) F, &W (&V) G)
```


Apply the functions `F` and `G` to each key/value pair in a map and return the transformed map.

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>str prefix(str s) { return "X" + s; }
str (str): function(|prompt:///|(0,37,<1,0>,<1,37>))
rascal>int incr(int x) { return x + 1; }
int (int): function(|prompt:///|(0,33,<1,0>,<1,33>))
rascal>mapper(("apple": 1, "pear": 2, "orange": 3), prefix, incr);
map[str, int]: ("Xapple":2,"Xorange":4,"Xpear":3)
```

## function range {#Map-range}

The range (set of values that correspond to its keys) of a map.

```rascal
set[&V] range(map[&K, &V] M)
```


Returns the range (set of values) of map `M`.

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>range(("apple": 1, "pear": 2));
set[int]: {1,2}
```

## function rangeR {#Map-rangeR}

Map restricted to certain values in (key,values) pairs.

```rascal
map[&K, &V] rangeR(map[&K, &V] M, set[&V] S)
```


Returns the map restricted to pairs with values in `S`.

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>rangeR(("apple": 1, "pear": 2, "orange": 3), {2, 3});
map[str, int]: ("pear":2,"orange":3)
```

## function rangeX {#Map-rangeX}

Map with certain values in (key,value) pairs excluded.

```rascal
map[&K, &V] rangeX(map[&K, &V] M, set[&V] S)
```


Returns the map restricted to pairs with values not in `S`.

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>rangeX(("apple": 1, "pear": 2, "orange": 3), {2, 3});
map[str, int]: ("apple":1)
```

## function size {#Map-size}

Number of (key, value) pairs in a map.

```rascal
int size(map[&K, &V] M)
```


Returns the number of pairs in map `M`.

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>size(("apple": 1, "pear": 2, "orange": 3));
int: 3
```

## function toList {#Map-toList}

Convert a map to a list of tuples.

```rascal
list[tuple[&K, &V]] toList(map[&K, &V] M)
```

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>toList(("apple": 1, "pear": 2, "orange": 3));
lrel[str,int]: [
  <"apple",1>,
  <"orange",3>,
  <"pear",2>
]
```

## function toRel {#Map-toRel}

Convert a map to a relation.

```rascal
rel[&K,&V] toRel(map[&K, set[&V]] M)

rel[&K,&V] toRel(map[&K, list[&V]] M)

default java rel[&K, &V] toRel(map[&K, &V] M)
```

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>toRel(("apple": 1, "pear": 2, "orange": 3));
rel[str,int]: {
  <"pear",2>,
  <"orange",3>,
  <"apple",1>
}
```

## function toString {#Map-toString}

Convert a map to a string.

```rascal
str toString(map[&K, &V] M)
```

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>toString(("apple": 1, "pear": 2, "orange": 3));
str: "(\"pear\":2,\"orange\":3,\"apple\":1)"
───
("pear":2,"orange":3,"apple":1)
───
```

## function itoString {#Map-itoString}

Convert a map to a indented string.

```rascal
str itoString(map[&K, &V] M)
```

#### Examples



```rascal-shell 
rascal>import Map;
ok
rascal>itoString(("apple": 1, "pear": 2, "orange": 3));
str: "(\"pear\":2,\"orange\":3,\"apple\":1)"
───
("pear":2,"orange":3,"apple":1)
───
```

