---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://2866c1ca-b8a8-4d92-a84d-f7175f224d68|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <5,25,|uuid://140a8593-f3fe-4d4b-8c25-768874649c75|>,
  <3,9,|uuid://8d49050f-99d3-4ddc-843d-537dd79fecf2|>,
  <2,4,|uuid://34f010cd-1ccb-4d94-aa65-214182b3b5ab|>,
  <7,49,|uuid://5b06888a-d9f1-47a5-b159-94d7d8cdf6d5|>,
  <6,36,|uuid://86141496-aab6-432e-af9c-7adee7937907|>,
  <10,100,|uuid://4b585994-825a-477d-b3ec-c0f8ffce945d|>,
  <8,64,|uuid://759b5662-4c8a-4a1c-a07d-ec103c2b0bfb|>,
  <4,16,|uuid://23d7d7ef-8810-41fe-87b4-6fcd5625c624|>,
  <9,81,|uuid://7ae3be4b-9800-4dc5-bf12-7440ddc0628e|>,
  <1,1,|uuid://ff99035b-111a-406f-b28d-c9c490905541|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://e4a1a864-dce2-4996-babd-71f358819282|,
  <2,4>:|uuid://b9beaf49-f0f4-41be-bd6c-bd070b8e539d|,
  <7,49>:|uuid://6a8e62c9-d6ee-437c-9f8a-0a55b85bdbb0|,
  <9,81>:|uuid://7360c71b-98bf-40b3-878f-dc410813f477|,
  <8,64>:|uuid://b1cb1883-f514-4d75-b9b8-c85a09062b7f|,
  <5,25>:|uuid://1365a458-4bba-4088-a65c-4069e29e4764|,
  <4,16>:|uuid://1e848912-25a4-4776-a2ba-4583fad95986|,
  <1,1>:|uuid://48bc5f7a-ed99-474e-99a2-3c29bc674e09|,
  <10,100>:|uuid://454ea309-8dd3-4077-b6d1-f436b0808bd5|,
  <3,9>:|uuid://2e719fb9-d6e4-4a7c-94a1-960dc38c8671|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

