---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://2a671859-98d3-40d5-86f2-27c3ed50c508|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <9,81,|uuid://09d54498-deac-4691-b753-bcb5cab51f43|>,
  <3,9,|uuid://86218be0-ce0b-45cc-b36a-269b09abf6cc|>,
  <6,36,|uuid://78ac8b4b-afdc-4098-a2e0-eb7377e94d9a|>,
  <2,4,|uuid://32342e2d-7c18-4d3e-bec8-9f83afc7f543|>,
  <1,1,|uuid://ce6c8233-8ce2-481a-8f74-e46344dce2ce|>,
  <10,100,|uuid://2dfaddbc-8002-42f5-b5a8-c6f6175b677e|>,
  <4,16,|uuid://cf362b8d-08ad-4659-9961-a5ccdb278e10|>,
  <8,64,|uuid://9c22d8e0-f324-4185-9096-31192b4b20c5|>,
  <7,49,|uuid://006cd4d8-bc61-4cdf-bade-b2fe3273773e|>,
  <5,25,|uuid://9f754ae8-53d4-4d12-835b-c15ae6fefd2c|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://18c6d6d1-14da-439c-86d0-6fbbd8a253ba|,
  <2,4>:|uuid://b1e523cd-82b6-4884-ad7d-c6104c367ac7|,
  <7,49>:|uuid://0613b289-b586-4800-9e65-0c0cf5422fc4|,
  <9,81>:|uuid://c4248581-df05-4179-8745-50d8261e17a3|,
  <8,64>:|uuid://2b82d57c-2d8b-467c-a6b3-2cfba59acdfc|,
  <5,25>:|uuid://c9dcf827-0902-438d-b493-cb3c658cbe44|,
  <4,16>:|uuid://2d9df1a1-6d00-4780-916d-f4edd446b7ed|,
  <1,1>:|uuid://8dfe09c9-915d-4756-ae46-d8612545d35c|,
  <10,100>:|uuid://5c5af67a-da9e-47ab-a4ce-a4cdf48a080c|,
  <3,9>:|uuid://e73a01fc-c1da-4d66-a0d0-4336f619004f|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

