---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://9fda45e4-2251-4cb2-944a-c68fcab6bc02|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <4,16,|uuid://dd5efba1-a9a0-457e-8b58-78ba0e1cb147|>,
  <7,49,|uuid://71b31c1a-5f3c-4bde-ac96-9ff4cd131999|>,
  <8,64,|uuid://1ca1ce54-190d-48b2-b316-5477973b94e1|>,
  <6,36,|uuid://fc5c98e5-db54-4133-b42b-2f73f834b101|>,
  <1,1,|uuid://9c6882ec-9b73-4188-a465-5d4fad9fcc6d|>,
  <9,81,|uuid://c8231a51-56da-4a86-aaf0-1a156a83a748|>,
  <3,9,|uuid://47021728-0655-4051-9913-2c6edea0024f|>,
  <5,25,|uuid://ec91ab8d-ccd1-4102-b999-e645eb1be9f8|>,
  <2,4,|uuid://fc4091f1-dc32-4b4f-a5c9-27d5ce09be03|>,
  <10,100,|uuid://469d51f7-bdb0-4b96-8c3a-41af10383180|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://13a65c9d-aa83-41a5-94b1-58848af52c66|,
  <2,4>:|uuid://4dcebcc0-1b41-4b83-95fe-4acbb4988238|,
  <7,49>:|uuid://731f8f64-0bfc-4f9e-9b7d-a8af07fe47f3|,
  <9,81>:|uuid://f42eec5c-edcb-4e17-93a2-0fe8eb70a264|,
  <8,64>:|uuid://fa8639b6-614a-4342-afaa-4c6014165a91|,
  <5,25>:|uuid://bf25b084-fb29-4e3b-b71e-86323ab8e595|,
  <4,16>:|uuid://684625d7-87b0-47f3-859e-e78022d1144c|,
  <1,1>:|uuid://cb8dcffd-2eb0-4b06-91ca-3ef3506c2fa2|,
  <10,100>:|uuid://47d7a211-ef28-4c8e-a767-8549fee0a94e|,
  <3,9>:|uuid://11020391-2129-49a0-afec-09ccaa149676|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

