---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://010aff64-2dc0-4515-b24c-35eaf9139089|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <5,25,|uuid://1e992fc0-4f47-43b1-ae49-21b8ed74f343|>,
  <2,4,|uuid://7e2e1502-8679-4619-9a97-12e64c96ccf6|>,
  <9,81,|uuid://c5588563-3055-47dc-b31d-d6d23b2d2048|>,
  <10,100,|uuid://5e22b06c-061c-4147-b10a-e46538382e24|>,
  <8,64,|uuid://71ae1821-549c-4c72-8a56-1edaa7816017|>,
  <4,16,|uuid://1d67d7c6-83f5-4a02-9876-6f45b0516736|>,
  <3,9,|uuid://571a92ef-266c-4c05-8b7d-577b387859bd|>,
  <1,1,|uuid://491a60d6-614e-4b5c-aa63-aef3d6d7293b|>,
  <6,36,|uuid://f7fecd3e-bebb-4ab2-8d94-630d9c831f4e|>,
  <7,49,|uuid://6d71541e-c9c8-461d-9261-f14546cb05d1|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://cec624ae-7cc9-4c8d-9250-9bc8b6bdda91|,
  <2,4>:|uuid://7ddbf4fc-f2c1-46f7-91f9-3041c7d0ccda|,
  <7,49>:|uuid://29148ee2-3d3b-4cb7-8f66-4637968b4f42|,
  <9,81>:|uuid://e6a8d6e6-9f28-4e67-bd7c-569aeae721e5|,
  <8,64>:|uuid://2cfd9ab6-b79b-4e24-b5b7-0543657d719f|,
  <5,25>:|uuid://5e833e35-cd53-431f-beaa-49b599f4ebd2|,
  <4,16>:|uuid://7a7d3013-e10b-4fdb-82e2-57f94afc7beb|,
  <1,1>:|uuid://0d3b2ebc-0f0e-4ce7-97c0-c9233f0e6f04|,
  <10,100>:|uuid://312d549e-a8cd-44fc-80c4-b8526f74ed5f|,
  <3,9>:|uuid://1fb76645-d96a-41fe-aa44-3954cbdc6ffa|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

