---
title: "module IO"
id: IO
slug: /Library//IO
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

Library functions for input/output.
#### Usage

```rascal
import IO;
```

#### Dependencies
```rascal
import Exception;
extend analysis::diff::edits::FileSystemChanges;
```

#### Description


The following input/output functions are defined:
* [classloading](..//Library/IO.md#IO-classloading)
* [reading](..//Library/IO.md#IO-reading)
* [resolving](..//Library/IO.md#IO-resolving)
* [watching](..//Library/IO.md#IO-watching)
* [writing](..//Library/IO.md#IO-writing)
* [IOCapability](..//Library/IO.md#IO-IOCapability)
* [appendToFile](..//Library/IO.md#IO-appendToFile)
* [appendToFileEnc](..//Library/IO.md#IO-appendToFileEnc)
* [arbLoc](..//Library/IO.md#IO-arbLoc)
* [bprintln](..//Library/IO.md#IO-bprintln)
* [canEncode](..//Library/IO.md#IO-canEncode)
* [capabilities](..//Library/IO.md#IO-capabilities)
* [charsets](..//Library/IO.md#IO-charsets)
* [copy](..//Library/IO.md#IO-copy)
* [copyDirectory](..//Library/IO.md#IO-copyDirectory)
* [copyFile](..//Library/IO.md#IO-copyFile)
* [createLink](..//Library/IO.md#IO-createLink)
* [created](..//Library/IO.md#IO-created)
* [exists](..//Library/IO.md#IO-exists)
* [find](..//Library/IO.md#IO-find)
* [findResources](..//Library/IO.md#IO-findResources)
* [getResource](..//Library/IO.md#IO-getResource)
* [iprint](..//Library/IO.md#IO-iprint)
* [iprintExp](..//Library/IO.md#IO-iprintExp)
* [iprintToFile](..//Library/IO.md#IO-iprintToFile)
* [iprintToString](..//Library/IO.md#IO-iprintToString)
* [iprintln](..//Library/IO.md#IO-iprintln)
* [iprintlnExp](..//Library/IO.md#IO-iprintlnExp)
* [isDirectory](..//Library/IO.md#IO-isDirectory)
* [isFile](..//Library/IO.md#IO-isFile)
* [isReadable](..//Library/IO.md#IO-isReadable)
* [isWritable](..//Library/IO.md#IO-isWritable)
* [lastModified](..//Library/IO.md#IO-lastModified)
* [listEntries](..//Library/IO.md#IO-listEntries)
* [md5Hash](..//Library/IO.md#IO-md5Hash)
* [md5HashFile](..//Library/IO.md#IO-md5HashFile)
* [mkDirectory](..//Library/IO.md#IO-mkDirectory)
* [move](..//Library/IO.md#IO-move)
* [print](..//Library/IO.md#IO-print)
* [printExp](..//Library/IO.md#IO-printExp)
* [println](..//Library/IO.md#IO-println)
* [printlnExp](..//Library/IO.md#IO-printlnExp)
* [readBase32](..//Library/IO.md#IO-readBase32)
* [readBase64](..//Library/IO.md#IO-readBase64)
* [readFile](..//Library/IO.md#IO-readFile)
* [readFileBytes](..//Library/IO.md#IO-readFileBytes)
* [readFileEnc](..//Library/IO.md#IO-readFileEnc)
* [readFileLines](..//Library/IO.md#IO-readFileLines)
* [readFileLinesEnc](..//Library/IO.md#IO-readFileLinesEnc)
* [registerLocations](..//Library/IO.md#IO-registerLocations)
* [remove](..//Library/IO.md#IO-remove)
* [rename](..//Library/IO.md#IO-rename)
* [resolveLocation](..//Library/IO.md#IO-resolveLocation)
* [rprint](..//Library/IO.md#IO-rprint)
* [rprintln](..//Library/IO.md#IO-rprintln)
* [setLastModified](..//Library/IO.md#IO-setLastModified)
* [toBase64](..//Library/IO.md#IO-toBase64)
* [touch](..//Library/IO.md#IO-touch)
* [unregisterLocations](..//Library/IO.md#IO-unregisterLocations)
* [unwatch](..//Library/IO.md#IO-unwatch)
* [uudecode](..//Library/IO.md#IO-uudecode)
* [uuencode](..//Library/IO.md#IO-uuencode)
* [watch](..//Library/IO.md#IO-watch)
* [writeBase32](..//Library/IO.md#IO-writeBase32)
* [writeBase64](..//Library/IO.md#IO-writeBase64)
* [writeFile](..//Library/IO.md#IO-writeFile)
* [writeFileBytes](..//Library/IO.md#IO-writeFileBytes)
* [writeFileEnc](..//Library/IO.md#IO-writeFileEnc)
* [writeFileLines](..//Library/IO.md#IO-writeFileLines)


## function registerLocations {#IO-registerLocations}

Register a logical file scheme including the resolution method via a table.

```rascal
void registerLocations(str scheme, str authority, map[loc logical, loc physical] m)
```


Logical source location schemes, such as `|java+interface://JRE/java/util/List|` are used for
precise qualified names of artifacts while abstracting from their physical location in a specific part
of a file on disk or from some webserver or source repository location.

Using this function you can create your own schemes. The authority field is used for scoping the 
names you wish to resolve to certain projects. This way one name can resolve to different locations 
in different projects.

#### Benefits


*  Logical source locations are supported by IDE features such as hyperlinks
*  Logical source locations are supported by all IO functions as well

#### Pitfalls


* Repeated calls to registerLocations for the same `scheme` and `authority` will overwrite the `m` map.
* The registry is an intentional memory leak; so make sure you use it wisely. See also [Unregister Locations](..//Library/IO.md#IO-unregisterLocations).
* When the files references by the physical locations are being written to (edited, removed), then you
may expect problems. The registry is not automatically invalidated.

## function unregisterLocations {#IO-unregisterLocations}

Undo the effect of ((registerLocations))

```rascal
void unregisterLocations(str scheme, str authority)
```


For debugging or for memory management you may wish to remove a lookup table.

## function resolveLocation {#IO-resolveLocation}

```rascal
loc resolveLocation(loc l)
```

## function findResources {#IO-findResources}

Finds files in the deployment locations of the configuration of the current runtime

```rascal
set[loc] findResources(str fileName)

set[loc] findResources(loc path)
```


* For the interpreter this looks at the source path and finds any file in there.
* For the compiler this looks at the classpath and finds resources in there.  

The goal of `findResources` is to access additional files that are distributed
with a Rascal program. For example, the `index.html` file that comes with a server implemented
using [Webserver](..//Library/util/Webserver.md) would be located using this function. Such an additional file is
essential to the functioning of the given Rascal code.

There is a difference between Rascal programs under development and Rascal programs deployed
(in a jar file). The idea is that calls to this function _behave the same_ regardless of this 
mode. To make this true, the implementation of `findResources` makes the following
assumptions:
* In interpreter mode, the additional files are stored relative to the source roots of the current project. 
This means that all source roots are searched as configured for the current interpreter. It is possible
that multiple files match the query, hence the result set. Because jar files and target folders are typically
added to the source roots by the configuration code for Rascal's IDEs, this works out smoothly also for deployed projects.
* In compiled mode, the additional files are stored relative to the roots of the classpath. This 
means they are jar files and target folders of active projects. Your Maven project configuration should
take care to _copy_ all relevant resources from source folders to target folders and to package those
relative to the root of the jar for this to work in development mode. For deployment mode, the setup 
works through the resources API of the JVM that uses the Java ClassLoader API.

#### Benefits


By satisfying the above two assumptions it is possible to write Rascal code that uses `findResources` that
is portable between interpreter and compiled mode, _and_ does not change between development
and deployed mode either.

#### Pitfalls


* Avoid nesting resources in sub-folders in jar files (see Maven resources plugin) since nested files will not be found.
* Inside jar files the root for module names should be the same as for resources
* Inside source folders, the names of resources should be relative to the same root as the Rascal module names
* `findResources` searches exhaustively for all files with the given name; which may come at an expensive for projects with
large search paths. It makes sense to keep this in a constant module variable that is initialized only once.
* Name clashes are bound to happen. Prepare to, at least, throw a meaningful error message in the case of name clashes. E.g. `findResources("index.html")`
is _very_ likely to produce more than one result. Either choose more unique names, or filter the result in a meaningful manner,
or throw an exception that explains the situation to the user of your code. Specifically library project maintainers have to consider
this happenstance. The recommendation is to nest resources next to qualified module names, like so: if `lang/x/myLanguage/Syntax.rsc` 
is a module name, then `lang/x/myLanguage/examples/myExampleFile.mL` is an example resource location.

## function getResource {#IO-getResource}

Search for a single resource instance, and fail if no or multiple instances are found

```rascal
loc getResource(str fileName) throws IO
```


This is a utility wrapper around [Find Resources](..//Library/IO.md#IO-findResources).
It processes the result set of [Find Resources](..//Library/IO.md#IO-findResources) to:
* return a singleton location if the `fileName`` was found.
* throw an IO exception if no instances of `fileName` was found.
* throw an IO exception if multiple instances of `fileName` were found.

#### Benefits


* Save some code to unpack of the set that [Find Resources](..//Library/IO.md#IO-findResources) produces.

#### Pitfalls


* [Get Resource](..//Library/IO.md#IO-getResource) searches for all instances in the entire run-time context of the current 
module. So if the search path (classpath) grows, new similar files may be added that match and this
function will start throwing IO exceptions. If you can influence the `fileName`, then make sure
to pick a name that is always going to be unique for the current project. 

## function appendToFile {#IO-appendToFile}

Append a value to a file.

```rascal
void appendToFile(loc file, value V..., str charset=DEFAULT_CHARSET, bool inferCharset=!(charset?))
throws PathNotFound, IO
```


Append a textual representation of some values to an existing or a newly created file:

*  If a value is a simple string, the quotes are removed and the contents are de-escaped.
*  If a value has a non-terminal type, the parse tree is unparsed to produce a value.
*  All other values are printed as-is.
*  Each value is terminated by a newline character.

The existing file can be stored using any character set possible, if you know the character set, please use [Append To File Enc](..//Library/IO.md#IO-appendToFileEnc).
Else the same method of deciding the character set is used as in [Read File](..//Library/IO.md#IO-readFile).

#### Pitfalls


*  The same encoding pitfalls as the [Read File](..//Library/IO.md#IO-readFile) function.

## function appendToFileEnc {#IO-appendToFileEnc}

Append a value to a file.

```rascal
void appendToFileEnc(loc file, str charset, value V...) throws PathNotFound, IO
```


Append a textual representation of some values to an existing or a newly created file:

*  If a value is a simple string, the quotes are removed and the contents are de-escaped.
*  If a value has a non-terminal type, the parse tree is unparsed to produce a value.
*  All other values are printed as-is.
*  Each value is terminated by a newline character.

Files are encoded using the charset provided.

## function charsets {#IO-charsets}

Returns all available character sets.

```rascal
set[str] charsets()
```

## function canEncode {#IO-canEncode}

Returns whether this charset can be used for encoding (use with ((writeFile)))

```rascal
set[str] canEncode(str charset)
```

## function bprintln {#IO-bprintln}

Print a value and return true.

```rascal
bool bprintln(value arg)
```


Print a value and return `true`. This is useful for debugging complex Boolean expressions or comprehensions.
The only difference between this function and [Println](..//Library/IO.md#IO-println) is that its return type is `bool` rather than `void`.

#### Examples



```rascal-shell 
rascal>import IO;
ok
rascal>bprintln("Hello World");
Hello World
bool: true
```

## function exists {#IO-exists}

Check whether a given location exists.

```rascal
bool exists(loc file)
```


Check whether a certain location exists, i.e., whether an actual file is associated with it.

#### Examples



```rascal-shell 
rascal>import IO;
ok
```

Does the library file `IO.rsc` exist?

```rascal-shell ,continue
rascal>exists(|std:///IO.rsc|);
bool: true
```

## function find {#IO-find}

Find a named file in a list of locations.

```rascal
loc find(str name, list[loc] path) throws PathNotFound
```

#### Examples



```rascal-shell 
rascal>import IO;
ok
```
Find the file `IO.rsc` in the standard library:

```rascal-shell ,continue
rascal>find("IO.rsc", [|std:///|]);
loc: |std:///IO.rsc|
```

## function isDirectory {#IO-isDirectory}

Check whether a given location is a directory.

```rascal
bool isDirectory(loc file)
```


Check whether the location `file` is a directory.

## function iprint {#IO-iprint}

Print an indented representation of a value.

```rascal
void iprint(value arg, int lineLimit = 1000)
```


See [IprintExp](..//Library/IO.md#IO-iprintExp) for a version that returns its argument as result
and [Iprintln](..//Library/IO.md#IO-iprintln) for a version that adds a newline
and [IprintToFile](..//Library/IO.md#IO-iprintToFile) for a version that prints to a file.

#### Examples



```rascal-shell 
rascal>import IO;
ok
rascal>iprint(["fruits", ("spider" : 8, "snake" : 0), [10, 20, 30]]);
[
  "fruits",
  ("snake":0,"spider":8),
  [10,20,30]
]
ok
```

## function iprintToFile {#IO-iprintToFile}

Print an indented representation of a value to the specified location.

```rascal
void iprintToFile(loc file, value arg, str charset=DEFAULT_CHARSET)
```


See [Iprint](..//Library/IO.md#IO-iprint) for a version that displays the result on the console
and [IprintExp](..//Library/IO.md#IO-iprintExp) for a version that returns its argument as result
and [Iprintln](..//Library/IO.md#IO-iprintln) for a version that adds a newline.

#### Examples



```rascal-shell 
rascal>import IO;
ok
rascal>iprintToFile(|file:///tmp/fruits.txt|, ["fruits", ("spider" : 8, "snake" : 0), [10, 20, 30]]);
ok
```

## function iprintToString {#IO-iprintToString}

```rascal
str iprintToString(value arg)
```

## function iprintExp {#IO-iprintExp}

Print an indented representation of a value and returns the value as result.

```rascal
&T iprintExp(&T v)
```


See [IprintlnExp](..//Library/IO.md#IO-iprintlnExp) for a version that adds a newline.

#### Examples



```rascal-shell 
rascal>import IO;
ok
rascal>iprintExp(["fruits", ("spider" : 8, "snake" : 0), [10, 20, 30]]);
[
  "fruits",
  ("snake":0,"spider":8),
  [10,20,30]
]
list[value]: [
  "fruits",
  ("snake":0,"spider":8),
  [10,20,30]
]
```

## function iprintlnExp {#IO-iprintlnExp}

Print an indented representation of a value followed by a newline and returns the value as result.

```rascal
&T iprintlnExp(&T v)
```


See [IprintExp](..//Library/IO.md#IO-iprintExp) for a version that does not add a newline.

#### Examples



```rascal-shell 
rascal>import IO;
ok
rascal>iprintlnExp(["fruits", ("spider" : 8, "snake" : 0), [10, 20, 30]]);
[
  "fruits",
  ("snake":0,"spider":8),
  [10,20,30]
]
list[value]: [
  "fruits",
  ("snake":0,"spider":8),
  [10,20,30]
]
```

## function iprintln {#IO-iprintln}

Print a indented representation of a value and add a newline at the end.

```rascal
void iprintln(value arg, int lineLimit = 1000)
```


See [IprintlnExp](..//Library/IO.md#IO-iprintlnExp) for a version that returns its argument as result
and [Iprint](..//Library/IO.md#IO-iprint) for a version that does not add a newline.

By default we only print the first 1000 lines, if you want to print larger values, either 
use [WriteTextValueFile](..//Library/ValueIO.md#ValueIO-writeTextValueFile) or change the limit with the lineLimit parameter.

#### Examples



```rascal-shell 
rascal>import IO;
ok
rascal>iprintln(["fruits", ("spider" : 8, "snake" : 0), [10, 20, 30]]);
[
  "fruits",
  ("snake":0,"spider":8),
  [10,20,30]
]
ok
rascal>iprintln([ {"hi"} | i <- [0..1000]], lineLimit = 10);
[
  {"hi"},
  {"hi"},
  {"hi"},
  {"hi"},
  {"hi"},
  {"hi"},
  {"hi"},
  {"hi"},
  {"hi"},
...
ok
```

## function isFile {#IO-isFile}

Check whether a given location is actually a file (and not a directory).

```rascal
bool isFile(loc file)
```


Check whether location `file` is actually a file.

## function isReadable {#IO-isReadable}

Check whether a given location is a readable file

```rascal
bool isReadable(loc file) throws PathNotFound
```


Tries to predict whether a `file` is readable. 
If the file does not exist, this will throw an exception

#### Pitfalls


There are (rare) cases where it will return `true`` and [Read File](..//Library/IO.md#IO-readFile) will still throw an exception (and even less likely but possible for the inverse).

## function isWritable {#IO-isWritable}

Check whether a given location is a writable file (and not readonly/non-existing).

```rascal
bool isWritable(loc file) throws PathNotFound
```


Tries to predict whether a file `file` is writable. 
If the file does not exist, this will throw an exception

#### Pitfalls


There are cases where it will return `true`` and [Write File](..//Library/IO.md#IO-writeFile) will still throw an exception (and even less likely but possible for the inverse).

## function lastModified {#IO-lastModified}

Last modification date of a location.

```rascal
datetime lastModified(loc file)
```


Returns last modification time of the file at location `file`.

#### Examples



```rascal-shell 
rascal>import IO;
ok
```
Determine the last modification date of the Rascal standard library:

```rascal-shell ,continue
rascal>lastModified(|std:///IO.rsc|);
datetime: $2025-12-12T12:55:52.612+00:00$
```

## function created {#IO-created}

Creation datetime of a location.

```rascal
datetime created(loc file)
```


Returns the creation time of the file at location `file`.

#### Examples



```rascal-shell 
rascal>import IO;
ok
```
Determine the last modification date of the Rascal standard library:

```rascal-shell ,continue
rascal>created(|std:///IO.rsc|);
datetime: $2025-12-12T12:55:52.612+00:00$
```

## function touch {#IO-touch}

Set the modification date of a file to `now` or create the file if it did not exist yet

```rascal
void touch(loc file)
```

## function setLastModified {#IO-setLastModified}

Set the modification date of a file to the timestamp

```rascal
void setLastModified(loc file, datetime timestamp)
```

## function listEntries {#IO-listEntries}

List the entries in a directory.

```rascal
list[str] listEntries(loc file)
```


List the entries in directory `file`.

#### Examples



```rascal-shell 
rascal>import IO;
ok
```
List all entries in the standard library:

```rascal-shell ,continue
rascal>listEntries(|std:///|);
list[str]: ["Grammar.rsc","Boolean.rsc","util","Prelude.class","Prelude$ByteBufferBackedInputStream.class","ParseTree.rsc","IO.rsc","Prelude$Less.class","Set.rsc","Prelude$2.class","index.md","Prelude$Sorting.class","Prelude$4.class","String.rsc","Node.rsc","Relation.rsc","ListRelation.rsc","Prelude$1.class","Type.rsc","Prelude$ReleasableCallback.class","analysis","Messages.class","Content.rsc","Prelude$Distance.class","Message.rsc","resource","Prelude.rsc","DateTime.rsc","List.rsc","Location.rsc","ValueIO.rsc","Map.rsc","vis","Prelude$NodeComparator.class","Prelude$3.class","Type.class","Prelude$Backtrack.class","lang","Exception.rsc"]
```

## function mkDirectory {#IO-mkDirectory}

Create a new directory.

```rascal
void mkDirectory(loc file)
throws PathNotFound, IO
```


Create a directory at location `file`.

## function print {#IO-print}

Print a value without subsequent newline.

```rascal
void print(value arg)
```


Print a value on the output stream.
See [Println](..//Library/IO.md#IO-println) for a version that adds a newline
and [PrintExp](..//Library/IO.md#IO-printExp) for a version that returns its argument as value.

#### Examples


Note that the only difference with [Println](..//Library/IO.md#IO-println) is that no newline is added after the value is printed

```rascal-shell 
rascal>import IO;
ok
rascal>print("Hello World");
Hello World
ok
```

NOTE: Since `print` does not add a newline, the prompt `ok` appears at a weird place, i.e., 
glued to the output of `print`.

## function printExp {#IO-printExp}

Print a value and return it as result.

```rascal
&T printExp(&T v)

&T printExp(str msg, &T v)
```

#### Examples



```rascal-shell 
rascal>import IO;
ok
rascal>printExp(3.14);
3.14
real: 3.14
rascal>printExp("The value of PI is approximately ", 3.14);
The value of PI is approximately 3.14
real: 3.14
```

## function println {#IO-println}

Print a value to the output stream and add a newline.

```rascal
void println(value arg)

void println()
```


Print a value on the output stream followed by a newline.
See [Print](..//Library/IO.md#IO-print) for a version that does not add a newline
and [PrintlnExp](..//Library/IO.md#IO-printlnExp) for a version that returns its argument as value.

#### Examples



```rascal-shell 
rascal>import IO;
ok
rascal>println("Hello World");
Hello World
ok
```
Introduce variable S and print it:

```rascal-shell ,continue
rascal>S = "Hello World";
str: "Hello World"
───
Hello World
───
rascal>println(S);
Hello World
ok
```
Introduce variable L and print it:

```rascal-shell ,continue
rascal>L = ["a", "b", "c"];
list[str]: ["a","b","c"]
rascal>println(L);
["a","b","c"]
ok
```
Use a string template to print several values:

```rascal-shell ,continue
rascal>println("<S>: <L>");
Hello World: ["a","b","c"]
ok
```
Just print a newline

```rascal-shell ,continue
rascal>println();
ok
```

## function printlnExp {#IO-printlnExp}

Print a value followed by a newline and return it as result.

```rascal
&T printlnExp(&T v)

&T printlnExp(str msg, &T v)
```

#### Examples



```rascal-shell 
rascal>import IO;
ok
rascal>printlnExp(3.14);
3.14
real: 3.14
rascal>printlnExp("The value of PI is approximately ", 3.14);
The value of PI is approximately 3.14
real: 3.14
```
NOTE: Since `printExp` does no produce a newline after its output, the result prompt `real: 3.14` is glued to the
output of `printExp`.

## function rprint {#IO-rprint}

Raw print of a value.

```rascal
void rprint(value arg)
```


#### Pitfalls


This function is only available for internal use in the Rascal development team.

## function rprintln {#IO-rprintln}

Raw print of a value followed by newline.

```rascal
void rprintln(value arg)
```


#### Pitfalls


This function is only available for internal use in the Rascal development team.

## function readFile {#IO-readFile}

Read the contents of a location and return it as string value.

```rascal
str readFile(loc file, str charset=DEFAULT_CHARSET, bool inferCharset=!(charset?))
throws PathNotFound, IO
```


Return the contents of a file location as a single string.
Also see [Read File Lines](..//Library/IO.md#IO-readFileLines).

#### Pitfalls


*  In case encoding is not known, we try to estimate as best as we can.
*  We default to UTF-8, if the file was not encoded in UTF-8 but the first characters were valid UTF-8, 
  you might get an decoding error or just strange looking characters.

## function readFileEnc {#IO-readFileEnc}

Read the contents of a location and return it as string value.

```rascal
str readFileEnc(loc file, str charset) throws PathNotFound, IO
```


Return the contents (decoded using the Character set supplied) of a file location as a single string.
Also see [Read File Lines Enc](..//Library/IO.md#IO-readFileLinesEnc).

## function readBase64 {#IO-readBase64}

Read the content of a file and return it as a base-64 encoded string.

```rascal
str readBase64(loc file, bool includePadding=true)
throws PathNotFound, IO
```


## function uuencode {#IO-uuencode}

```rascal
str uuencode(loc file)
```

## function writeBase64 {#IO-writeBase64}

Decode a base-64 encoded string and write the resulting bytes to a file.

```rascal
void writeBase64(loc file, str content)
throws PathNotFound, IO
```


## function uudecode {#IO-uudecode}

```rascal
void uudecode(loc file, str content)
```

## function readBase32 {#IO-readBase32}

Read the content of a file and return it as a base-32 encoded string.

```rascal
str readBase32(loc file, bool includePadding=true)
throws PathNotFound, IO
```


## function writeBase32 {#IO-writeBase32}

Decode a base-32 encoded string and write the resulting bytes to a file.

```rascal
void writeBase32(loc file, str content)
throws PathNotFound, IO
```


## function readFileBytes {#IO-readFileBytes}

Read the contents of a file and return it as a list of bytes.

```rascal
list[int] readFileBytes(loc file)
throws PathNotFound, IO
```

## function readFileLines {#IO-readFileLines}

Read the contents of a file location and return it as a list of strings.

```rascal
list[str] readFileLines(loc file, str charset=DEFAULT_CHARSET)
throws PathNotFound, IO
```


Return the contents of a file location as a list of lines.
Also see [Read File](..//Library/IO.md#IO-readFile).

#### Pitfalls


*  In case encoding is not known, we try to estimate as best as we can (see [readFile]).
*  We default to UTF-8, if the file was not encoded in UTF-8 but the first characters were valid UTF-8, 
  you might get an decoding error or just strange looking characters (see [Read File](..//Library/IO.md#IO-readFile)).

## function writeFileLines {#IO-writeFileLines}

Writes a list of strings to a file, where each separate string is ended with a newline

```rascal
void writeFileLines(loc file, list[str] lines, str charset=DEFAULT_CHARSET)
```

#### Benefits


* mirrors [Read File Lines](..//Library/IO.md#IO-readFileLines) in its functionality

#### Pitfalls


* if the individual elements of the list also contain newlines, the output may have more lines than list elements

## function readFileLinesEnc {#IO-readFileLinesEnc}

Read the contents of a file location and return it as a list of strings.

```rascal
list[str] readFileLinesEnc(loc file, str charset)
throws PathNotFound, IO
```


Return the contents (decoded using the Character set supplied) of a file location as a list of lines.
Also see [Read File Lines](..//Library/IO.md#IO-readFileLines).

## function remove {#IO-remove}

Remove files or directories

```rascal
void remove(loc file, bool recursive=true) throws IO
```

## function rename {#IO-rename}

Rename files or directories

```rascal
void rename(loc from, loc to, bool overwrite=false) throws IO
```

#### Benefits


* will rename between schemes and within schemes, seemlessly.
* within schemes renaming is implemented as close to the operating system rename functionality as possible. This can be very fast.

#### Pitfalls


* Between-scheme renaming can cause large recursive copy operations:
   - that can take a lot more time
   - if interrupted on the OS level will leave semi-copied target files and not-yet-removed origins

## function writeFile {#IO-writeFile}

Write values to a file.

```rascal
void writeFile(loc file, value V..., str charset=DEFAULT_CHARSET)
throws PathNotFound, IO
```


Write a textual representation of some values to a file:

*  If a value is a simple string, the quotes are removed and the contents are de-escaped.
*  If a value has a non-terminal type, the parse tree is unparsed to produce a value.
*  All other values are printed as-is.
*  Each value is terminated by a newline character.

Files are encoded in UTF-8, in case this is not desired, use [Write File Enc](..//Library/IO.md#IO-writeFileEnc).

## function writeFileBytes {#IO-writeFileBytes}

Write a list of bytes to a file.

```rascal
void writeFileBytes(loc file, list[int] bytes)
throws PathNotFound, IO
```

## function writeFileEnc {#IO-writeFileEnc}

Write values to a file.

```rascal
void writeFileEnc(loc file, str charset, value V...) throws PathNotFound, IO
```


Write a textual representation of some values to a file:

*  If a value is a simple string, the quotes are removed and the contents are de-escaped.
*  If a value has a non-terminal type, the parse tree is unparsed to produce a value.
*  All other values are printed as-is.
*  Each value is terminated by a newline character.

Files are encoded using the charset provided.

## function md5HashFile {#IO-md5HashFile}

Read the contents of a location and return its MD5 hash.

```rascal
str md5HashFile(loc file)
throws PathNotFound, IO
```


MD5 hash the contents of a file location.

## function md5Hash {#IO-md5Hash}

```rascal
str md5Hash(value v)
```

## function createLink {#IO-createLink}

```rascal
str createLink(str title, str target)
```

## function toBase64 {#IO-toBase64}

```rascal
str toBase64(loc file, bool includePadding=true)
throws PathNotFound, IO
```

## function copy {#IO-copy}

```rascal
void copy(loc source, loc target, bool recursive=false, bool overwrite=true) throws IO
```

## function copyFile {#IO-copyFile}

```rascal
void copyFile(loc source, loc target)
```

## function copyDirectory {#IO-copyDirectory}

```rascal
void copyDirectory(loc source, loc target)
```

## function move {#IO-move}

Is an alias for ((rename))

```rascal
void move(loc source, loc target, bool overwrite=true) throws IO
```

## function arbLoc {#IO-arbLoc}

```rascal
loc arbLoc()
```

## function watch {#IO-watch}

```rascal
void watch(loc src, bool recursive, void (FileSystemChange event) watcher)
```

## function unwatch {#IO-unwatch}

```rascal
void unwatch(loc src, bool recursive, void (FileSystemChange event) watcher)
```

## data IOCapability {#IO-IOCapability}
Categories of IO capabilities for loc schemes

```rascal
data IOCapability  
     = reading()
     | writing()
     | classloading()
     | resolving()
     | watching()
     ;
```


* [Reading](..//Library/IO.md#IO-reading) includes at least these functions:
   * [Read File](..//Library/IO.md#IO-readFile)
   * [Last Modified](..//Library/IO.md#IO-lastModified) and [Created](..//Library/IO.md#IO-created)
   * [Exists](..//Library/IO.md#IO-exists)
   * [Is File](..//Library/IO.md#IO-isFile) and [Is Directory](..//Library/IO.md#IO-isDirectory)
   * `loc.ls`, and [List Entries](..//Library/IO.md#IO-listEntries) 
* [Writing](..//Library/IO.md#IO-writing) includes at least these functions:
   * [Write File](..//Library/IO.md#IO-writeFile)
   * [Mk Directory](..//Library/IO.md#IO-mkDirectory)
   * [Remove](..//Library/IO.md#IO-remove)
   * [Rename](..//Library/IO.md#IO-rename) and [Move](..//Library/IO.md#IO-move)
* [Classloading](..//Library/IO.md#IO-classloading) means that for this scheme a specialized/optimized ClassLoader can be produced at run-time. By
default an abstract (slower) ClassLoader can be built using `read` capabilities on `.class` files.
* [Watching](..//Library/IO.md#IO-watching) means that for this scheme a file watcher can be instantiated. By default 
a slower more generic file watcher is created based on capturing `write` actions.
   * [Resolving](..//Library/IO.md#IO-resolving) schemes provide a transparent facade for more abstract URIs. The abstract URI is 
rewritten to a more concrete URI on-demand. Logical URIs can be nested arbitrarily.

These capabilities extend naturally to other IO functions that use the internal versions of the above functionality, 
such as used in [Value IO](..//Library/ValueIO.md) and [IO](..//Library/lang/json/IO.md), etc.

#### Pitfalls


* IO capabilities are not to be confused with file _permissions_. It is still possible that a file
has write capabilities, but writing is denied by the OS due to a lack of permissions.

## function capabilities {#IO-capabilities}

List the IO capabilities of a loc (URI) scheme

```rascal
set[IOCapability] capabilities(loc location)
```

