---
title: "module util::Sampling"
id: Sampling
slug: /Library/util/Sampling
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

Utilities to randomly select smaller datasets from larger datasets
#### Usage

```rascal
import util::Sampling;
```

#### Dependencies
```rascal
import util::Math;
import Map;
import List;
import Set;
```

#### Description


Sampling is important when the analysis algorithms do not scale to the size of 
the original corpus, or when you need to train an analysis on a representative
set without overfitting on the entire corpus. These sampling functions all
assume that a uniformly random selection is required.


## function sample {#util-Sampling-sample}

Reduce the arity of a set by selecting a uniformly distributed sample.

```rascal
set[&T] sample(set[&T] corpus, int target)
```


A uniform subset is computed by iterating over the set and skipping every element
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new set of
expected `target` size, but most probably a little smaller or larger.

#### Examples



```rascal-shell 
rascal>import util::Sampling;
ok
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"b","c","f","g","h","j","k"}
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"a","e","f","g","i","j","k"}
rascal>sample({"a","b","c","e","f","g","h","i","j","k"}, 4)
set[str]: {"b","h"}
```

## function sample {#util-Sampling-sample}

Reduce the length of a list by selecting a uniformly distributed sample.

```rascal
list[&T] sample(list[&T] corpus, int target)
```


The random selection of elements does not change their initial order in the list.
A uniform sublist is computed by iterating over the list and skipping every element
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new list of
expected `target` size, but most probably a little smaller or larger.

#### Examples



```rascal-shell 
rascal>import util::Sampling;
ok
rascal>sample([1..1000], 30)
list[int]: [39,71,74,173,238,239,253,257,282,377,400,493,561,625,641,680,686,694,742,765,782,815,838,847,914,929,980]
rascal>sample([1..1000], 30)
list[int]: [2,39,122,129,189,199,223,224,267,311,335,411,417,501,502,593,627,644,676,691,736,766,832,841,862,877,886,892,898,915,929,934,947,983,987]
rascal>sample([1..1000], 30)
list[int]: [105,123,156,217,272,274,298,303,338,367,439,461,482,485,508,526,565,568,610,626,736,748,750,779,835,926,929,945]
```

## function sample {#util-Sampling-sample}

Reduce the size of a map by selecting a uniformly distributed sample.

```rascal
map[&T,&U] sample(map[&T,&U] corpus, int target)
```


A uniform submap is computed by iterating over the map's keys and skipping every key
with a probability of `1/(size(corpus) / target)`. This rapidly generates a new map of
expected `target` size, but most probably a little smaller or larger.

