---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://3b71cb97-a88a-4824-823b-8ed0fc2d37db|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <5,25,|uuid://fd2e7428-466f-40d7-9b8f-183ef087d1de|>,
  <8,64,|uuid://839f0dff-bdc3-4d87-9986-e052a9a7cfd1|>,
  <10,100,|uuid://83d55b70-0d49-484c-b64a-2bdc632ad094|>,
  <1,1,|uuid://432c2b4c-1917-48cc-a982-492f82c241c0|>,
  <2,4,|uuid://80e61a93-9b91-42d8-9ff0-f6552a66b629|>,
  <7,49,|uuid://9608ff87-18ad-4eb7-b673-366d2a326359|>,
  <4,16,|uuid://9cda6a62-9e8e-4943-b691-f83e0775dfa5|>,
  <3,9,|uuid://877d68a2-3665-40a9-994c-ab3ea4f638d0|>,
  <6,36,|uuid://b514390e-726b-44a6-ae5f-90a4b08f4b9b|>,
  <9,81,|uuid://efb3672e-6077-4fcb-b8eb-a7c7db0532ec|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://2273d205-85d2-4af2-917f-c31a7006a7cc|,
  <2,4>:|uuid://1349a7e5-eb2f-40c9-9ab1-f22b3ea5a612|,
  <7,49>:|uuid://50ccf382-ea9a-40cb-8072-0361aa770eb3|,
  <9,81>:|uuid://6b14baf0-15a8-4ff4-bd69-39c0bdfd9f70|,
  <8,64>:|uuid://7b898a66-8e4b-4e53-ac0a-9979b6d496b5|,
  <5,25>:|uuid://d573fc8d-6960-4b60-ad6d-4e1b5a3f0242|,
  <4,16>:|uuid://44c2f1ab-1cc3-4b3c-b323-4ae691032928|,
  <1,1>:|uuid://4597ad6d-b727-4764-9073-88fa35fd7932|,
  <10,100>:|uuid://a7513211-893f-4813-a87f-60d593cf9933|,
  <3,9>:|uuid://42e2024d-4f87-4935-b3a2-04584e6953d4|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

