---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://92b41dcf-9793-461b-866f-b067574a37c0|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <9,81,|uuid://b4eaed61-7854-49ec-8f99-67bc34049b7e|>,
  <8,64,|uuid://7bd03c22-6e0f-46f0-bcfa-181eb5fe79f9|>,
  <7,49,|uuid://08a962dd-1d5b-4ecb-8596-5a588870a3aa|>,
  <6,36,|uuid://ef2ae900-eb8c-4cde-93e8-083184097a76|>,
  <5,25,|uuid://355f8d50-cc1c-4520-b0d7-b003458b29fa|>,
  <1,1,|uuid://e58789d5-ce55-4185-98a5-68bbb9a48d2d|>,
  <10,100,|uuid://c6112949-fdbf-429b-8630-278809588f5d|>,
  <4,16,|uuid://236821af-d7d4-4976-a9a2-87bbb955c583|>,
  <3,9,|uuid://35f36fa8-ce6e-4e4b-b1c2-13c18db27716|>,
  <2,4,|uuid://45436020-127e-4616-a1ce-0f3dca9e873e|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://25f37760-30f9-4000-aa3d-422d63192dbb|,
  <2,4>:|uuid://dc16acf1-912d-4b61-ac02-8b42101af939|,
  <7,49>:|uuid://361c1b9e-9d4d-4842-a628-4ad8e19aec24|,
  <9,81>:|uuid://52f87c5d-7bce-4fc0-afc4-7f22a8dfa1c9|,
  <8,64>:|uuid://6b9f1a90-f861-4ac2-bffd-2a3a7b1f6eb7|,
  <5,25>:|uuid://c59665f3-7296-4f7a-b9de-0144432f9f87|,
  <4,16>:|uuid://21d9a8cf-c7d3-477c-b365-2180044cd61b|,
  <1,1>:|uuid://54ab2b88-b6e8-4785-84f7-e3c5f34ff447|,
  <10,100>:|uuid://9e2a2864-e766-4d5a-8bb3-903453e6b8b5|,
  <3,9>:|uuid://2fde72e1-288b-458d-af25-097c2fe92d45|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

