---
title: "module util::UUID"
id: UUID
slug: /Library/util/UUID
---

<div class="theme-doc-version-badge badge badge--secondary">rascal-Not specified</div>

#### Usage

```rascal
import util::UUID;
```



## function uuid {#util-UUID-uuid}

generates a unique identifier shaped as a `loc`

```rascal
loc uuid()
```


This function generates a UUID, see <http://en.wikipedia.org/wiki/Universally_unique_identifier>.
Since UUIDs are useful to assign an opaque and unique identity to data, the function returns
a location (which is the preferred representation for encoding **identities** in Rascal)

#### Examples



```rascal-shell 
rascal>import util::UUID;
ok
```

The uuid() function generates a location with the authority showing the literal canonical UUID string

```rascal-shell ,continue
rascal>uuid()
loc: |uuid://4600b1ef-46fe-47c7-9658-7bfd0e2eedde|
```

Use it to relate identies to data objects, as in this example which adds a field to a relation:


```rascal-shell ,continue
rascal>myData = { <i,i*i> | i <- [1..11] }; 
rel[int,int]: {
  <10,100>,
  <7,49>,
  <1,1>,
  <3,9>,
  <9,81>,
  <2,4>,
  <4,16>,
  <6,36>,
  <5,25>,
  <8,64>
}
rascal>rel[int n, int square, loc id] myUniqueData = { <i,j,uuid()> | <i,j> <- myData };
rel[int n,int square,loc id]: {
  <1,1,|uuid://e71e011d-68c5-4728-a639-b5626a871ae8|>,
  <5,25,|uuid://64f5a140-305a-4420-8769-9b03be945427|>,
  <4,16,|uuid://c862b4b1-e21f-4ad2-b2ff-865f5dce14d4|>,
  <8,64,|uuid://9af2d8b6-2aa0-4cee-9da9-9370fd7a0b78|>,
  <7,49,|uuid://2f49c280-1a98-43cc-a984-b7561bd6aad3|>,
  <9,81,|uuid://4e761a00-2656-4e64-8490-e69b9ccb6656|>,
  <3,9,|uuid://9d59ab27-1b6a-4fab-bd67-eeade123c0aa|>,
  <10,100,|uuid://ec37c27c-8c3b-459e-a900-05339ac44958|>,
  <2,4,|uuid://5dae2d2d-9fc4-4326-b0b7-8215877ecfd0|>,
  <6,36,|uuid://9b8b9f86-7e04-4b3a-b7a6-908c97de0812|>
}
rascal>map[tuple[int i, int j] t, loc id] myUniqueMap = (<i,j>:uuid() | <i,j> <- myData );
map[tuple[int i,int j] t, loc id]: (
  <6,36>:|uuid://1cec80ef-6072-4a50-a3ac-bebe756fae81|,
  <2,4>:|uuid://8e1e0c86-4b7e-41bb-8928-a8b01bcf93b7|,
  <7,49>:|uuid://8c85e3f0-071b-4791-bb06-dcd3493ab2fe|,
  <9,81>:|uuid://c540da60-e95e-45bd-9f1f-6e85282ca9ec|,
  <8,64>:|uuid://64099a99-9ce3-4e92-a09a-b2abaf410704|,
  <5,25>:|uuid://a7fa26de-08ee-4f29-87d7-c965a911b6c0|,
  <4,16>:|uuid://8308b5cf-afa2-4184-9d9e-7fca7970dec9|,
  <1,1>:|uuid://2f1d0cec-4702-4654-9dc9-98012a5a9dfe|,
  <10,100>:|uuid://bef73619-2e06-49dd-82e0-bb05a530641a|,
  <3,9>:|uuid://c4685115-5758-4385-b720-44978ada94c6|
)
```
Note how uuid() should always generate a fresh value:

```rascal-shell ,continue
rascal>assert uuid() != uuid(); 
bool: true
```

#### Benefits


* Locations are used for identifying program elements or model elements in Rascal. The uuid() function provides
an quick-and-easy way of acquiring such an identity without having to design a naming scheme.

#### Pitfalls


*  UUIDs are a quick and dirty way of identifying data which may lead to hard to debug code. A naming scheme for locations is better because it generates human readable
locations which carry meaning. For example consider the difference in readability between these two values:
`|uuid://47fdcd64-4fd0-41a1-8aa3-61c5b272c3fc|` and `|java+class:///java/lang/Object|`. Both may lead to the same 
results in your computation, but if we print either of them out, one of them is opaque and the other is transparent. A transparent naming scheme is preferable for
debugging purposes.

## function uuidi {#util-UUID-uuidi}

see [uuid], this function does the same except return the UUID as an int.

```rascal
int uuidi()
```

#### Pitfalls


*  beware that this integer is almost guaranteed to use 128 bits, so communicating it outside of
Rascal should not be done via a Java 32-bit integer.

